#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/05/31
@file: check_disk_module_compatibility.py
@function: 更换后检查硬盘兼容性
@modify:
"""
from py.common.entity.exception import BusinessException
from py.common.entity.item_status import ItemStatus
from py.common.entity.user_select import has_return_result, \
    UserConfirmWithGuide, set_json_result, get_user_select_value
from py.common.service import resource_service, logger_factory
from py.common.service.connection.redfish_connection_service import \
    RedfishService
from py.common.service.connection.ssh_connection_service import SshService
from py.fusion_cube.common.context import disk_context_util, \
    common_context_util
from py.fusion_cube.common.record import record_sg_device_info
from py.fusion_cube.common.service.bmc import chassis_resource_util
from py.fusion_cube.common.service.os_util import hardware_info_util


def execute(context):
    # 不涉及场景
    if _is_not_involved(context):
        return ItemStatus.NOT_INVOLVED, ""
    # 已鉴权场景
    if context.getBmcNode() is not None:
        return DiskModuleCompatibilityCheckItem(context).check()
    # 未鉴权场景，需要用户手动确认
    if not has_return_result(context):
        save_confirm_result(context)
        return ItemStatus.WAITING_CONFIRM, ""
    return judge_user_confirm(context)


def _is_not_involved(context):
    return disk_context_util.is_nvme_disk(context) or \
           common_context_util.is_proprietary_hardware(context) or \
           common_context_util.is_hg_server(context)


def save_confirm_result(context):
    result = UserConfirmWithGuide(
        resource_service.get_msg('check_compatibility_user_confirm_message'),
        context.getItemDocLink('manual_confirm_guide'),
        resource_service.get_msg('check_compatibility_user_confirm_title'))
    set_json_result(context, result)


def judge_user_confirm(context):
    is_user_confirm = bool(get_user_select_value(context))
    logger_factory.create_logger(__file__).info(
        "user confirm check compatibility result is {}".format(is_user_confirm))
    return (ItemStatus.ALREADY_CONFIRM, "") if is_user_confirm else (
        ItemStatus.FAILED,
        resource_service.get_msg('check_compatibility_user_confirm_failed'))


class DiskModuleCompatibilityCheckItem(object):
    def __init__(self, context):
        self.context = context

    def check(self):
        back_plane_ids = self._get_back_plane_ids()
        if not self._contains_risk_back_plane_id(back_plane_ids):
            return ItemStatus.PASS, ""
        service = SshService(self.context.getNode())
        sg_device_infos = hardware_info_util.get_sg_device_infos(service)
        # 不包含0x0083时，说明全是0x0073才做背板固件版本的检查
        if "0x0083" not in back_plane_ids and not self._contains_risk_back_plane_fw_version(
                sg_device_infos):
            return ItemStatus.PASS, ""
        try:
            if not self._is_risk_replace_disk_mode(sg_device_infos):
                return ItemStatus.PASS, ""
            return ItemStatus.NOT_PASS, resource_service.get_msg(
                "disk.not.compatibility")
        except BusinessException as ex:
            return ItemStatus.NOT_PASS, ex.err_msg

    def _contains_risk_back_plane_id(self, back_plane_ids):
        for back_plane_id in back_plane_ids:
            if self._is_risk_back_plane_id(back_plane_id):
                return True
        return False

    def _is_risk_back_plane_id(self, board_id):
        return board_id in ["0x0083", "0x0073"]

    def _get_back_plane_ids(self):
        back_plane_ids = list()
        service = RedfishService(self.context.getBmcNode())
        back_plane_infos = chassis_resource_util.get_assign_board_infos(
            service, "DiskBackplane")
        for link in back_plane_infos.get("Links", {}):
            back_plane_info_uri = link.get("@odata.id")
            if back_plane_info_uri:
                res_dict = service.exec_get(back_plane_info_uri)
                back_plane_ids.append(res_dict.get("BoardId"))
        return back_plane_ids

    def _contains_risk_back_plane_fw_version(self, sg_device_infos):
        for fields in sg_device_infos:
            if len(fields) >= 5 and fields[3] == "Expander":
                device_type = fields[1]
                device_version = fields[5]
                if device_type == "enclosu" and device_version != "331":
                    return True
        return False

    def _is_risk_replace_disk_mode(self, sg_device_infos):
        before_replace_sg_device_infos = record_sg_device_info.get_recorded_sg_device_infos(
            self.context)
        before_replace_disk_drive_letter_2_mode = self._get_disk_drive_letter_2_mode(
            before_replace_sg_device_infos)
        after_replace_disk_drive_letter_2_mode = self._get_disk_drive_letter_2_mode(
            sg_device_infos)
        replace_disk_drive_letters = self._get_replace_disk_drive_letters(
            after_replace_disk_drive_letter_2_mode,
            before_replace_disk_drive_letter_2_mode)
        # 没有新增盘符。盘符为更换前盘符的即为更换盘。
        if not replace_disk_drive_letters:
            old_disk_drive_letter = record_sg_device_info.get_recorded_disk_drive_letter(
                self.context)
            if old_disk_drive_letter in after_replace_disk_drive_letter_2_mode:
                return self._contains_risk_disk_mode(
                    [after_replace_disk_drive_letter_2_mode.get(
                        old_disk_drive_letter)])
            raise BusinessException(err_msg=resource_service.get_msg(
                "not.found.replace.disk.mode"))
        # 有差异，按理说只能1块差异，多块时先也都判断Mode是否有风险
        return self._contains_risk_disk_mode(
            [after_replace_disk_drive_letter_2_mode.get(disk_drive_letter)
             for disk_drive_letter in replace_disk_drive_letters])

    def _get_replace_disk_drive_letters(
            self,
            after_replace_disk_drive_letter_2_mode,
            before_replace_disk_drive_letter_2_mode):
        """
        找出更换后硬盘信息中盘符不在更换前硬盘信息中的所有盘符
        :param after_replace_disk_drive_letter_2_mode: 更换后的
        :param before_replace_disk_drive_letter_2_mode: 更换前的
        :return:
        """
        replace_disk_drive_letters = list()
        for disk_drive_letter in after_replace_disk_drive_letter_2_mode.keys():
            if disk_drive_letter not in before_replace_disk_drive_letter_2_mode:
                replace_disk_drive_letters.append(disk_drive_letter)
        return replace_disk_drive_letters

    def _get_disk_drive_letter_2_mode(self, sg_device_infos):
        disk_drive_letter_2_mode = dict()
        for fields in sg_device_infos:
            if len(fields) >= 6:
                device_type = fields[1]
                drive_letter = fields[5]
                device_mode = fields[3]
                if device_type == "disk":
                    disk_drive_letter_2_mode[drive_letter] = device_mode
        return disk_drive_letter_2_mode

    def _contains_risk_disk_mode(self, disk_modes):
        for disk_mode in disk_modes:
            if disk_mode in ["HUS726T6TALE600", "WUS721010ALE6L4"]:
                return True
        return False
