#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/09/06
@file: clear_old_disk_alarm.py
@function: 清除旧盘告警
@modify:
"""
import time

from py.common.entity.exception import ConnectionException
from py.common.entity.item_status import ItemStatus
from py.common.entity.user_select import AbstractUserSelectTableQueryData, \
    UserSelectTableQueryResult
from py.common.service import resource_service
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.fusion_cube.common import constant
from py.fusion_cube.common.context import disk_context_util

MATCH_ALARM_LIST_KEY = "match_alarm_list_key"
ALARM_STATUS_UN_RECOVER = 1
ALARM_TYPE_FAULT = 4
INVOLVE_ALARM_ID_LIST = \
    ['0xF000A0132', '0xF0017001D', '0xF000A0138', '0xF000A0136',
     '0xF000A012D', '0xF000A0134', '0xF000A0135', '0xF00D80152']


def execute(context):
    """
    执行入口，内部存在两个分支：查询与清除
    已有查询结果会走清除逻辑，无查询结果走查询逻辑；执行失败框架会清除查询结果
    :param context:
    :return:
    """
    if not _has_query_result(context):
        query_and_save_match_alarms(context)
        if _has_query_result(context):
            return ItemStatus.WAITING_CONFIRM, ""
        return ItemStatus.NOT_INVOLVED, ""
    return clear_chose_alarms(context)


def query_and_save_match_alarms(context):
    """
    查询符合条件的告警
    1. 查询类型为故障且状态为未恢复的告警
    2. 筛选更换前盘相关的告警
    3. 存入上下文
    :param context:
    :return:
    """
    rest_service = CubeRestService(context.getCluster())
    res_dict = rest_service.exec_get(
        constant.ClusterRestUri.BATCH_QUERY_ALARMS,
        params={
            "filter": "alarmStatus::" + str(ALARM_STATUS_UN_RECOVER)
        }
    )
    match_alarm_list = _filter_involve_alarm_list(
        res_dict.get("data", []),
        INVOLVE_ALARM_ID_LIST,
        disk_context_util.get_disk_sn(context))
    _save_alarm_list(context, match_alarm_list)


def clear_chose_alarms(context):
    select_sequence_list = context.getEnv().get(
        disk_context_util.get_select_key(context))
    if not select_sequence_list:
        return ItemStatus.NOT_INVOLVED, ""
    select_sequence_list = [str(sequence) for sequence in select_sequence_list]
    rest_service = CubeRestService(context.getCluster())
    try:
        rest_service.exec_post(constant.ClusterRestUri.BATCH_CLEAN_ALARMS,
                               params={"sn": select_sequence_list})
    except ConnectionException as ex:
        alarm_id_list = _get_alarm_ids_by_sequences(
            context, ex.response.getContent().get('fail', []))
        return ItemStatus.FAILED, resource_service\
            .get_msg('clear.old.disk.alarm.failed').format(alarm_id_list)
    return ItemStatus.SUCCESS, ""


def _get_alarm_ids_by_sequences(context, sequence_list):
    match_alarm_list = context.getEnv().get(MATCH_ALARM_LIST_KEY)
    alarm_ids = []
    for alarm in match_alarm_list:
        if alarm.get('sequence') in sequence_list:
            alarm_ids.append(alarm.get('id'))
    return alarm_ids


def _filter_involve_alarm_list(res_alarm_list, involve_alarm_ids, esn):
    match_alarm_list = list()
    for alarm in res_alarm_list:
        if alarm.get('alarm_id') in involve_alarm_ids and \
                (not esn or esn in alarm.get('alarm_params')):
            match_alarm_list.append(alarm)
    return match_alarm_list


def _has_query_result(context):
    result_key = disk_context_util.get_result_key(context)
    return context.getEnv().get(result_key)


def _save_alarm_list(context, res_alarm_list):
    if not res_alarm_list:
        return
    user_select_table_result = UserSelectTableQueryResult(
        ShowAlarm.get_select_key(),
        resource_service.get_msg("clear.old.disk.alarm.user.select.title"))
    user_select_table_result.set_col_list(ShowAlarm.get_show_col_list())
    for alarm in res_alarm_list:
        user_select_table_result.add_data_2_list(
            ShowAlarm(alarm, _is_alarm_selected(context)).get_show_data_dict())
    context.getEnv()[disk_context_util.get_result_key(context)] = \
        user_select_table_result.get_json_result()
    context.getEnv()[MATCH_ALARM_LIST_KEY] = res_alarm_list


def _is_alarm_selected(context):
    # 旧盘SN不为空时，告警默认勾选
    return bool(disk_context_util.get_disk_sn(context))


class ShowAlarm(AbstractUserSelectTableQueryData):
    @staticmethod
    def get_select_key():
        return 'sequence'

    @staticmethod
    def get_severity_dict():
        return {
            3: resource_service.get_msg('alarm.severity.warning'),
            5: resource_service.get_msg('alarm.severity.major'),
            6: resource_service.get_msg('alarm.severity.critical')
        }

    @staticmethod
    def get_alarm_status_dict():
        return {
            1: resource_service.get_msg('alarm.status.unrecovered'),
            2: resource_service.get_msg('alarm.status.cleared'),
            4: resource_service.get_msg('alarm.status.recovered')
        }

    def __init__(self, res_alarm, is_selected):
        self.alarm_obj_type = res_alarm.get('alarmObjType')
        self.alarm_status = res_alarm.get('alarmStatus')
        self.alarm_id = res_alarm.get('alarm_id')
        self.name = res_alarm.get('alarm_name')
        self.params = res_alarm.get('alarm_params')
        self.type = res_alarm.get('alarm_type')
        self.clear_time = res_alarm.get('clear_time')
        self.clear_type = res_alarm.get('clear_type')
        self.clear_user = res_alarm.get('clear_user')
        self.create_time = res_alarm.get('create_time')
        self.description = res_alarm.get('description')
        self.detail = res_alarm.get('detail')
        self.position = res_alarm.get('position')
        self.sequence = res_alarm.get('sequence')
        self.severity = res_alarm.get('severity')
        self.suggestion = res_alarm.get('suggestion')
        self.is_selected = is_selected

    @staticmethod
    def get_show_col_list():
        return [
            {"key": "alarm_status",
             "name": resource_service.get_msg('alarm.property.status'),
             "show": False},
            {"key": "alarm_id",
             "name": resource_service.get_msg('alarm.property.id'),
             "show": True},
            {"key": "alarm_name",
             "name": resource_service.get_msg('alarm.property.name'),
             "show": True},
            {"key": "severity",
             "name": resource_service.get_msg('alarm.property.severity'),
             "show": True},
            {"key": "sequence",
             "name": resource_service.get_msg('alarm.property.sequence'),
             "show": False},
            {"key": "description",
             "name": resource_service.get_msg('alarm.property.description'),
             "show": False},
            {"key": "suggestion",
             "name": resource_service.get_msg('alarm.property.suggestion'),
             "show": False},
            {"key": "create_time",
             "name": resource_service.get_msg('alarm.property.create.time'),
             "show": True},
        ]

    def get_show_data_dict(self):
        return {
            "alarm_id": self.alarm_id,
            "severity": self._trans_alarm_severity(self.severity),
            "create_time": self._trans_alarm_time(self.create_time),
            "sequence": self.sequence,
            "alarm_name": self.name,
            "description": self.description,
            "suggestion": self.suggestion,
            "alarm_status": self._trans_alarm_status(self.alarm_status),
            "is_selected": self.is_selected
        }

    def _trans_alarm_severity(self, ori_severity):
        return self.get_severity_dict().get(
            ori_severity, resource_service.get_msg('alarm.property.unknown'))

    def _trans_alarm_status(self, ori_status):
        return self.get_alarm_status_dict().get(
            ori_status, resource_service.get_msg('alarm.property.unknown'))

    @staticmethod
    def _trans_alarm_time(ori_time_stamp):
        return time.strftime("%Y-%m-%d %H:%M:%S", time.gmtime(ori_time_stamp))
