#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/03
@file: query_replace_disk_info.py
@function: 查询更换盘的信息
@modify:
"""
import time

from py.common.entity.item_status import ItemStatus
from py.common.service import resource_service, logger_factory
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.common.service.connection.ssh_connection_service import SshService, Cmd
from py.fusion_cube.common.record import record_node_disk_info, record_raid_card_type
from py.fusion_cube.common.context import disk_context_util, common_context_util
from py.fusion_cube.common.service.disk_init_factory.disk_info_query_trans_dict import DEV_ROLE_TRANS_DICT
from py.fusion_cube.common.service.disk_init_factory.init_service_factory import DiskInitServiceFactory
from py.fusion_cube.common.service.hardware_operation_factory.operation_factory import HardwareOperationFactory
from py.fusion_cube.common.service.os_util import disk_info_util
from py.fusion_cube.common.service.os_util import raid_card_util
from py.fusion_cube.common.service.os_util.raid_card_util import raid_card_version_is_1880

NORMAL = 0
# 查询超时时间 20分钟
QUERY_NODE_STATUS_TIMEOUT = 20 * 60
# 默认重试间隔
DEFAULT_INTERVAL = 10


def execute_in_replace_cache(context):
    before_replace_node_disks = record_node_disk_info.get_recorded_node_disks(
        context)
    replace_disk_slot = disk_context_util.get_disk_slot(context)
    before_replace_slot_2_disk = _get_slot_2_disk(before_replace_node_disks)
    after_replace_slot_2_disk = _query_after_replace_slot_2_disk(context)
    replace_disk_slot_int = _int_slot(replace_disk_slot)
    # 小于零时为硬盘不在位场景
    if replace_disk_slot_int < 0:
        return _handle_not_in_position_replace_disk(
            after_replace_slot_2_disk, before_replace_slot_2_disk, context)
    else:
        return _handle_in_position_replace_disk(
            context, after_replace_slot_2_disk, replace_disk_slot)


def execute_in_replace_pilot(context):
    before_replace_slot_2_disk = _get_slot_2_disk(
        record_node_disk_info.get_recorded_node_disks(context))
    after_replace_slot_2_disk = _query_after_replace_slot_2_disk(context)
    replace_disk_slot_int = _int_slot(disk_context_util.get_disk_slot(context))
    if not disk_context_util.is_disk_exist(context):
        return _handle_not_in_position_replace_disk(
            after_replace_slot_2_disk, before_replace_slot_2_disk, context)
    else:
        return _handle_in_position_replace_disk(
            context, after_replace_slot_2_disk, replace_disk_slot_int)


@auto_brush_progress(60)
def execute_in_replace_sys(context):
    logger = logger_factory.create_logger(__file__)
    # 两大洋设备需要检查OS是否已启动
    if common_context_util.is_proprietary_hardware(context):
        logger.info("start check node power status.")
        flag, err_msg = _check_os_is_startup(context, logger)
        if not flag:
            return ItemStatus.FAILED, err_msg
    # 硬RAID查询
    if record_raid_card_type.get_raid_card_type(context):
        return _query_hard_card_replace_sys_disk(context)
    before_replace_slot_2_disk = _get_slot_2_disk(
        record_node_disk_info.get_recorded_node_disks(context))
    after_replace_slot_2_disk = _query_after_replace_slot_2_disk(context)
    result, msg = _handle_not_in_position_replace_disk(
        after_replace_slot_2_disk, before_replace_slot_2_disk, context)
    # 系统盘新盘接口与旧盘查询接口不一致，新盘Byte容量保存在capacityForByte中
    if result == ItemStatus.SUCCESS:
        new_disk = context.getDataAfterReplace()
        new_disk.setCapacity(new_disk.getCapacityForByte())
    return result, msg


def _query_hard_card_replace_sys_disk(context):
    old_disk = context.getData()
    new_disks = DiskInitServiceFactory(context).get_sys_init_service() \
        .query_node_sys_disks(disk_context_util.get_belong_mgmt_ip(context))
    for disk in new_disks:
        if disk.getSlotNo() == old_disk.getSlotNo():
            if disk.getEsn() == old_disk.getEsn():
                return ItemStatus.FAILED, \
                       resource_service.get_msg("not.found.replace.disk")
            else:
                context.setDataAfterReplace(disk)
                return ItemStatus.SUCCESS, ""
    return ItemStatus.FAILED, resource_service.get_msg(
        "not.found.replace.disk")


@auto_brush_progress(60)
def execute_in_replace_meta(context):
    replace_disk_slot = disk_context_util.get_disk_slot(context)
    after_replace_slot_2_disk = _query_after_replace_slot_2_disk(context)
    before_replace_slot_2_disk = _get_slot_2_disk(
        record_node_disk_info.get_recorded_node_disks(context))
    if replace_disk_slot not in after_replace_slot_2_disk:
        # 若是3152卡，还需要将工作模式设置为5
        flag, error_msg = raid_card_util.raid_card_set_work_mode(context)
        if not flag:
            return ItemStatus.FAILED, error_msg
        # 非3152/3008/1880卡，通过storcli设置Jbod
        flag, description_info = _modify_meta_disk_status_to_jbod_with_judge(
            context, replace_disk_slot)
        if not flag:
            return ItemStatus.FAILED, resource_service.get_msg(
                "meta.disk.modify.jbod.failed").format(description_info)
        # 1880卡，通过hiraidadm设置Jbod
        flag, desc = _modify_1880_meta_disk_to_jbod(context)
        if not flag:
            return ItemStatus.FAILED, resource_service.get_msg(
                "meta.disk.modify.jbod.failed").format(description_info)

        # 修改为jbod后重新查询盘信息
        after_replace_slot_2_disk = \
            _polling_query_after_replace_slot_2_disk(
                context, replace_disk_slot, after_replace_slot_2_disk)
    # 先查询同槽位的
    status, msg = _handle_in_position_replace_disk(
        context, after_replace_slot_2_disk, replace_disk_slot)
    if status == ItemStatus.SUCCESS:
        return status, msg
    # 再查询新增槽位盘（只查空闲盘）
    return _handle_not_in_position_replace_disk(after_replace_slot_2_disk,
                                                before_replace_slot_2_disk,
                                                context, True)


def _modify_1880_meta_disk_to_jbod(context):
    ssh_service = SshService(context.getNode())
    if not raid_card_version_is_1880(ssh_service):
        return True, ""
    return disk_info_util.modify_1880_meta_disk_to_jbod(ssh_service)


def _modify_meta_disk_status_to_jbod_with_judge(context, replace_disk_slot):
    ssh_service = SshService(context.getNode())
    if disk_info_util.need_jbod(ssh_service):
        return disk_info_util.modify_metadata_disk_status_to_jbod(
            ssh_service, replace_disk_slot)
    return True, ""


def _polling_query_after_replace_slot_2_disk(
        context, replace_disk_slot, after_replace_slot_2_disk):
    start_time = time.time()
    # 轮询5分钟，直到查到同槽位盘
    while time.time() - start_time < 300:
        after_replace_slot_2_disk = _query_after_replace_slot_2_disk(context)
        if replace_disk_slot in after_replace_slot_2_disk:
            return after_replace_slot_2_disk
        time.sleep(10)
    return after_replace_slot_2_disk


def _query_after_replace_slot_2_disk(context):
    after_replace_node_disks = DiskInitServiceFactory(context) \
        .get_match_init_service().query_node_disks(
        disk_context_util.get_belong_mgmt_ip(context))
    return _get_slot_2_disk(after_replace_node_disks)


def _handle_in_position_replace_disk(context, after_replace_slot_2_disk,
                                     replace_disk_slot):
    if replace_disk_slot not in after_replace_slot_2_disk:
        return ItemStatus.FAILED, resource_service.get_msg(
            "not.found.replace.disk")
    new_disk = after_replace_slot_2_disk.get(replace_disk_slot)
    if disk_context_util.get_disk_sn(context) == new_disk.getEsn():
        return ItemStatus.FAILED, resource_service.get_msg(
            "found.same.replace.disk")
    context.setDataAfterReplace(new_disk)
    return ItemStatus.SUCCESS, ""


def _handle_not_in_position_replace_disk(after_replace_slot_2_disk,
                                         before_replace_slot_2_disk, context,
                                         ignore_role_disk=False):
    new_slot_disks = _get_new_slot_disks(
        before_replace_slot_2_disk, after_replace_slot_2_disk, ignore_role_disk)
    if not new_slot_disks:
        return ItemStatus.FAILED, resource_service.get_msg(
            "not.found.replace.disk")
    if len(new_slot_disks) == 1:
        context.setDataAfterReplace(new_slot_disks[0])
        return ItemStatus.SUCCESS, ""
    # 发现多个新盘时无法确认哪个是更换盘
    return ItemStatus.FAILED, resource_service.get_msg(
        "found.more.than.one.replace.disk").format(
        str([disk.getEsn() for disk in new_slot_disks]))


def _int_slot(slot):
    try:
        return int(slot)
    except ValueError:
        return -1


def _get_slot_2_disk(disks):
    slot_2_disk = dict()
    for disk in disks:
        slot_2_disk[disk.getSlotNo()] = disk
    return slot_2_disk


def _get_new_slot_disks(before_replace_slot_2_disk,
                        after_replace_slot_2_disk, ignore_role_disk=False):
    new_slot_disks = list()
    for slot, disk in after_replace_slot_2_disk.items():
        # 忽略有角色信息的新盘
        if ignore_role_disk and disk.getMediaRole().toString() != \
                DEV_ROLE_TRANS_DICT.get("no_use"):
            continue
        if slot not in before_replace_slot_2_disk:
            new_slot_disks.append(disk)
    return new_slot_disks


def execute(context):
    old_disks = record_node_disk_info.get_recorded_node_disks(context)
    new_disks = DiskInitServiceFactory(context) \
        .get_match_init_service().query_node_disks(
        disk_context_util.get_belong_mgmt_ip(context))
    old_sn_2_disk = trans_disks_to_sn_2_disk(old_disks)
    new_sn_2_disk = trans_disks_to_sn_2_disk(new_disks)
    replace_disks = find_replace_disks(new_sn_2_disk, old_sn_2_disk)

    if not replace_disks:
        return ItemStatus.FAILED, resource_service.get_msg(
            "not.found.replace.disk")
    elif len(replace_disks) == 1:
        context.setDataAfterReplace(replace_disks[0])
        return ItemStatus.SUCCESS, ""
    else:
        # 发现多个新盘时无法确认哪个是更换盘
        return ItemStatus.FAILED, resource_service.get_msg(
            "found.more.than.one.replace.disk").format(
            str([disk.getEsn() for disk in replace_disks]))


def find_replace_disks(new_sn_2_disk, old_sn_2_disk):
    replace_disks = list()
    for esn in new_sn_2_disk.keys():
        # 排除空sn场景
        if esn and esn not in old_sn_2_disk.keys():
            replace_disks.append(new_sn_2_disk.get(esn))
    return replace_disks


def trans_disks_to_sn_2_disk(disks):
    sn_2_disk = dict()
    for disk in disks:
        sn_2_disk[disk.getEsn()] = disk
    return sn_2_disk


def _check_os_is_startup(context, logger):
    """
    检查当前节点OS可用，依次检查:节点是否上电->节点状态是否正常->关键进程是否启动
    :param context: 上下文
    :param logger:  日志记录器
    :return:  True:可用；False：不可用
    """
    service = HardwareOperationFactory(context).get_match_init_service()
    start_time = time.time()
    if not service.is_device_power_on(start_time, QUERY_NODE_STATUS_TIMEOUT, DEFAULT_INTERVAL):
        logger.error("node {} power may be off.".format(context.getClusterNode().getManagementIp()))
        common_context_util.set_cur_task_suggestion(context,
                                                    resource_service.get_msg("device.not.power.on.suggestion"))
        return False, resource_service.get_msg("device.not.power.on")
    if not _check_node_status_normal(context, start_time, QUERY_NODE_STATUS_TIMEOUT, DEFAULT_INTERVAL):
        logger.error("node {} status is abnormal.".format(context.getClusterNode().getManagementIp()))
        common_context_util.set_cur_task_suggestion(context,
                                                    resource_service.get_msg("node.status.abnormal.suggestion"))
        return False, resource_service.get_msg("node.status.abnormal")
    if not _check_node_management_process_startup(context, start_time, QUERY_NODE_STATUS_TIMEOUT, DEFAULT_INTERVAL):
        logger.error("node {} key processes are not started.".format(context.getClusterNode().getManagementIp()))
        common_context_util.set_cur_task_suggestion(context,
                                                    resource_service.get_msg(
                                                        "node.management.agent.not.started.suggestion"))
        return False, resource_service.get_msg("node.management.agent.not.started")
    return True, ""


def _check_node_status_normal(context, start_time, time_out, interval):
    while True:
        if time.time() - start_time > time_out:
            return False
        rest_service = CubeRestService(context.getCluster())
        res_dict = rest_service.exec_get("/api/v2/cluster/servers")
        select_node_manage_ip = disk_context_util.get_belong_mgmt_ip(context)
        for node_data in res_dict.get("data", []):
            management_ip = node_data.get("management_ip", "")
            if management_ip != select_node_manage_ip:
                continue
            node_status = node_data.get("status", -1)
            if node_status == NORMAL:
                return True
            else:
                break
        time.sleep(interval)


def _check_node_management_process_startup(context, start_time, time_out, interval):
    while True:
        if time.time() - start_time > time_out:
            return False
        ssh_service = SshService(context.getNode())
        cmd = "ps -ef|grep /opt/dfv/oam/oam-u/nma/bin/nmagent.bin|grep -v grep |wc -l"
        cli_ret = ssh_service.exec_cmd(Cmd(cmd))
        for line in cli_ret.splitlines():
            if line.strip() == "1":
                return True
        time.sleep(interval)
