#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/14
@file: setting_server_storage_mode.py
@function: 服务器置为维护模式/关闭维护模式
@modify:
"""
from py.common.entity.exception import ConnectionException
from py.common.entity.item_status import ItemStatus
from py.common.service import resource_service
from py.common.service.connection.cube_rest_connection_service import CubeRestService
from py.common.service.connection.ssh_connection_service import SshService
from py.fusion_cube.common.constant import ClusterRestUri
from py.fusion_cube.common.context import disk_context_util, \
    common_context_util
from py.fusion_cube.common.service.cluster import query_disk_info_service, \
    cluster_info_util
from py.fusion_cube.common.service.os_util import disk_info_util

SERVER_NORMAL_MODE = 0
SERVER_MAINTENANCE_MODE = 1


def setting_to_normal(context):
    if _is_involved_setting_maintenance(context):
        return SettingServerStorageMode(context).setting_to_normal()
    return ItemStatus.NOT_INVOLVED, ""


def setting_to_maintenance(context):
    if _is_involved_setting_maintenance(context):
        return SettingServerStorageMode(context).setting_to_maintenance()
    return ItemStatus.NOT_INVOLVED, ""


def _is_involved_setting_maintenance(context):
    """
    判断是否涉及设置维护模式
    已知不涉及：1. zk集群超冗余故障 2. 系统盘且不为专有硬件 3. 当前节点未加入存储池
    :param context: 上下文
    :return: True:涉及 False:不涉及
    """
    if cluster_info_util.is_zk_fault_exceed_limit(CubeRestService(
            context.getCluster())):
        return False
    if disk_context_util.is_sys_module(context) and \
            not common_context_util.is_proprietary_hardware(context):
        return False
    if not _is_node_has_pool(context):
        return False
    return True


def _is_node_has_pool(context):
    mgr_ip = disk_context_util.get_belong_mgmt_ip(context)
    return mgr_ip in query_disk_info_service.query_all_pool_node_mgr_ips(context)


class SettingServerStorageMode(object):
    def __init__(self, context):
        self._context = context
        self._rest_service = CubeRestService(context.getCluster())
        self._node_mgr_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._ssh_service = SshService(context.getNode())

    def setting_to_normal(self):
        """
        节点切换为正常模式
        """
        if self._query_storage_model() == SERVER_MAINTENANCE_MODE:
            try:
                self._set_storage_mode(SERVER_NORMAL_MODE)
            except ConnectionException as ex:
                return ItemStatus.FAILED, resource_service.get_msg(
                    "failed.setting.server.normal").format(
                    str(self._node_mgr_ip),
                    CubeRestService.get_description(ex.response))
            return ItemStatus.SUCCESS, ""
        return ItemStatus.NOT_INVOLVED, ""

    def setting_to_maintenance(self):
        """
        节点置为维护模式
        """
        try:
            self._set_storage_mode(SERVER_MAINTENANCE_MODE)
        except ConnectionException as ex:
            # 兼容当前节点没有OSD进程时，设置维护模式失败的情况
            if not disk_info_util.is_osd_process_online(self._ssh_service):
                return ItemStatus.NOT_INVOLVED, ""
            return ItemStatus.FAILED, resource_service.get_msg(
                "failed.setting.server.maintenance").format(
                str(self._node_mgr_ip),
                CubeRestService.get_description(ex.response))
        return ItemStatus.SUCCESS, ""

    def _set_storage_mode(self, model):
        self._rest_service.exec_post(
            ClusterRestUri.SET_SERVER_MODEL,
            {"nodeMgrIp": self._node_mgr_ip, "storageMode": model})

    def _query_storage_model(self):
        res_dict = self._rest_service.exec_get(
            ClusterRestUri.QUERY_SERVER_MODEL,
            params={"nodeMgrIp": self._node_mgr_ip})
        return res_dict.get("storageMode", -1)
