#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2024. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/09/25
@file: common_context_util.py
@function: 上下文工具
@modify:
"""
from py.common.adapter import java_adapter
from py.common.service import logger_factory
from py.fusion_storage.common.context import disk_context_util
from py.fusion_storage.common.record.record_raid_card_type import HAS_HARD_RAID_CARD


def set_cur_task_suggestion(context, suggestion):
    context.getCurrentTask().updateSuggestion(suggestion)


def is_proprietary_hardware(context):
    """
    判断当前所在节点硬件是否为专有硬件
    :param context: 上下文
    :return: 是否是专有硬件
    """
    return context.getClusterNode().isProprietaryHardware()


def is_proprietary_hardware_with_soft_raid(context):
    """
    判断当前所在节点硬件是否为专有硬件加软raid
    :param context: 上下文
    :return: 是否是专有硬件
    """
    if disk_context_util.is_sys_module(context):
        return context.getClusterNode().isProprietaryHardware() and not context.getEnv().get(HAS_HARD_RAID_CARD)
    return context.getClusterNode().isProprietaryHardware()


def is_two_oceans_hardware(context):
    """
    判断当前所在节点硬件是否是除了东海、司南的专有硬件，东海、司南支持硬Raid卡。
    :param context: 上下文
    :return: 是否是两洋专有硬件
    """
    return context.getClusterNode().isProprietaryHardware() and \
           not context.getClusterNode().isSiNanNode() and \
           not context.getClusterNode().isEastSeaNode()


def is_sinan_server(context):
    """
    判断当前所在节点硬件是否为司南设备
    :param context: 上下文
    :return: 是否是司南硬件
    """
    return context.getClusterNode().isSiNanNode()


def is_a310_server(context):
    """
    判断当前所在节点硬件是否为新大西洋硬件
    :param context: 上下文
    :return: 是否是新大西洋硬件
    """
    return context.getClusterNode().isA310Node()


def is_pacific_server(context):
    """
    判断当前所在节点硬件是否为太平洋硬件
    :param context: 上下文
    :return: 是否是太平洋硬件
    """
    return context.getClusterNode().isPacificNode()


def is_east_sea_server(context):
    """
    判断当前所在节点硬件是否为东海设备
    :param context: 上下文
    :return: 是否是东海设备
    """
    return context.getClusterNode().isEastSeaNode()


def is_hg_server(context):
    """
    判断当前所在节点是否为海光服务器
    :param context: 上下文
    :return: 是否是海光服务器
    """
    return context.getClusterNode().isHGNode()


def get_cluster_all_node_ip(context):
    return [node.getManagementIp()
            for node in context.getCluster().getClusterNodes()]


def get_product_version(context):
    """
    获取产品软件版本
    :param context: 上下文
    :return: 产品软件版本
    """
    return context.getFruDevice().getBusinessDevice() \
        .getDevNode().getProductVersion()


def get_active_fsm_node(context):
    return java_adapter.get_dev_info_util_class() \
        .getActiveFsmNode(context.getFsmNodeList())


def get_task_auth_nodes_list(context):
    node_ids = context.getNodeIdList()
    if not node_ids:
        return list()
    bmc_nodes_map = java_adapter.get_device_util_instance().getAuthDevNodeByNodeIdList(context.getFruDevice(), node_ids)
    logger_factory.create_logger(__file__).info("get task auth node_ids: {}".format(node_ids))
    return bmc_nodes_map.values()


def get_proprietary_hardware_storage_ip_list(context):
    """
    获取两大洋的存储节点列表
    :param context: 上下文
    :return: IP列表
    """
    node_ip_list = []
    for node in context.getCluster().getClusterNodes():
        if node.isSiNanNode() or node.isEastSeaNode():
            continue
        if node.isProprietaryHardware() and is_node_has_storage_role(node):
            node_ip_list.append(node.getManagementIp())
    return node_ip_list


def is_node_has_storage_role(node):
    return node.getRoles().contains('storage')


def get_current_node_management_ip(context):
    return context.getClusterNode().getManagementIp()


def should_task_terminate(context):
    return context.isNeedTerminate()
