#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.


"""
@version: SmartKit V200R007C00
@time: 2021/08/09
@file: record_meta_usage_mode.py
@function: 记录元数据盘的使用方式
@modify:
"""
from py.common.entity.exception import BusinessException
from py.common.service import logger_factory, resource_service
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.rest_connection_service import RestService
from py.common.service.connection.ssh_connection_service import SshService, Cmd
from py.fusion_storage.common import constant
from py.fusion_storage.common.context import disk_context_util
from py.fusion_storage.common.service.cluster import query_disk_info_service

META_USAGE_MODE = "META_USAGE_MODE"
IS_CONTROL_USE_ALONE = "IS_CONTROL_USE_ALONE"
IS_REPLICATION_USE_ALONE = "IS_REPLICATION_USE_ALONE"


def record(context):
    # 当前盘不在位，且当前节点有复制集群时，工具暂不支持
    if not disk_context_util.is_disk_exist(context) and \
            _has_replication_cluster(context):
        logger_factory.create_logger(__file__).error(
            'not support scene: disk not exist and has replication cluster')
        raise BusinessException(err_msg=resource_service.get_msg(
            "meta.disk.not.exist.and.has.replication.cluster.not.support"))

    ssh_service = SshService(context.getNode())
    # 控制集群元数据盘盘符, 查询出来盘符的为软连接, 需要readlink获取真实盘符
    control_meta_drive_letter_cmd = ssh_cmd_util.get_control_meta_drive_letter_cmd(ssh_service.is_mini_system)
    if ssh_service.is_mini_system:
        control_meta_drive_letter = \
            _query_drive_letter(ssh_service, control_meta_drive_letter_cmd)
    else:
        control_meta_drive_letter = _read_link(
            ssh_service, _query_drive_letter(ssh_service, control_meta_drive_letter_cmd))
    # 复制集群元数据盘盘符
    replication_meta_drive_letter_cmd = ssh_cmd_util.get_replication_meta_drive_letter_cmd(ssh_service.is_mini_system)
    replication_meta_drive_letter = \
        _query_drive_letter(ssh_service, replication_meta_drive_letter_cmd)
    # 当前所选盘的盘符
    cur_meta_drive_letter = query_disk_info_service.query_disk_drive_letter(
        context, disk_context_util.get_disk_sn(context))

    # 元数据盘符为空为不在位, 但是肯定属于控制集群
    if cur_meta_drive_letter == '':
        context.getEnv()[IS_CONTROL_USE_ALONE] = \
            replication_meta_drive_letter != control_meta_drive_letter
    else:
        context.getEnv()[IS_CONTROL_USE_ALONE] = \
            cur_meta_drive_letter == control_meta_drive_letter and \
            cur_meta_drive_letter != replication_meta_drive_letter
    # 不支持此场景
    context.getEnv()[IS_REPLICATION_USE_ALONE] = False

    logger = logger_factory.create_logger(__file__)
    logger.info('Meta Disk Init Usage Mode End: is_control_use_alone: {}, '
                'is_replication_use_alone: {}'
                .format(context.getEnv()[IS_CONTROL_USE_ALONE],
                        context.getEnv()[IS_REPLICATION_USE_ALONE]))


def _query_drive_letter(ssh_service, cmd):
    res = ssh_service.exec_cmd(Cmd(cmd)).splitlines()[1]
    if ssh_service.is_mini_system and len(res.split("=")) > 1:
        return res.split("=")[1]
    return res


def _read_link(ssh_service, link):
    read_link_cmd = "readlink -f {}".format(link)
    res = ssh_service.exec_cmd(Cmd(read_link_cmd))
    return res.splitlines()[1]


def is_control_use_alone(context):
    return context.getEnv().get(IS_CONTROL_USE_ALONE)


def is_replication_use_alone(context):
    return context.getEnv().get(IS_REPLICATION_USE_ALONE)


def _has_replication_cluster(context):
    rest_service = RestService(context.getCluster())
    res_dict = rest_service.exec_post(constant.ClusterRestUri.DSWARE_SERVER_CMD, {
        "serviceType": "dr",
        "subOp": "QueryStorageServiceProcessInfo",
        "op": "drCmd"
    })
    for cmd_data in res_dict.get("serviceCmdData", []):
        mgr_ip = cmd_data.get('ip')
        if mgr_ip == disk_context_util.get_belong_mgmt_ip(context):
            return True
    return False
