#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/06/09
@file: chassis_resource_util.py
@function:
@modify:
"""
import time

from py.fusion_storage.common.constant import RaidCardType

CHASSIS_URI = "/redfish/v1/Chassis/"
CHASSIS_BOARDS_URI = CHASSIS_URI + "{}/Boards/"
MANAGER_URI = "/redfish/v1/Managers/"
GET_SYS_RESOURCE_URI = "/redfish/v1/Systems/{}"
POWER_OFF_URI = "/redfish/v1/Systems/{}/Actions/ComputerSystem.Reset"
# 查询设备下电超时时间 5分钟
QUERY_TASK_TIMEOUT = 5 * 60
# 默认重试间隔
DEFAULT_INTERVAL = 10


def get_assign_resource_path(connection):
    """
    获取指定资源的访问路径
    :param connection: 连接
    :return: 路径
    """
    response = connection.exec_get(CHASSIS_URI)
    # 如果是一框两节点的场景，优先使用节点级的信息而不是框的，一般是专有硬件场景
    if len(response.get("Members")) >= 2:
        return response.get("Members")[1].get("@odata.id")
    return response.get("Members")[0].get("@odata.id")


def get_resource_id(connection):
    """
    获取资源id
    :param connection: 连接
    :return: resource_id 资源id
    """
    return get_assign_resource_path(connection).split("/")[-1]


def get_boards_resource_info(connection):
    """
    获取boards资源信息
    :param connection: 连接
    :return: resource信息
    """
    resource_id = get_resource_id(connection)
    return connection.exec_get(CHASSIS_BOARDS_URI.format(
        resource_id))


def get_assign_board_infos(connection, board_name):
    """
    获取指定board信息
    :param connection: 连接
    :param board_name: 已知有DiskBackplane、PCIeRiserCard、RAIDCard
    :return: dict：大概如下结构，包含MaxNumberSupported和Links
    {
        "MaxNumberSupported": 5,
        "Links": [
            {
                "@odata.id": "/redfish/v1/Chassis/1/Boards/chassisDiskBP1"
            },
            {
                "@odata.id": "/redfish/v1/Chassis/1/Boards/chassisDiskBP3"
            },
            {
                "@odata.id": "/redfish/v1/Chassis/1/Boards/chassisDiskBP5"
            }
        ]
    }
    """
    resource = get_boards_resource_info(connection)
    # 在没有board情况下，会没有oem
    if "Oem" in resource:
        object_tmp = resource.get("Oem").get("Huawei")
        if not object_tmp:
            return resource.get("Oem").get("Public").get("Boards").get(board_name)
        return object_tmp.get("Boards").get(board_name)
    return dict()


def check_description_contain_raid_card(description):
    support_list = [RaidCardType.SAS3108, RaidCardType.SAS3152,
                    RaidCardType.SAS3408, RaidCardType.SAS3416,
                    RaidCardType.SAS3508, RaidCardType.SAS3908,
                    RaidCardType.RAID1880I16, RaidCardType.RAID1880I8]
    return any(card_type in description for card_type in support_list)


def get_raid_type_by_raid_card(connection):
    """
    通过RAIDCard信息获取RAID卡类型
    :param connection: 连接
    :return: RAID卡类型
    """
    raid_infos = get_assign_board_infos(connection, "RAIDCard")
    if not raid_infos:
        return ""
    for link in raid_infos.get("Links", {}):
        raid_info_uri = link.get("@odata.id")
        if raid_info_uri:
            res_dict = connection.exec_get(raid_info_uri)
            if res_dict.get("Description") \
                    and check_description_contain_raid_card(res_dict.get("Description")):
                return res_dict.get("Description")
    return ""


def get_raid_type_by_pcie(connection):
    """
    通过PCIE信息获取RAID卡类型
    :param connection: 连接
    :return: RAID卡类型
    """
    resource_id = get_resource_id(connection)
    resource = connection.exec_get(CHASSIS_URI + resource_id)
    if "Links" not in resource:
        return ""
    for data_dic in resource.get("Links").get("PCIeDevices"):
        response = connection.exec_get(data_dic.get("@odata.id"))
        if response.get("Description") \
                and check_description_contain_raid_card(response.get("Description")):
            return response.get("Description")
        if response.get("Model") \
                and check_description_contain_raid_card(response.get("Model")):
            return response.get("Model")
    return ""


def get_raid_type_by_pcie_extend(connection):
    """
    通过PCIE拓展信息获取RAID卡类型
    :param connection: 连接
    :return: raid卡类型
    """
    resource_id = get_resource_id(connection)
    resource = connection.exec_get(CHASSIS_URI + resource_id)
    if "Links" not in resource:
        return ""
    for data_dic in resource.get("Links").get("PCIeDevices"):
        response = connection.exec_get(data_dic.get("@odata.id"))
        product_name = response.get("Oem", {}).get("Huawei", {}).get("ProductName")
        if product_name and check_description_contain_raid_card(product_name):
            return product_name
    return ""


def get_device_slot_id(connection):
    """
    获取设备槽位号
    :param connection: 连接
    :return: 槽位号
    """
    response = connection.exec_get(MANAGER_URI)
    if "Members" in response and len(response.get("Members")) > 0:
        slot_info = response.get("Members")[0].get("@odata.id")
        if slot_info and len(slot_info.strip()) > 0:
            return slot_info.split("/")[4]
    return ""


def is_device_power_on(connection, slot_id):
    """
    获取设备电源状态
    :param connection: 连接
    :param slot_id: 槽位号
    :return: 电源状态，错误信息
    """
    response = connection.exec_get(GET_SYS_RESOURCE_URI.format(slot_id))
    return "PowerState" in response and response.get("PowerState") == "On"


def power_off_device(connection):
    """
    下电设备
    :param connection: 连接
    :return: True:成功/False:失败
    """
    slot_id = get_device_slot_id(connection)
    if not is_device_power_on(connection, slot_id):
        return True
    reset_type = dict(ResetType="GracefulShutdown")
    connection.exec_post(POWER_OFF_URI.format(slot_id), reset_type)
    start_time = time.time()
    while time.time() - start_time < QUERY_TASK_TIMEOUT:
        if not is_device_power_on(connection, slot_id):
            return True
        time.sleep(DEFAULT_INTERVAL)
    return False
