#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2024. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2022/01/06
@file: system_disk_selector_factory.py
@function: 系统盘查询相关操作工厂类
@modify:
"""
import re

from py.common.entity.exception import BusinessException
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.ssh_connection_service import Cmd
from py.fusion_storage.common.context import common_context_util

ERROR_CMD_RES = ['No such file or directory']
BLANK_PATTERN = re.compile(r'\s+')


def handle_res_func(res):
    if len(res.splitlines()) > 2:
        return '\n'.join(res.strip().splitlines()[1:-1])
    return res


def _get_selector(context, ssh, logger, is_pre_check):
    if common_context_util.is_a310_server(context):
        return A310HardwareSelector(ssh, logger, is_pre_check)
    if common_context_util.is_proprietary_hardware(context):
        return ProprietaryHardwareSelector(ssh, logger, is_pre_check)
    return CommonKunPengSelector(ssh, logger, is_pre_check)


def _cmd_has_error_msg(disk_ata):
    for item in ERROR_CMD_RES:
        if item in disk_ata:
            return True
    return False


class SystemDiskSelectorFactory(object):
    """
    系统盘查询相关工厂类
    """

    def __init__(self, context, ssh, logger, is_pre_check=False):
        self._selector = _get_selector(context, ssh, logger, is_pre_check)

    def exec_for_system_disk(self):
        """
        查询系统盘盘符名称
        :return: 系统盘名称列表
        """
        return self._selector.exec_for_system_disk()

    def find_disk_detail(self):
        """
        查询系统盘盘符树，并封装为DiskInfo对象
        :return: DiskInfo
        """
        return self._selector.find_disk_detail()


class Selector(object):
    """
    系统盘盘符查询执行器父类
    不同设备类型具有不同实现子类
    ProprietaryHardwareSelector 专有硬件实现
    CommonKunPengSelector 通用鲲鹏实现
    """

    def __init__(self, ssh, logger, is_pre_check):
        self._ssh = ssh
        self._system_disk_list = list()
        self._logger = logger
        self._is_pre_check = is_pre_check
        self._partitioned_disk = list()
        self._not_partition_disk = list()

    def _find_system_disk(self, cmds):
        for cmd in cmds:
            disk_ata = self._ssh.exec_cmd_handle_res(Cmd(cmd), handle_res_func=handle_res_func)
            if disk_ata and not _cmd_has_error_msg(disk_ata):
                self._system_disk_list.append(disk_ata)
                continue
            if not self._is_pre_check:
                raise BusinessException(err_code='find.system.disk.failed', err_msg=cmd)

    def exec_for_system_disk(self):
        raise BusinessException(err_msg='method not implement')

    def find_disk_detail(self):
        self._check_system_disk_has_find()
        cmd = 'lsblk'
        disk_info = self._ssh.exec_cmd_handle_res(Cmd(cmd), handle_res_func=handle_res_func)
        # 找出已分区的系统盘
        self._find_partitioned_disk(disk_info)
        # 找出未分区系统盘
        self._find_not_partition_disk(self._partitioned_disk)
        return self._partitioned_disk, self._not_partition_disk, disk_info

    def _find_partitioned_disk(self, disk_info):
        re_s = '\s.+|'.join(self._system_disk_list)
        pattern = re.compile(r'({}\s.+)(?=disk\s+\n[^a-zA-Z])'.format(re_s))
        partitioned_disk_list = pattern.findall(disk_info)
        # 根据文档，系统盘只会有两个，更换条件下一个未分区，一个已分区
        for item in partitioned_disk_list:
            partition_disk = BLANK_PATTERN.split(item)[0]
            self._partitioned_disk.append(DiskInfoParse.parse_disk_info(disk_info, partition_disk))

    def _find_not_partition_disk(self, partition_disk):
        if len(partition_disk) == len(self._system_disk_list):
            # 已分区盘数量和系统盘数量相同，没有未分区盘
            return
        disk_name = list()
        for disk in partition_disk:
            disk_name.append(disk.name)
        for item in self._system_disk_list:
            if item not in disk_name:
                self._not_partition_disk.append(item)

    def _check_system_disk_has_find(self):
        if len(self._system_disk_list) < 1:
            self.exec_for_system_disk()
        if len(self._system_disk_list) < 1:
            raise BusinessException(err_msg='system disk not found')


class ProprietaryHardwareSelector(Selector):
    """
    专有硬件系统盘相关查询类
    """

    def __init__(self, ssh, logger, is_pre_check):
        super(ProprietaryHardwareSelector, self).__init__(ssh, logger, is_pre_check)

    def exec_for_system_disk(self):
        self._find_system_disk(ssh_cmd_util.get_proprietary_hardware_system_disk_cmd(self._ssh.is_mini_system))
        return self._system_disk_list


class A310HardwareSelector(Selector):
    """
    新大西洋硬件系统盘相关查询类
    """

    def __init__(self, ssh, logger, is_pre_check):
        super(A310HardwareSelector, self).__init__(ssh, logger, is_pre_check)

    def exec_for_system_disk(self):
        self._find_system_disk(ssh_cmd_util.get_a310_system_disk_cmd(self._ssh.is_mini_system))
        return self._system_disk_list


class CommonKunPengSelector(Selector):
    """
    通用鲲鹏设备系统盘相关查询类
    """

    def __init__(self, ssh, logger, is_pre_check):
        super(CommonKunPengSelector, self).__init__(ssh, logger, is_pre_check)

    def exec_for_system_disk(self):
        self._find_system_disk(ssh_cmd_util.get_common_kunpeng_system_disk_cmd(self._ssh.is_mini_system))
        return self._system_disk_list


class DiskInfo(object):
    def __init__(self, name, rm=None, size=None, ro=None, disk_type=None, parent=None, mount_point=None):
        self.name = name
        self.rm = rm
        self.size = size
        self.ro = ro
        self.type = disk_type
        self.mount_point = mount_point
        self.parent = parent
        self.children = list()
        self.first_char_index = 0
        self.sync_complete = False


class DiskInfoParse(object):
    @staticmethod
    def parse_disk_info(disk_info, partitioned_disk):
        '''
        解析盘符树
        :param disk_info: lsblk命令的回显树
        :param partitioned_disk: 已分区的盘符
        :return:盘符树的对象表示
        '''

        root_disk = None
        last_line_disk = None
        disk_lines = disk_info.splitlines()
        find_disk_line = False
        for item in disk_lines:
            line_disk_info = BLANK_PATTERN.split(item.strip())
            # 找到盘符树以已分区盘符开头的一行，再到下一个以字母开头的行，中间即已分区盘符的具体信息
            if item.startswith(partitioned_disk):
                name, rm, size, ro, line_type = DiskInfoParse._get_split_info(line_disk_info)
                find_disk_line = True
                root_disk = DiskInfo(name, rm, size, ro, line_type)
                last_line_disk = root_disk
                continue
            # 已找到已分区盘符， 下一行又是以字母开头,说明是新的盘符开始
            if find_disk_line and item[0].isalpha():
                break
            if find_disk_line:
                #  取当前这一行的字母起始位置，用于判断当前行和上一行是同级还是子集还是父集
                search_obj = re.search(r'[a-z]', item, re.IGNORECASE)
                if not search_obj:
                    continue
                name, rm, size, ro, line_type = DiskInfoParse._get_split_info(line_disk_info)
                line_disk = DiskInfo(name, rm, size, ro, line_type)
                DiskInfoParse._set_parent_children(last_line_disk, line_disk, search_obj)
                last_line_disk = line_disk
        return root_disk

    @staticmethod
    def _get_split_info(line_info):
        length = len(line_info)
        disk_name = ''
        disk_rm = ''
        disk_size = ''
        disk_ro = ''
        disk_line_type = ''
        index = 0
        if length >= 1:
            disk_name = DiskInfoParse.get_name(line_info, index)
        if length >= 3:
            disk_rm = line_info[2]
        if length >= 4:
            disk_rm = line_info[3]
        if length >= 5:
            disk_ro = line_info[4]
        if length >= 6:
            disk_line_type = line_info[5]
        return disk_name, disk_rm, disk_size, disk_ro, disk_line_type

    @staticmethod
    def get_name(line_info, index):
        if index >= len(line_info):
            return ''
        name = re.sub(r'^[^a-z]+', '', line_info[index])
        if not name:
            return DiskInfoParse.get_name(line_info, index + 1)
        return name

    @staticmethod
    def _set_parent_children(last_line_disk, line_disk, search_obj):
        fist_char_index = search_obj.span()[0]
        line_disk.first_char_index = fist_char_index
        if fist_char_index > last_line_disk.first_char_index:
            # 当前行首字母索引大于上一上首字母索引，当前上为上一行的子集
            line_disk.parent = last_line_disk
            last_line_disk.children.append(line_disk)
        elif fist_char_index == last_line_disk.first_char_index:
            # 和上一层同级
            line_disk.parent = last_line_disk.parent
            last_line_disk.parent.children.append(line_disk)
        else:
            # 比上一层高一级
            line_disk.parent = last_line_disk.parent.parent
            last_line_disk.parent.parent.children.append(line_disk)
