#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/15
@file: disk_info_util.py
@function:
@modify:
"""
import os
import re

from py.common.service import ssh_info_handle_util, logger_factory, \
    resource_service
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.ssh_connection_service import Cmd, \
    NoNeedReturnCmd, NoLogCmd, SshService
from py.fusion_storage.common.context import common_context_util
from py.fusion_storage.common.service.os_util import raid_card_util
from py.common.service.connection.rest_connection_service import RestService
from py.fusion_storage.common.constant import ClusterRestUri
from py.common.entity.exception import ConnectionException

JBOD = "JBOD"
SMART_CACHE_SWITCH_ON = 1


def get_show_disk_infos(connection):
    """
    执行show_disk.sh命令，解析回显成格式化结果
    :param connection:
    :return:
    """
    cmd = ssh_cmd_util.get_show_disk_infos_cmd(connection.is_mini_system)
    ssh_info = connection.exec_cmd(Cmd(cmd))
    # 找到第一个\n|的地方，即为去掉表格前面的其他无关信息"
    ssh_info = ssh_info[ssh_info.index("\n|") + 1:]
    return ssh_info_handle_util.parse_horizontal_table_by_chars(ssh_info, "|")


def get_mdc_cmd_101_infos(connection, pool_id=None):
    """
    执行mdc_cmd.sh 101命令，解析回显成格式化结果
    :param connection:
    :param pool_id: 801是存储池id，810及之后，都是硬盘id
    :return:
    """
    cmd = "mdc_cmd.sh 101 {}".format(pool_id) if pool_id else "mdc_cmd.sh 101"
    origin_info = connection.exec_cmd(Cmd(cmd))
    origin_info_lines = origin_info.splitlines()
    title_index = _find_title_line_index(
        origin_info_lines, ["OSD ESN", "CACHE ESN", "OSD NID"])
    field_values = ssh_info_handle_util.parse_horizontal_table_by_chars(
        "\n".join(origin_info_lines[title_index:]), "|")
    for field_value in field_values:
        if "OSD ESN" in field_value:
            sn = field_value.get("OSD ESN").split("OSD")[-1].strip()
            slot = field_value.get("DISK SLOT")
            field_value["OSD ESN"] = _remove_slot_mark(sn, slot)
    return field_values


def get_mdc_cmd_102_infos(connection):
    """
    执行mdc_cmd.sh 102命令，解析回显成格式化结果
    :param connection:
    :return:
    """
    cmd = "mdc_cmd.sh 102"
    origin_info = connection.exec_cmd(Cmd(cmd))
    origin_info_lines = origin_info.splitlines()
    title_index = _find_title_line_index(
        origin_info_lines, ["NID", "FAULT DETAIL"])
    field_values = ssh_info_handle_util.parse_horizontal_table_by_chars(
        "\n".join(origin_info_lines[title_index:]), "|")
    for field_value in field_values:
        if "NID" in field_value:
            field_value["NID"] = field_value.get("NID").split("OSD")[-1].strip()
    return field_values


def _find_title_line_index(origin_info_lines, title_line_keys):
    def keys_in_line(line):
        for key in title_line_keys:
            if key not in line:
                return False
        return True

    for index, line in enumerate(origin_info_lines):
        if keys_in_line(line):
            return index
    raise Exception("not find title line")


def get_mdc_cmd_168_main_disk_infos(connection):
    """
    执行mdc_cmd.sh 168命令，查询主存盘信息, 解析回显成格式化结果, 并将osd esn处理
    成正确的esn
    :param connection: SshService
    :return:
    """
    res_list = connection.exec_cmd(Cmd('mdc_cmd.sh 168')).splitlines()
    begin_line = 0
    end_line = len(res_list) - 1
    for index, line in enumerate(res_list):
        if "POOL OSD INFO:" in line:
            begin_line = index
        if "POOL CACHE INFO:" in line:
            end_line = index
    main_disk_list = ssh_info_handle_util.parse_horizontal_table_by_chars(
        os.linesep.join(res_list[begin_line + 1:end_line]), "|")
    # 转化OSD ESN
    for disk in main_disk_list:
        disk["OSD ESN"] = _remove_slot_mark(disk.get("OSD ESN"),
                                            disk.get("DISK SLOT"))
    return main_disk_list


def get_mdc_cmd_168_cache_disk_infos(connection):
    """
    执行mdc_cmd.sh 168命令，查询缓存盘信息, 解析回显成格式化结果
    :param connection: SshService
    :return:
    """
    res_list = connection.exec_cmd(Cmd('mdc_cmd.sh 168')).splitlines()
    begin_line = 0
    for index, line in enumerate(res_list):
        if "POOL CACHE INFO:" in line:
            begin_line = index
    return ssh_info_handle_util.parse_horizontal_table_by_chars(
        os.linesep.join(res_list[begin_line + 1:]), "|")


def _remove_slot_mark(sn, slot):
    # nvme的sn后面被加上了_槽位号
    if sn.endswith("_" + slot):
        return sn[:-(len(slot) + 1)]
    return sn


def is_osd_process_online(ssh_service):
    # 避免info中存在dsware_osd影响后面的判断，此处筛选dsware_os
    cmd = ssh_cmd_util.get_osd_process_online_cmd(ssh_service.is_mini_system)
    info = ssh_service.exec_cmd(Cmd(cmd))
    return "dsware_osd" in info


def get_metadata_disk_eid_and_status(connection, slot):
    """
    执行/opt/MegaRAID/storcli/storcli64 /c0/fall del
    :param connection:
    :param slot: 存储池id
    :return:
    """
    cmd = ssh_cmd_util.get_metadata_disk_eid_and_status_cmd(connection.is_mini_system)
    ssh_info = connection.exec_cmd(Cmd(cmd))
    if "PD LIST :" in ssh_info:
        pd_list = ssh_info.split("PD LIST :")[1]
        slot_info = ":" + str(slot)
        reg = r"\s*(\d+)" + slot_info + r"\s+\d+\s+(\w+)\s+"
        match = re.findall(reg, pd_list)
        if match:
            return True, match[0][0], match[0][1]
    return False, "", ""


def modify_metadata_disk_status_to_jbod(connection, slot):
    """
    1、清除元数据盘的UGooD F状态
    2、修改元数据盘的状态为JBOD
    3、查询元数据盘状态
    :param connection:
    :param slot: 存储池id
    :return:
    """
    flag, eid, status = get_metadata_disk_eid_and_status(connection, slot)
    if not flag:
        return False, ""
    if status == JBOD:
        return True, ""
    clean_status_cmd = ssh_cmd_util.get_metadata_clean_status_cmd(connection.is_mini_system)
    ssh_info = connection.exec_cmd(Cmd(clean_status_cmd))
    if not is_exec_success(ssh_info):
        return False, get_description_info(ssh_info, "Description =")
    modify_status_cmd = ssh_cmd_util.get_metadata_modify_status_cmd(connection.is_mini_system, eid, slot)
    ssh_info = connection.exec_cmd(Cmd(modify_status_cmd))
    if not is_exec_success(ssh_info):
        return False, get_description_info(ssh_info, "Description =")
    return get_metadata_disk_status(connection, slot) == JBOD, ""


def modify_1880_meta_disk_to_jbod(conn):
    ssh_ret = conn.exec_cmd(Cmd("hiraidadm show allctrl"))
    match = re.findall(r"Controller\s+Id\s+\|\s+(\d+)", ssh_ret)
    if not match:
        return False, resource_service.get_msg("query.hiraidadm.raid.card.id.failed")
    ctrl_id = match[0]

    if not is_1880_jbod_on(conn, ctrl_id):
        ssh_ret = conn.exec_cmd(Cmd("hiraidadm c{} set jbod=rawdrive".format(ctrl_id)))
        if not is_exec_success(ssh_ret):
            return False, get_description_info(ssh_ret, "Description =")
    return is_1880_jbod_on(conn, ctrl_id), ""


def is_1880_jbod_on(conn, ctrl_id):
    ssh_ret = conn.exec_cmd(Cmd("hiraidadm c{} show jbod".format(ctrl_id)))
    return "rawdrive" in ssh_ret


def is_exec_success(info):
    return "Status = Success" in info


def get_metadata_disk_status(connection, slot):
    flag, eid, status = get_metadata_disk_eid_and_status(connection, slot)
    if flag:
        return status
    return ""


def need_jbod(connection):
    """
    判断是否需要盘处于JBOD状态
    当前排除的卡：3152，3008
    :return:
    """
    if raid_card_util.raid_card_version_is_smart_raid_3152(connection):
        return False
    if raid_card_util.raid_card_version_is_LSI_SAS_3008(connection):
        return False
    return True


def replace_zk_disk(connection, node_ip, esn, context, replace_zk="true"):
    cmd_list = ssh_cmd_util.get_replace_zk_disk_cmd(connection.is_mini_system, connection.is_sandbox_open,
                                                    context.getDswareNode())
    cmd_list[-1] = NoNeedReturnCmd(cmd_list[-1].format(node_ip, esn, replace_zk))
    cmd_list.extend(
        [
            NoNeedReturnCmd("y"),
            NoNeedReturnCmd(context.getCluster().getLoginUser().getUserName()),
            NoLogCmd(context.getCluster().getLoginUser().getPassword())
        ]
    )
    if connection.is_sandbox_open:
        result_info = connection.exec_dsware_cmd_with_sandbox_open(cmd_list)
    else:
        result_info = connection.exec_dsware_cmd(cmd_list)
    logger_factory.create_logger(__file__).info(result_info)
    flag = "result code:0" in result_info.lower()
    return flag, get_description_info(result_info, "description:")


def ensure_zk_recover(node_ip, esn, context):
    connection = SshService(context.getNode())
    zk_para_cmd = ssh_cmd_util.get_ensure_zk_recover_cmd(connection.is_mini_system)
    res = connection.exec_cmd(Cmd(zk_para_cmd))
    if esn in res:
        return False, resource_service.get_msg("ensure.zk.recover.failed")
    return replace_zk_disk(SshService(context.getDswareNode()),
                           node_ip, esn, context, replace_zk="false")


def get_description_info(failed_info, description_key):
    lines = failed_info.splitlines()
    for line in lines:
        if description_key in line:
            return line.split(description_key)[-1]
    return ""


def get_disk_manufacturer(connection, disk_sn):
    cmd = ssh_cmd_util.get_show_disk_infos_cat_cmd(connection.is_mini_system)
    ssh_info = connection.exec_cmd(Cmd(cmd))
    vmr_start_index = 0
    vmr_end_index = 0
    # 先找的表头行
    info_lines = ssh_info.splitlines()
    for line in info_lines:
        if "Esn" in line and "VMR" in line:
            temp_lines = line.split("VMR", 1)
            vmr_start_index = temp_lines[0].rfind("|")
            vmr_end_index = temp_lines[1].find("|") + len(temp_lines[0])

    if vmr_start_index <= 0 and vmr_end_index <= 0:
        return ""
    # 根据下标获取SN和厂商
    for line in info_lines:
        if len(line) < vmr_end_index + 1:
            continue
        if disk_sn in line:
            return line[vmr_start_index + 1: vmr_end_index].split(",")[
                0].strip()
    return ""


def clean_smart_cache(service, storage_pool_id):
    try:
        result = service.exec_post(ClusterRestUri.SMART_CACHE_INFO,
                                   {"type": 8,
                                    "param_type": 0,
                                    "pool_id": storage_pool_id,
                                    "inner_type": 0,
                                    "meta_scan": 0})
        clean_result_info = result.get("result", {})
        if clean_result_info == 0 or clean_result_info.get("code") == 0:
            return True, ""
        return False, clean_result_info.get("description",
                                            "") + clean_result_info.get(
            "suggestion", "")
    except ConnectionException as e:
        return False, RestService.get_description(e.response)


def smart_cache_switch_is_on(service, storage_pool_id):
    try:
        smart_cache_info = service.exec_get(ClusterRestUri.SMART_CACHE_INFO,
                                            params={"type": 8,
                                                    "param_type": 0,
                                                    "pool_id": storage_pool_id})
        return smart_cache_info.get("data").get(
            "switch") == SMART_CACHE_SWITCH_ON
    except Exception:
        return False


def query_node_main_disk_slots(context, pool_ids, node_ip):
    """
    查询某节点的主存盘在指定存储池里的槽位号列表（用于判断是否踢盘）
    :param context: 上下文
    :param pool_ids: 存储池ID列表
    :param node_ip: 管理IP
    :return: 槽位号列表
    """
    disk_slots = list()
    for pool_id in pool_ids:
        disk_slots.extend(_query_main_disk_slots(context, pool_id, node_ip))
    logger_factory.create_logger(__file__).info(
        'node_ip: {}, pool_ids: {}, query main disk slot result: {}'.format(
            node_ip, pool_ids, disk_slots))
    return disk_slots


def _query_main_disk_slots(context, pool_id, node_ip):
    """
    查询某节点的主存盘在某存储池里的槽位号列表（用于判断是否踢盘）
    :param context: 上下文
    :param pool_id: 存储池ID
    :param node_ip: 管理IP
    :return: 槽位号列表
    """
    connection = SshService(context.getDswareNode())
    cmd_list = ssh_cmd_util.get_query_main_disk_slots_cmd(connection.is_mini_system,
                                               connection.is_sandbox_open,
                                               context.getDswareNode(),
                                               pool_id)
    cmd_list.extend(
        [
            NoNeedReturnCmd(context.getCluster().getLoginUser().getUserName()),
            NoLogCmd(context.getCluster().getLoginUser().getPassword())
        ]
    )
    if connection.is_sandbox_open:
        res = connection.exec_dsware_cmd_with_sandbox_open(cmd_list)
    else:
        res = connection.exec_dsware_cmd(cmd_list)
    # 解析对应节点对应池的数据
    res_list = res.splitlines()
    begin_line_pattern = "nodeMgrIp:{}\\s*poolId:{}.*".format(
        node_ip.replace('.', '\\.'), pool_id)
    disk_slot_pattern = 'diskSlot:(\\d+)'
    result_disk_slot_list = list()
    has_find_title = False
    for line in res_list:
        if not has_find_title and re.match(begin_line_pattern, line):
            has_find_title = True
            continue
        if has_find_title and not line.endswith("diskInfo:"):
            slot_res = re.search(disk_slot_pattern, line)
            if slot_res:
                result_disk_slot_list.append(slot_res.group(1))
            else:
                break
    return result_disk_slot_list
