#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2024. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/08/06
@file: recover_cluster_ccdb_service.py
@function: 恢复集群的ccdb
@modify:
"""
import time
import traceback

from py.common.entity.item_status import ItemStatus
from py.common.service import logger_factory, resource_service
from py.common.service.connection.ssh_connection_service import SshService, \
    NoNeedReturnCmd, NoLogCmd, Cmd
from py.fusion_storage.common.context import disk_context_util, common_context_util
from py.fusion_storage.common.service.cluster import query_disk_info_service


SERVICE_MEDIA_CONFIG = \
    """<?xml version=\\"1.0\\" encoding=\\"utf-8\\"?>
<cluster mediaTypeKey=\\"mediaType\\" mediaTypeValue=\\"physicalVolume\\" section=\\"GLOBAL\\">
    <server manageIp=\\"{}\\">
            <media deviceName=\\"{}\\" mediaKey=\\"ccdb_volume\\" slotId=\\"{}\\" />
    </server>
</cluster>"""


class AbstractRecoverClusterCCDBService(object):
    def __init__(self, context):
        self._context = context
        self._mgr_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._slot_id = disk_context_util.get_replace_disk_slot(context)
        self._logger = logger_factory.create_logger(__file__)

    def _get_query_cluster_id_cmd(self):
        """
        返回查询集群ID的命令，命令结果需为数字
        :return: 查询集群ID的命令
        """
        raise NotImplementedError

    def _get_query_ccdb_server_process_id_cmd(self):
        """
        返回查询ccdb_server的命令，命令结果需为数字
        :return:
        """
        raise NotImplementedError

    def _get_replace_disk_cmd(self):
        """
        获取换盘命令
        :return:
        """
        raise NotImplementedError

    def _get_remote_conf_dir_path(self):
        """
        获取远端模板配置文件的目录
        :return:
        """
        raise NotImplementedError

    def _is_need_y(self):
        """
        执行命令后，是否需要输入Y确认
        :return:
        """
        raise False

    def recover_ccdb(self):
        """
        恢复ccbd
        1. 修改模板盘符
        2. 查询集群ID与节点ccdb_server的Process ID
        3. 执行换盘命令
        :return:
        """
        try:
            ModifyMediaConfig(self._context).modify_media_config(
                self._get_remote_conf_dir_path(), self._get_modify_media_config_cmd())
            cluster_id, process_id = self._get_cluster_id_and_process_id()
            self._exec_replace_cmd(cluster_id, process_id)
        except Exception:
            self._logger.error("recover control cluster ccdb failed. {}"
                               .format(traceback.format_exc()))
            return ItemStatus.FAILED, resource_service.get_msg(
                "recover.ccdb.failed")
        return ItemStatus.SUCCESS, ""

    def _get_modify_media_config_cmd(self):
        """
        获取minisystem模式下模板配置文件修改命令
        :return:
        """
        raise NotImplementedError

    def _exec_replace_cmd(self, cluster_id, process_id):
        service = SshService(self._context.getDswareNode())
        cmd_list = self._get_replace_disk_cmd()
        cmd_list[-1] = NoNeedReturnCmd(cmd_list[-1].format(cluster_id, process_id, self._mgr_ip))
        cmd_list.extend(
            [
                NoNeedReturnCmd(self._context.getCluster().getLoginUser().getUserName()),
                NoLogCmd(self._context.getCluster().getLoginUser().getPassword())
            ]
        )
        if self._is_need_y():
            cmd_list.insert(-2, NoNeedReturnCmd("y"))
        if service.is_sandbox_open:
            res = service.exec_dsware_cmd_with_sandbox_open(cmd_list)
        else:
            res = service.exec_dsware_cmd(cmd_list)
        if "error" in res.lower() or "failed" in res.lower():
            raise Exception('Failed to execute replace disk cmd. result: {}'
                            .format(res.replace(self._context.getCluster().getLoginUser().getPassword(), "******")))

    def _get_cluster_id_and_process_id(self):
        return self._get_cluster_id(), self._get_ccdb_process_id()

    def _get_cluster_id(self):
        return self._get_execute_cmd_digit_result(
            self._get_query_cluster_id_cmd())

    def _get_ccdb_process_id(self):
        return self._get_execute_cmd_digit_result(
            self._get_query_ccdb_server_process_id_cmd())

    def _get_execute_cmd_digit_result(self, cmd):
        ssh_service = SshService(self._context.getNode())
        res = ssh_service.exec_cmd(Cmd(cmd))
        res_list = res.splitlines()
        if ssh_service.is_mini_system:
            if len(res_list) > 1 and len(res_list[1].split("=")) > 1:
                cluster_id = res_list[1].split("=")[1]
                if cluster_id.isdigit():
                    return cluster_id
        else:
            if len(res_list) > 1 and res_list[1].isdigit():
                return res_list[1]
        raise Exception('execute cmd:{} failed, execute result is:{}'
                        .format(cmd, res))


class ModifyMediaConfig(object):
    def __init__(self, context):
        self._context = context
        self._mgr_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._device_name = \
            query_disk_info_service.query_disk_drive_letter(
                context, disk_context_util.get_replace_disk_sn(context))
        self._slot_id = disk_context_util.get_replace_disk_slot(context)
        self._ssh_service = SshService(context.getDswareNode())

    def modify_media_config(self, remote_file_path, modify_config_cmd):
        """
        修改模板盘符
        1. 判断是沙箱开启状态，通过fsm命令修改远端文件，否则执行2
        2. 根据上下文信息生成模板内容
        3. 将模板内容写至远端文件
        :param modify_config_cmd: mini模式下修改文件命令
        :param remote_file_path: 远端文件路径
        """
        if self._ssh_service.is_mini_system:
            self._modify_config_in_mini(modify_config_cmd)
        else:
            self._write_config_file(remote_file_path, self._generate_config_content())

    def _modify_config_in_mini(self, modify_config_cmd):
        if self._device_name.startswith("/dev/"):
            self._device_name = self._device_name.split("/dev/")[1]
        modify_config_cmd = modify_config_cmd.format(self._mgr_ip, self._device_name, self._trans_slot_id())
        cmd_list = [
            NoNeedReturnCmd("fsm_cli"),
            NoNeedReturnCmd(modify_config_cmd),
            NoNeedReturnCmd("exit"),
            NoNeedReturnCmd("y")
        ]
        for cmd in cmd_list:
            self._ssh_service.exec_cmd(cmd)
            time.sleep(1)

    def _generate_config_content(self):
        return SERVICE_MEDIA_CONFIG.format(self._mgr_ip, self._device_name,
                                           self._trans_slot_id())

    def _write_config_file(self, remote_file_path, media_config_content):
        cmd = 'echo -e "{}" > {}'.format(media_config_content, remote_file_path)
        self._ssh_service.exec_cmd(Cmd(cmd))

    def _trans_slot_id(self):
        return "" if disk_context_util.is_nvme_disk(self._context) \
            else str(self._slot_id)
