#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

"""
@version: SmartKit 22.0.3
@time: 2023/07/02
@file: change_node_fail_over_status.py
@function: 倒换/回切节点
@modify:
"""
import time

from py.common.adapter import java_adapter
from py.common.adapter.java_adapter import get_node_pool_service, get_rest_connection_manager_class, \
    get_rest_util_class, get_switch_over_enum
from py.common.entity.item_status import ItemStatus
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.rest_connection_service import RestService
from py.common.service.connection.ssh_connection_service import SshService, Cmd
from py.fusion_storage.common.constant import ClusterRestUri
from py.fusion_storage.common.context import disk_context_util
from py.common.service import resource_service, logger_factory

BLOCK_LIST = ["8.0.1", "8.1.0", "8.1.1", "8.1.2", "8.1.3"]
BASE_POOL_URL = "/dsware/service/resource/queryPoolBasicInfo"
FAIL_OVER_URL = "/dsware/service/fail_over/failover"
FAIL_BACK_URL = "/dsware/service/fail_over/failback"
QUERY_TASK_URL = "/dsware/service/task/queryTaskInfo"


@auto_brush_progress(180)
def switch_over_service(context):
    fail_over_service = NodeFailOverService(context)
    if "True" == fail_over_service.check_node_in_fail_over_status():
        return ItemStatus.SUCCESS, ""
    if not fail_over_service.check_is_vnode_ec_pool():
        return ItemStatus.NOT_INVOLVED, ""
    try:
        flag, msg = fail_over_service.switch_over_node()
        if not flag:
            return ItemStatus.FAILED, msg
        return ItemStatus.SUCCESS, ""
    except java_adapter.get_java_exception_class() as e:
        return ItemStatus.FAILED, resource_service.get_msg('node.switchover.startup.failed')


@auto_brush_progress(180)
def fail_back_service(context):
    fail_over_service = NodeFailOverService(context)
    if not fail_over_service.check_is_vnode_ec_pool():
        return ItemStatus.NOT_INVOLVED, ""
    try:
        if "True" == fail_over_service.check_node_in_fail_over_status():
            flag, msg = fail_over_service.fail_back_node()
            if not flag:
                return ItemStatus.FAILED, msg
        return ItemStatus.SUCCESS, ""
    except java_adapter.get_java_exception_class() as e:
        return ItemStatus.FAILED, resource_service.get_msg('node.failback.startup.failed')


class NodeFailOverService:
    def __init__(self, context):
        self._disk_infos = context.getEnv().getOrDefault("NODE_DISKS", [])
        self._cluster = context.getCluster()
        self._manage_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._base_url_head = get_rest_util_class().getDstorageUrlHead(self._cluster)
        self._rest_conn = get_rest_connection_manager_class().getRestConnection(self._cluster)
        self._product_version = context.getFruDevice().getBusinessDevice().getDevNode().getProductVersion()
        self._ssh = SshService(context.getNode())
        self._logger = logger_factory.create_logger(__file__)

    @staticmethod
    def check_ha_status_normal(result):
        standby_status, active_status = False, False
        for line in result.splitlines()[1:4]:
            if "active" in line and "normal" in line:
                active_status = True
            if "standby" in line and "normal" in line:
                standby_status = True
        return standby_status and active_status

    def check_is_vnode_ec_pool(self):
        url = self._base_url_head + BASE_POOL_URL
        for disk_info in self._disk_infos:
            for pool_id in disk_info.getBelongPoolIds():
                if get_node_pool_service().isVnodeEcPool(self._rest_conn, url, str(pool_id)):
                    return True
        return False

    def switch_over_node(self):
        url = self._base_url_head + FAIL_OVER_URL
        if not self.switch_fsm_active_node_if_need():
            return False, resource_service.get_msg('node.switch.active.failed')
        if not self.is_login_success():
            return False, resource_service.get_msg('node.switch.active.failed')
        if not get_node_pool_service().switchoverNode(self._rest_conn, url, self._manage_ip):
            return False, resource_service.get_msg('node.switchover.startup.failed')
        if not self.check_task_is_completed(get_switch_over_enum().SWITCHOVER):
            return False, resource_service.get_msg('node.switchover.status.failed')
        return True, ""

    def fail_back_node(self):
        url = self._base_url_head + FAIL_BACK_URL
        if not get_node_pool_service().switchbackNode(self._rest_conn, url, self._manage_ip):
            return False, resource_service.get_msg('node.failback.startup.failed')
        if "False" == self.check_node_in_fail_over_status():
            return True, ""
        if not self.check_task_is_completed(get_switch_over_enum().FAILBACK):
            return False, resource_service.get_msg('node.failback.status.failed')
        return True, ""

    def check_task_is_completed(self, task_name):
        url = self._base_url_head + QUERY_TASK_URL
        if not get_node_pool_service().taskIsCompleted(self._rest_conn, task_name, url):
            return False
        return True

    def check_node_in_fail_over_status(self):
        # 8.0.1~8.1.3均不支持该接口
        for block_version in BLOCK_LIST:
            if block_version in self._product_version:
                return "NOT_INVOLVED"
        service = RestService(self._cluster, extra_success_code=4)
        fail_over_dict = service.exec_get(ClusterRestUri.QUERY_FAIL_OVER_STATUS)
        # result是0，说明不存在故障倒换状态节点
        if fail_over_dict.get("result", -1) != 0:
            for info in fail_over_dict.get("detail", []):
                if info.get("ip") == self._manage_ip:
                    return "True"
        return "False"

    def switch_fsm_active_node_if_need(self):
        check_active_cmd = ssh_cmd_util.get_check_active_cmd(self._ssh.is_mini_system)
        if "active" not in self._ssh.exec_cmd(Cmd(check_active_cmd)):
            return True
        switch_active_cmd = ssh_cmd_util.get_switch_active_cmd(self._ssh.is_mini_system)
        if "successfully" not in self._ssh.exec_cmd(Cmd(switch_active_cmd)):
            return False
        check_hs_status_cmd = ssh_cmd_util.get_check_hs_status_cmd(self._ssh.is_mini_system)
        wait_sec = 0
        while wait_sec < 600:
            try:
                if self.check_ha_status_normal(self._ssh.exec_cmd(Cmd(check_hs_status_cmd))):
                    return True
            finally:
                wait_sec += 10
                time.sleep(10)
        return "active" not in self._ssh.exec_cmd(Cmd(check_active_cmd))

    def is_login_success(self):
        wait_sec = 0
        while wait_sec < 1200:
            try:
                self._rest_conn.reLogin()
                if self._rest_conn.isLogin():
                    return True
            except java_adapter.get_java_exception_class():
                self._logger.info("Current rest not ready,please wait")
            finally:
                wait_sec += 30
                time.sleep(30)
        return False
