#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2024. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/08/10
@file: check_disk_specification.py
@function: 更换后检查硬盘规格是否满足要求
@modify:
"""

from py.common.adapter import java_adapter
from py.common.service.connection.rest_connection_service import RestService
from py.common.entity.item_status import ItemStatus
from py.common.service import resource_service
from py.fusion_storage.common.context import disk_context_util
from py.fusion_storage.common.record import record_raid_card_type
from py.fusion_storage.common.service.os_util import disk_info_util
from py.common.service.connection.ssh_connection_service import SshService


def execute(context):
    # 系统盘硬RAID场景不涉及
    if disk_context_util.is_sys_module(context) \
            and record_raid_card_type.get_raid_card_type(context):
        return ItemStatus.NOT_INVOLVED, ""
    return DiskSpecificationCheckItem(context).check()


class DiskSpecificationCheckItem(object):
    def __init__(self, context):
        self.context = context
        self.service = RestService(context.getCluster())
        self.old_disk = context.getData()
        self.new_disk = context.getDataAfterReplace()
        self.err_msg = list()

    def check(self):
        # 介质类型需要一致
        if self.old_disk.getMediaType() is not None \
                and self.old_disk.getMediaType() != java_adapter.get_media_type_class().UNKNOWN \
                and self.old_disk.getMediaType().toString() \
                != self.new_disk.getMediaType().toString():
            self.err_msg.append(resource_service
                                .get_msg("replace.disk.type.not.same")
                                .format(self.old_disk.getMediaType(),
                                        self.new_disk.getMediaType()))
        # 新盘容量需要 >= 旧盘容量
        if self.new_disk.getCapacity() < self.old_disk.getCapacity():
            old_disk_capacity = self._trans_disk_capacity(self.old_disk.getMediaType(), self.old_disk.getCapacity())
            new_disk_capacity = self._trans_disk_capacity(self.new_disk.getMediaType(), self.new_disk.getCapacity())
            self.err_msg.append(resource_service.get_msg("replace.disk.capacity.less")
                                .format(old_disk_capacity, new_disk_capacity))
        # # 扇区要求一致
        self.check_sector_size()
        if self.err_msg:
            return ItemStatus.NOT_PASS, "\n".join(self.err_msg)
        return ItemStatus.PASS, ""

    def _trans_disk_capacity(self, media_type, capacity):
        rate = 1000.0 if media_type == "SSD_CARD" else 1024.0
        value = 0
        if disk_context_util.is_meta_module(self.context) \
                or disk_context_util.is_pilot_module(self.context):
            value = capacity * rate * rate
        elif disk_context_util.is_sys_module(self.context):
            value = capacity
        elif disk_context_util.is_main_module(self.context) \
                or disk_context_util.is_cache_module(self.context):
            value = capacity * rate * rate * rate
        units = ['B', 'KB', 'MB', 'GB', 'TB']
        for unit in units:
            if value >= rate:
                value = value / rate
            else:
                return "{}{}".format("%.2f" % value, unit)

    def check_sector_size(self):
        old_sector_size = self.old_disk.getSectorSize()
        old_logic_sector_size = self.old_disk.getLogicSectorSize()
        # 旧盘可能一开始就已经下电，查不到，此情况无法检查，通过
        if old_sector_size and old_sector_size != self.new_disk.getSectorSize():
            self.err_msg.append(resource_service
                                .get_msg("sectors.not.equal")
                                .format(self.old_disk.getSectorSize(),
                                        self.new_disk.getSectorSize()))
        if old_logic_sector_size and old_logic_sector_size != self.new_disk.getLogicSectorSize():
            self.err_msg.append(resource_service
                                .get_msg("logic.sectors.not.equal")
                                .format(self.old_disk.getLogicSectorSize(),
                                        self.new_disk.getLogicSectorSize()))

    def check_manufacturer(self):
        old_disk_manufacturer = self.old_disk.getManufacturer()
        # 旧盘可能一开始就已经下电，查不到，此情况无法检查，通过
        if not old_disk_manufacturer:
            return
        ssh_service = SshService(self.context.getNode())
        new_disk_manufacturer = disk_info_util.get_disk_manufacturer(
            ssh_service,
            self.new_disk.getEsn())
        if new_disk_manufacturer != old_disk_manufacturer:
            self.err_msg.append(resource_service
                                .get_msg("manufacturer.not.equal")
                                .format(old_disk_manufacturer,
                                        new_disk_manufacturer))
