#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/09/06
@file: clear_old_disk_alarm.py
@function: 清除旧盘告警
@modify:
"""
import time
import traceback

from py.common.entity.exception import ConnectionException
from py.common.entity.item_status import ItemStatus
from py.common.entity.user_select import AbstractUserSelectTableQueryData, \
    UserSelectTableQueryResult
from py.common.service import resource_service, logger_factory
from py.common.service.connection.rest_connection_service import RestService
from py.fusion_storage.common import constant
from py.fusion_storage.common.context import disk_context_util

MATCH_ALARM_LIST_KEY = "match_alarm_list_key"
ALARM_STATUS_UN_RECOVER = 1
ALARM_TYPE_FAULT = 4
# 白名单告警、对应槽位号参数坐标；坐标以1开始，0表示无槽位号参数
ALARM_ID_2_SLOT_DICT = {
    '0xF000A0132': 4,
    '0xF0017001D': 0,
    '0xF000A0138': 2,
    '0xF000A0136': 2,
    '0xF000A012D': 2,
    '0xF000A0134': 2,
    '0xF000A0135': 2,
    '0xF00D80152': 2
}


def execute(context):
    """
    执行入口，查询旧盘相关告警并自动清除
    :param context: 上下文
    :return: 结果
    """
    try:
        return AlarmClean(context).clean_old_disk_alarms()
    except Exception as ex:
        logger_factory.create_logger(__file__).error("clean alarm failed. {}".format(traceback.format_exc()))
        return ItemStatus.NOT_INVOLVED, ""


class AlarmClean(object):
    def __init__(self, context):
        self._context = context
        self._node_ip = disk_context_util.get_belong_mgmt_ip(context)
        self._disk_slot = disk_context_util.get_replace_disk_slot(context)
        self._disk_esn = disk_context_util.get_disk_sn(context)
        self._rest_service = RestService(context.getCluster())
        self._logger = logger_factory.create_logger(__file__)

    def clean_old_disk_alarms(self):
        match_alarms = self._query_match_alarms()
        self._logger.info("match alarms: {}".format(match_alarms))
        if not match_alarms:
            return ItemStatus.NOT_INVOLVED, ""
        return self._clean_alarms(match_alarms)

    def _clean_alarms(self, alarms):
        sequences = [str(alarm.get('sequence')) for alarm in alarms]
        self._logger.info("clean sequence: {}".format(sequences))
        try:
            self._rest_service.exec_post(constant.ClusterRestUri.BATCH_CLEAN_ALARMS,
                                         params={"sn": sequences})
            return ItemStatus.SUCCESS, ""
        except ConnectionException as ex:
            alarm_id_list = self._get_alarm_ids_by_sequences(ex.response.getContent().get('fail', []), alarms)
            return ItemStatus.FAILED, resource_service \
                .get_msg('clear.old.disk.alarm.failed').format(alarm_id_list)

    def _query_match_alarms(self):
        res_dict = self._rest_service.exec_get(
            constant.ClusterRestUri.BATCH_QUERY_ALARMS,
            params={
                "filter": "alarmStatus::" + str(ALARM_STATUS_UN_RECOVER)
            }
        )
        return self._filter_involve_alarm_list(res_dict.get("data", []))

    @staticmethod
    def _get_alarm_ids_by_sequences(sequence_list, alarms):
        alarm_ids = []
        for alarm in alarms:
            if alarm.get('sequence') in sequence_list:
                alarm_ids.append(alarm.get('id'))
        return alarm_ids

    def _filter_involve_alarm_list(self, res_alarm_list):
        match_alarm_list = list()
        for alarm in res_alarm_list:
            if alarm.get('alarm_id') not in ALARM_ID_2_SLOT_DICT.keys():
                continue
            if self._is_alarm_match_old_disk(alarm):
                match_alarm_list.append(alarm)
        return match_alarm_list

    def _is_alarm_match_old_disk(self, alarm):
        if self._disk_esn:
            return self._disk_esn in alarm.get('alarm_params')
        params = alarm.get('alarm_params').split(',')
        if self._node_ip not in params:
            return False
        slot_param_index = ALARM_ID_2_SLOT_DICT.get(alarm.get('alarm_id'), 0)
        if slot_param_index == 0 or slot_param_index > len(params):
            return False
        return str(self._disk_slot) == params[slot_param_index - 1]


class ShowAlarm(AbstractUserSelectTableQueryData):
    @staticmethod
    def get_select_key():
        return 'sequence'

    @staticmethod
    def get_severity_dict():
        return {
            3: resource_service.get_msg('alarm.severity.warning'),
            5: resource_service.get_msg('alarm.severity.major'),
            6: resource_service.get_msg('alarm.severity.critical')
        }

    @staticmethod
    def get_alarm_status_dict():
        return {
            1: resource_service.get_msg('alarm.status.unrecovered'),
            2: resource_service.get_msg('alarm.status.cleared'),
            4: resource_service.get_msg('alarm.status.recovered')
        }

    def __init__(self, res_alarm, is_selected):
        self.alarm_obj_type = res_alarm.get('alarmObjType')
        self.alarm_status = res_alarm.get('alarmStatus')
        self.id = res_alarm.get('alarm_id')
        self.name = res_alarm.get('alarm_name')
        self.params = res_alarm.get('alarm_params')
        self.type = res_alarm.get('alarm_type')
        self.clear_time = res_alarm.get('clear_time')
        self.clear_type = res_alarm.get('clear_type')
        self.clear_user = res_alarm.get('clear_user')
        self.create_time = res_alarm.get('create_time')
        self.description = res_alarm.get('description')
        self.detail = res_alarm.get('detail')
        self.position = res_alarm.get('position')
        self.sequence = res_alarm.get('sequence')
        self.severity = res_alarm.get('severity')
        self.suggestion = res_alarm.get('suggestion')
        self.is_selected = is_selected

    @staticmethod
    def get_show_col_list():
        return [
            {"key": "alarm_status",
             "name": resource_service.get_msg('alarm.property.status'),
             "show": False},
            {"key": "alarm_id",
             "name": resource_service.get_msg('alarm.property.id'),
             "show": True},
            {"key": "alarm_name",
             "name": resource_service.get_msg('alarm.property.name'),
             "show": True},
            {"key": "severity",
             "name": resource_service.get_msg('alarm.property.severity'),
             "show": True},
            {"key": "sequence",
             "name": resource_service.get_msg('alarm.property.sequence'),
             "show": False},
            {"key": "description",
             "name": resource_service.get_msg('alarm.property.description'),
             "show": False},
            {"key": "suggestion",
             "name": resource_service.get_msg('alarm.property.suggestion'),
             "show": False},
            {"key": "create_time",
             "name": resource_service.get_msg('alarm.property.create.time'),
             "show": True},
        ]

    def get_show_data_dict(self):
        return {
            "alarm_id": self.id,
            "severity": self._trans_alarm_severity(self.severity),
            "create_time": self._trans_alarm_time(self.create_time),
            "sequence": self.sequence,
            "alarm_name": self.name,
            "description": self.description,
            "suggestion": self.suggestion,
            "alarm_status": self._trans_alarm_status(self.alarm_status),
            "is_selected": self.is_selected
        }

    def _trans_alarm_severity(self, ori_severity):
        return self.get_severity_dict().get(
            ori_severity, resource_service.get_msg('alarm.property.unknown'))

    def _trans_alarm_status(self, ori_status):
        return self.get_alarm_status_dict().get(
            ori_status, resource_service.get_msg('alarm.property.unknown'))

    @staticmethod
    def _trans_alarm_time(ori_time_stamp):
        return time.strftime("%Y-%m-%d %H:%M:%S", time.gmtime(ori_time_stamp))
