#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2022/01/06
@file: planned_partitions_for_new_disk.py
@function: 软RAID为新盘划分分区
@modify:
"""
import re

from py.common.entity.exception import BusinessException
from py.common.entity.item_status import ItemStatus
from py.common.service import resource_service
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.ssh_connection_service import SshService, Cmd, CustomEndingJudgeCmd
from py.common.service import logger_factory
from py.fusion_storage.common.service.disk_select_factory.system_disk_selector_factory import DiskInfo
from py.fusion_storage.common.record import record_raid_card_type
from py.fusion_storage.common.service.disk_select_factory.system_disk_selector_factory \
    import SystemDiskSelectorFactory, handle_res_func
from py.fusion_storage.common.service.raid_option_util.raid_sync_complete_check_util import get_raid_sync_query_cmds

BLANK_PATTERN = re.compile(r'\s+')
ERROR_CMD_RES = ['No such file or directory']
NOT_REGISTRY_ERROR = 'unrecognised disk label'
CMD_RES_STR = 'cmd:{},res:{}'
START_KEY = 'start'
END_KEY = 'end'


def execute(context):
    if record_raid_card_type.get_raid_card_type(context):
        return ItemStatus.NOT_INVOLVED, ''
    return PlanPartition(context).do_partition()


def cmd_has_error_msg(disk_ata):
    for item in ERROR_CMD_RES:
        if item in disk_ata:
            return True
    return False


def check_raid_add(res):
    if 'added' not in res:
        raise BusinessException(err_code='add.raid.failed')


class PlanPartition(object):

    def __init__(self, context):
        self._ssh = SshService(context.getNode())
        self._context = context
        self._logger = logger_factory.create_logger(__file__)
        self._complete = True
        self._partitioned_disk = None
        self._system_disk_list = set()
        self._new_partitioned_disk = list()
        self._not_partition_disk = list()
        self._exist_device_detail = list()
        self._new_disk_detail = list()
        self._disk_selector = SystemDiskSelectorFactory(context, self._ssh, self._logger)

    def do_partition(self):
        try:
            # 查看系统盘符
            self._exec_for_system_disk()
            # 解析盘符详细信息
            self._find_disk_detail()
            # 获取已分区盘符的具体数据
            self._get_partition_detail()
            # 根据老盘分区数据，分区新盘
            self._do_partition_for_new()
            # 查看新盘分区好的数据
            self._check_new_disk_detail()
            # 新盘添加raid
            self._add_new_to_raid()
            # 将需要检查的raid命令拼接好存入上下文
            self._set_need_check_raid_sync_complete()
            return ItemStatus.SUCCESS, ''
        except BusinessException as e:
            return ItemStatus.FAILED, resource_service.get_msg(e.err_code).format(e.err_msg)

    def _exec_for_system_disk(self):
        self._system_disk_list = self._disk_selector.exec_for_system_disk()

    def _find_disk_detail(self):
        partitioned_disk, not_partition_disk, disk_info = self._disk_selector.find_disk_detail()
        # 根据文档，系统盘只会有两个，更换条件下一个未分区，一个已分区
        if not partitioned_disk or len(partitioned_disk) > 1:
            # 没有已分区系统盘或多于一个已分区的系统盘，错误场景
            self._logger.error('the count of partitioned disk error:{}'.format(disk_info))
            raise BusinessException(
                err_code='partitioned.disk.count.less.error' if not partitioned_disk else 'partitioned.disk.'
                                                                                          'count.more.error')
        self._partitioned_disk = partitioned_disk[0]
        self._not_partition_disk = not_partition_disk

    def _find_not_partition_disk(self, partition_disk):
        for item in self._system_disk_list:
            if partition_disk != item:
                self._not_partition_disk.append(item)

    def _get_partition_detail(self):
        cmd = 'fdisk -l /dev/{}'.format(self._partitioned_disk.name)
        res = self._ssh.exec_cmd_handle_res(Cmd(cmd), handle_res_func=handle_res_func)
        self._parse_disk_detail_start_and_end(res, self._exist_device_detail)
        if not self._exist_device_detail:
            self._logger.error('find partitioned disk error')
            raise BusinessException(err_code='partitioned.disk.detail.error')

    def _parse_disk_detail_start_and_end(self, res, detail_list):
        device_lines = res.splitlines()
        find_detail = False
        for item in device_lines:
            # 找到表头行
            if 'Start' in item and 'End' in item:
                # 表格头，忽略
                find_detail = True
                continue
            if find_detail:
                self._parse_disk_detail_data(item, detail_list)

    def _parse_disk_detail_data(self, disk_str_line, detail_list):
        if not disk_str_line:
            raise BusinessException('partitioned.disk.detail.error')
        split_item = BLANK_PATTERN.split(disk_str_line.strip())
        device = dict()
        device[START_KEY] = split_item[1]
        device[END_KEY] = split_item[2]
        detail_list.append(device)

    def _do_partition_for_new(self):
        for item in self._not_partition_disk:
            res = self._create_label_for_new(item)
            if not res:
                raise BusinessException(err_code='partitioned.new.disk.error')
            self._partition_with_old_detail(item)

    def _create_label_for_new(self, item):
        end_strs = ['(parted) ', 'label type? ', 'Yes/No? ', ':/>', ':/', ']', '#', '$', '~>', ':~ #']
        part_cmd = 'parted /dev/{}'.format(item)
        res = self._ssh.exec_cmd(CustomEndingJudgeCmd(part_cmd, end_strs))
        if not res or not res.strip().endswith('(parted)'):
            error_msg = CMD_RES_STR.format(part_cmd, res)
            self._logger.error('parted result error.{}'.format(error_msg))
            return False
        mklabel_cmd = 'mklabel'
        res = self._ssh.exec_cmd(CustomEndingJudgeCmd(mklabel_cmd, end_strs))
        if not res or not res.strip().endswith('label type?'):
            error_msg = CMD_RES_STR.format(mklabel_cmd, res)
            self._logger.error('mklabel result error.{}'.format(error_msg))
            return False
        gpt_cmd = 'gpt'
        res = self._ssh.exec_cmd(CustomEndingJudgeCmd(gpt_cmd, end_strs))
        if res.strip().endswith('Yes/No?'):
            self._logger.info('cmd:{} executed with result：{}'.format(gpt_cmd, res))
            res = self._ssh.exec_cmd(CustomEndingJudgeCmd('Yes', end_strs))
        if not res or not res.strip().endswith('(parted)'):
            error_msg = CMD_RES_STR.format(gpt_cmd, res)
            self._logger.error('gpt result error.{}'.format(error_msg))
            return False
        q_cmd = 'q'
        self._ssh.exec_cmd(Cmd(q_cmd))
        partprobe_cmd = 'partprobe'
        self._ssh.exec_cmd(Cmd(partprobe_cmd))
        return True

    def _partition_with_old_detail(self, device):
        cmd = ssh_cmd_util.get_mk_part_logical_cmd(self._ssh.is_mini_system)
        for item in self._exist_device_detail:
            format_cmd = cmd.format(device, item['start'], item['end'])
            cmd_obj = Cmd(format_cmd)
            res = self._ssh.exec_cmd_handle_res(cmd_obj, handle_res_func=handle_res_func)
            if NOT_REGISTRY_ERROR in res:
                self._registry_label(device)
                # 新盘出现未注册错误，注册后重试一次
                self._ssh.exec_cmd_handle_res(cmd_obj, handle_res_func=handle_res_func)

    def _registry_label(self, device):
        cmd = 'parted -s /dev/{} mklabel gpt'.format(device)
        self._ssh.exec_cmd_handle_res(Cmd(cmd), handle_res_func=handle_res_func)

    def _add_new_to_raid(self):
        cmd = 'mdadm --manage /dev/{} --add /dev/{}'
        # 表示替换非数字开头的所有字符
        pattern = re.compile('^[^0-9]+')
        for not_par_disk in self._not_partition_disk:
            self._add_to_every_raid(cmd, not_par_disk, pattern)

    def _add_to_every_raid(self, cmd, not_par_disk, pattern):
        new_root_disk = DiskInfo(not_par_disk)
        for item in self._partitioned_disk.children:
            md_list = item.children
            if not md_list:
                continue
            # 将从开头到第一个数字部分替换为新的盘符名字
            sd_name = pattern.sub(not_par_disk, item.name)
            sd_disk = DiskInfo(sd_name)
            sd_disk.parent = new_root_disk
            new_root_disk.children.append(sd_disk)
            for md in md_list:
                md_disk = DiskInfo(md.name)
                md_disk.parent = sd_disk
                sd_disk.children.append(md_disk)
                format_cmd = cmd.format(md.name, sd_name)
                res = self._ssh.exec_cmd_handle_res(Cmd(format_cmd), handle_res_func=handle_res_func)
                check_raid_add(res)
        self._new_partitioned_disk.append(new_root_disk)

    def _check_new_disk_detail(self):
        cmd = 'fdisk -l /dev/{}'
        for item in self._not_partition_disk:
            res = self._ssh.exec_cmd_handle_res(Cmd(cmd.format(item)), handle_res_func=handle_res_func)
            self._logger.info("new disk detail:\n{}".format(res))
            self._check_new_equals_old(res)

    def _check_new_equals_old(self, res):
        self._parse_disk_detail_start_and_end(res, self._new_disk_detail)
        self._logger.info("new disk detail:{}".format(res))
        if len(self._new_disk_detail) != len(self._exist_device_detail):
            raise BusinessException(err_code='partitioned.new.disk.detail.error')
        for index in range(len(self._new_disk_detail)):
            new_detail = self._new_disk_detail[index]
            old_detail = self._exist_device_detail[index]
            if new_detail[START_KEY] != old_detail[START_KEY] or new_detail[END_KEY] != old_detail[END_KEY]:
                raise BusinessException(err_code='partitioned.new.disk.detail.error')

    def _set_need_check_raid_sync_complete(self):
        self._context.getEnv().put(self._context.getNode().getIp() + '_check_raid_cmds',
                                   get_raid_sync_query_cmds(self._new_partitioned_disk))
