#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2023. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/13
@file: processing_node.py
@function:
@modify:
"""
import re

from py.common.service.connection.rest_connection_service import RestService
from py.fusion_storage.common.context import disk_context_util, common_context_util
from py.common.entity.item_status import ItemStatus
from py.common.service.connection.ssh_connection_service import SshService, \
    NoNeedReturnCmd, KeepAliveCmd, NoLogCmd
from py.common.service import resource_service
from py.fusion_storage.common.record import record_cache_disk_replace_type
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.fusion_storage.common.service.disk_out_storage_pool_service import \
    get_disk_pool_id_2_storage_pool_id
from py.common.service.connection import ssh_cmd_util

DESCRIPTION_REG = re.compile(r"description:(.+)\n")

TYPE_PRE = "PRE"
TYPE_POST = "POST"
# 刷盘时间和当前缓存盘数据大小，业务压力有关，大集群环境存在刷盘40分钟的场景，这里设置超时时间为60分钟
TIME_OUT = 3600


def _process(context, cmd, cmd_type):
    """
    执行处理节点
    :param context: 业务上下文
    :param cmd: 命令
    :return: True/False 执行成功失败
    """
    # 离线不涉及,虚拟盘不区分
    if not record_cache_disk_replace_type.is_online_replace_type(context) \
            and not disk_context_util.is_virtual_disk(context):
        return ItemStatus.NOT_INVOLVED, ""
    service = SshService(context.getDswareNode())
    pool_ids = disk_context_util.get_disk_belong_pool_ids(context)
    mgmt_ip = disk_context_util.get_belong_mgmt_ip(context)
    disk_pool_id_2_storage_pool_id = get_disk_pool_id_2_storage_pool_id(
        context, pool_ids, RestService(context.getCluster()))
    disk_sn = disk_context_util.get_disk_sn(context)
    for pool_id in pool_ids:
        if disk_context_util.is_virtual_disk(context):
            cmd = cmd + _get_sriov_param(disk_pool_id_2_storage_pool_id, pool_id, mgmt_ip, cmd_type, disk_sn)
        else:
            cmd = cmd + _get_param(disk_pool_id_2_storage_pool_id, pool_id, mgmt_ip, cmd_type)
        if not service.is_mini_system and not service.is_sandbox_open:
            origin_info = service.exec_dsware_cmd(
                [NoNeedReturnCmd(cmd),
                 NoNeedReturnCmd("y"),
                 NoNeedReturnCmd(context.getCluster().getLoginUser().getUserName()),
                 KeepAliveCmd(context.getCluster().getLoginUser().getPassword(), TIME_OUT, is_need_log=False)]
            )
        elif service.is_mini_system:
            cmd = cmd.split("/")[-1]
            origin_info = service.exec_dsware_cmd_with_sandbox_open(
                [NoNeedReturnCmd(cmd),
                 NoNeedReturnCmd("y"),
                 NoNeedReturnCmd(context.getCluster().getLoginUser().getUserName()),
                 KeepAliveCmd(context.getCluster().getLoginUser().getPassword(), TIME_OUT, is_need_log=False)]
            )
        else:
            cmd = cmd.split("/")[-1]
            password = ssh_cmd_util.get_root_user_pass(context.getDswareNode())
            origin_info = service.exec_dsware_cmd_with_sandbox_open(
                [NoNeedReturnCmd("minisystem"),
                 NoLogCmd(password),
                 NoNeedReturnCmd(cmd),
                 NoNeedReturnCmd("y"),
                 NoNeedReturnCmd(context.getCluster().getLoginUser().getUserName()),
                 KeepAliveCmd(context.getCluster().getLoginUser().getPassword(), TIME_OUT, is_need_log=False)]
            )
        if "completed!" not in origin_info:
            match = DESCRIPTION_REG.search(origin_info)
            if match:
                return ItemStatus.FAILED, resource_service.get_msg(
                    "failed.process.node") + " :" + match.groups()[0]
            return ItemStatus.FAILED, resource_service.get_msg(
                    "failed.process.node")
    return ItemStatus.SUCCESS, ""


def _get_param(disk_pool_id_2_storage_pool_id, pool_id, mgmt_ip, cmd_type):
    if cmd_type == TYPE_PRE:
        return " -p {} -a {}".format(pool_id, mgmt_ip)
    if disk_pool_id_2_storage_pool_id.get(pool_id) is not None:
        return " -p {} -d {}  -a {}".format(
            disk_pool_id_2_storage_pool_id.get(pool_id), pool_id,
            mgmt_ip)
    else:
        return " -p {} -a {}".format(pool_id, mgmt_ip)


def _get_sriov_param(disk_pool_id_2_storage_pool_id, pool_id, mgmt_ip, cmd_type, disk_sn):
    if cmd_type == TYPE_PRE:
        return " -p {} -a {} -o {}".format(pool_id, mgmt_ip, disk_sn)
    else:
        return " -p {} -d {} -a {}".format(
            disk_pool_id_2_storage_pool_id.get(pool_id), pool_id,
            mgmt_ip)


@auto_brush_progress(300)
def pre_processing(context):
    if disk_context_util.is_virtual_disk(context):
        return _process(
            context,
            "sh /opt/dsware/tools/ops_tool/replace_sriov/preprocessing_replace_sriov_cache.sh",
            TYPE_PRE)
    return _process(
        context,
        "sh /opt/dsware/tools/ops_tool/replace_ssd/preprocessing_replace_ssd_cache.sh",
        TYPE_PRE)


@auto_brush_progress(500)
def post_processing(context):
    if disk_context_util.is_virtual_disk(context):
        return _process(
            context,
            "sh /opt/dsware/tools/ops_tool/replace_sriov/postprocessing_replace_sriov_cache.sh",
            TYPE_POST)
    return _process(
        context,
        "sh /opt/dsware/tools/ops_tool/replace_ssd/postprocessing_replace_ssd_cache.sh",
        TYPE_POST)
