#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2022. All rights reserved.

"""
@version: SmartKit V200R007C00
@time: 2021/07/22
@file: restore_zooKeeper.py
@function: 更换后检查，节点后处理，控制集群和复制集群共用元数据盘，恢复zk
@modify:
"""
import traceback
import time

from py.common.entity.item_status import ItemStatus
from py.common.service import logger_factory
from py.common.service import resource_service
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.common.service.connection.ssh_connection_service import SshService, Cmd
from py.fusion_storage.common.context import common_context_util
from py.fusion_storage.common.context import disk_context_util
from py.fusion_storage.common.service.cluster import cluster_info_util
from py.fusion_storage.common.service.os_util import disk_info_util


@auto_brush_progress(180)
def execute(context):
    return RestoreZooKeeper(context).check()


class RestoreZooKeeper(object):
    def __init__(self, context):
        self.replace_disk_esn = disk_context_util.get_replace_disk_sn(context)
        self.context = context
        self.logger = logger_factory.create_logger(__file__)
        self.ip = disk_context_util.get_belong_mgmt_ip(context)
        self.dsware_service = SshService(self.context.getDswareNode())
        self.ssh_service = SshService(self.context.getNode())

    def check(self):
        """
        执行命令恢复zk
        """
        try:
            flag, description_info = disk_info_util.replace_zk_disk(
                self.dsware_service, self.ip, self.replace_disk_esn, self.context)
            if not flag:
                self.check_format(description_info)
                return ItemStatus.FAILED, self.get_error_msg(description_info)
            if not cluster_info_util.is_ocean_stor_pacific(self.context):
                return ItemStatus.SUCCESS, ""
            if self._polling_check_zk_process_started():
                return ItemStatus.SUCCESS, ""
            flag, description_info = disk_info_util.ensure_zk_recover(
                self.ip, self.replace_disk_esn, self.context)
            if not flag:
                return ItemStatus.FAILED, self.get_error_msg(description_info)
        except Exception:
            self.logger.error("Restore zk has error: {}".format(
                traceback.format_exc()))
            return ItemStatus.FAILED, resource_service.get_msg(
                "restore.zk.error")
        return ItemStatus.SUCCESS, ""

    def check_format(self, desc):
        if "sector format" in desc:
            common_context_util.set_cur_task_suggestion(
                self.context, resource_service.get_msg("sector.format.error"))

    @staticmethod
    def get_error_msg(description_info):
        if description_info:
            return resource_service.get_msg("restore.zk.failed").format(
                description_info)
        return resource_service.get_msg("restore.zk.error")

    def _polling_check_zk_process_started(self):
        zk_process_cmd = "ps -ef | grep zk_inner | grep QuorumPeer | grep -v grep"
        start_time = time.time()
        while time.time() - start_time < 60 * 3:
            res = self.ssh_service.exec_cmd(Cmd(zk_process_cmd))
            if "zk_inner" in res.replace(zk_process_cmd, ""):
                return True
            time.sleep(20)
        return False
