#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

"""
@version: SmartKit 24.0.0
@time: 2024/11/20
@file: set_fan_speed.py
@function: 更换硬盘备件时设置风扇为全速模式
"""

import time
from py.common.service import logger_factory, resource_service
from py.common.service.auto_brush_item_progress import auto_brush_progress
from py.common.service.connection import ssh_cmd_util
from py.common.service.connection.ssh_connection_service import SshService, KeepAliveCmd
from py.common.entity.item_status import ItemStatus
from py.fusion_storage.common.context import common_context_util


# 设置转速重试超时时间暂定3分钟，与BMC复位时间一致
TIMEOUT_S = 180
# 单次SSH命令超时
SSH_TIMEOUT_S = 30
RETRY_INTERVAL_S = 10


class SetFanSpeed(object):
    def __init__(self, context):
        self._context = context
        self._logger = logger_factory.create_logger(__file__)
        self._ssh_service = SshService(context.getNode())

    # 判断命令是否存在
    def _is_command_exist(self):
        cmd_str = ssh_cmd_util.get_fan_ctrl_cmd(self._ssh_service.is_mini_system, "-h")
        cmd = KeepAliveCmd(cmd_str, SSH_TIMEOUT_S)
        ret = self._ssh_service.exec_cmd(cmd, retry_times=3)
        if "pangea_fan_ctrl.sh fullspeed" in ret:
            return True
        else:
            return False

    def _try_set_once(self, mode, normal_str):
        cmd_str = ssh_cmd_util.get_fan_ctrl_cmd(self._ssh_service.is_mini_system, mode)
        cmd = KeepAliveCmd(cmd_str, SSH_TIMEOUT_S)
        self._ssh_service.exec_cmd(cmd, retry_times=1, abnormal_judge_func=lambda x: normal_str not in x)

    def _try_set(self, mode, normal_str):
        start_time = time.time()
        while time.time() < start_time + TIMEOUT_S:
            try:
                self._try_set_once(mode, normal_str)
                return True
            except Exception as e:
                self._logger.error("Set fan speed failed: {}".format(e))
                time.sleep(RETRY_INTERVAL_S)
        return False

    def set_full_speed(self):
        if not self._is_command_exist():
            return ItemStatus.NOT_INVOLVED, ""
        if self._try_set("fullspeed", "Set fan to full speed succeeded."):
            return ItemStatus.SUCCESS, ""
        else:
            return ItemStatus.FAILED, resource_service.get_msg("set.fan.speed.failed")

    def set_auto_speed(self):
        if not self._is_command_exist():
            return ItemStatus.NOT_INVOLVED, ""
        if self._try_set("autospeed", "Set fan to automatic speed succeeded."):
            return ItemStatus.SUCCESS, ""
        else:
            return ItemStatus.FAILED, resource_service.get_msg("set.fan.speed.failed")


@auto_brush_progress(TIMEOUT_S)
def set_full_speed(context):
    # 仅太平洋硬件有此策略
    if not common_context_util.is_pacific_server(context):
        return ItemStatus.NOT_INVOLVED, ""
    return SetFanSpeed(context).set_full_speed()


@auto_brush_progress(TIMEOUT_S)
def set_auto_speed(context):
    # 仅太平洋硬件有此策略
    if not common_context_util.is_pacific_server(context):
        return ItemStatus.NOT_INVOLVED, ""
    return SetFanSpeed(context).set_auto_speed()
