# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
import time
import re
import os
import json
import traceback
import codecs

from cbb.business.operate.fru.common import SftpFactory
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
from cbb.frame.context import contextUtil
from cbb.common.query.hardware import controller

from com.huawei.ism.tool.obase.exception import ToolException
import com.huawei.json.JSONException as Json_Exception

# IP地址正则
IP_RE_PATTERN = r"\b(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25" \
                r"[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\b"


def collect(context, local_dir, current_node, ctrl_id):
    """收集控制器日志

    :param context: 上下文
    :param local_dir: 本地保存路径
    :param current_node: 当前登录的节点
    :param ctrl_id: 待收集的控制ID
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    is_dorado_high_end = contextUtil.getItem(context, "isDorado18000")
    # 文件路径预处理：将单斜杠替换为双斜杠
    if not re.search("\\\\", local_dir):
        local_dir = local_dir.replace('\\', '\\\\')
    file_name = "log_controller_{}.tar.bz2".format(ctrl_id)
    contextUtil.setItem(context, "log_file_path",
                        os.path.join(local_dir, file_name))
    # 目标节点
    node_flag, target_node = controller.get_node_by_ctrl(
        is_dorado_high_end, ctrl_id)
    logger.logInfo("target node is:{}".format(target_node))
    if not node_flag:
        return

    need_heartbeat = target_node != current_node
    # 执行日志导出
    if is_dorado_high_end or need_heartbeat:
        export_result = export_other_ctrl(context, ctrl_id, target_node)
    else:
        export_result = export_current_ctrl(context, ctrl_id)
    # 下载日志文件
    remote_path = "/tmp/log_controller_{}.tar.bz2".format(ctrl_id)
    if export_result and SftpFactory.Sftp.download(context, remote_path,
                                                   local_dir):
        contextUtil.setItem(context, "export_log_suc", True)
        logger.logInfo("Succeed to download log file.")
    else:
        logger.logInfo("Failed to download log file.")
    return


def export_other_ctrl(context, ctrl_id, target_node):
    """心跳到其他控制器收集日志

    :param context:
    :param ctrl_id:
    :param target_node:
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    devObj = contextUtil.getDevObj(context)
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        # 心跳到目标节点
        heart_cmd = "sshtoremoteExt {}".format(target_node)
        heart_ret = cliUtil.sshToRemoteContr(cli, heart_cmd,
                                             devObj.get("pawd"), lang)
        # 心跳失败
        if "You have accessed the system." not in heart_ret[1]:
            logger.logInfo("Failed to heartbeat to node:{}".format(
                target_node))
            return False
        logger.logInfo("Succeed to heartbeat to node:{}".format(target_node))
        # 取到目标节点的内部IP
        target_ip = re.findall(IP_RE_PATTERN, heart_ret[1])[0]
        # 在目标节点小系统下执行日志导出
        collect_cmd = "os_backup_info.sh -f log_controller_{}.tar.bz2".format(
            ctrl_id)
        collect_ret = cliUtil.excuteCmdInMinisystemModel(cli, collect_cmd,
                                                         lang)
        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        if collect_ret[0] is not True:
            logger.logInfo("Failed to backup info.")
            # 收集失败
            return False

        # 将目标节点上的日志拷贝到当前节点上
        src_file = "/tmp/log_controller_{}.tar.bz2".format(ctrl_id)
        dst_dir = "/tmp"
        get_file_cmd = "getremotefile {0} {1} {2}".format(target_ip, src_file,
                                                          dst_dir)
        get_ret = cliUtil.excuteCmdInMinisystemModel(cli, get_file_cmd, lang,
                                                     devObj.get("pawd"))
        logger.logInfo("get remote file result:{}".format(get_ret[0]))
        return get_ret[0]
    except Exception as ex:
        logger.logException(ex)
    finally:
        # 如果登录就是小系统，则可能把连接退掉。
        flag, ret, msg = cliUtil.enterCliModeFromSomeModel(cli, lang)
        if flag is not True:
            cli.reConnect()


def export_current_ctrl(context, ctrl_id):
    """收集当前工具连接的控制器日志

    :param context:
    :param ctrl_id:
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    # 在小系统下执行日志导出
    collect_cmd = "os_backup_info.sh -f log_controller_{}.tar.bz2".format(
        ctrl_id)
    collect_ret = cliUtil.excuteCmdInMinisystemModel(cli, collect_cmd, lang)
    # 退出到cli模式
    cliUtil.enterCliModeFromSomeModel(cli, lang)
    return collect_ret[0]


def delete_log_file(context):
    """删除日志文件

    :param context:
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        if contextUtil.getItem(context, "export_log_suc"):
            log_file_path = contextUtil.getItem(context, "log_file_path")
            if os.path.isfile(log_file_path):
                os.remove(log_file_path)
                logger.info("log file has been deleted.")
    except Exception as ex:
        logger.logException(ex)
    finally:
        return


def collect_work_load_data(data_dict, save_path, start_day, end_day):
    """
    :param data_dict:
    :param save_path:
    :param start_day: 开始时间
    :param end_day: 结束时间
    :return:
    """
    max_size = 1024 * 1024
    current_size = 0
    logger = data_dict["logger"]
    param_dict = {"start_day": start_day, "end_day": end_day}
    params_str = restUtil.CommonRest.getParamsJsonStr(param_dict, False)
    cmd = "api/v2/rwp/get_rwp_data"
    save_file_tmp_name = "{}_{}.txt"
    java_map = contextUtil.getContext(data_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    base_uri = restUtil.get_base_uri_v2(rest_conn_wrapper.getBaseUri())
    rest_connection = rest_conn_wrapper.getRest()
    loop_count = 1
    start_time = time.time()
    try:
        while True:
            # 通过接口收集workload数据，最多收集2分钟，超时即停。
            if time.time() - start_time > 120:
                logger.info("The workload data collection duration exceeds 2 minutes.")
                break
            logger.info("params_str:{}".format(params_str))
            response_info = rest_connection.execGetNotCheckResp(
                "{}{}".format(base_uri, cmd), params_str, None, None
            )
            data_string = response_info.getContent()
            # 不支持收集的场景，界面报收集成功。
            if not is_support_collect(data_string):
                return True
            current_size += len(data_string)
            records = json.loads(data_string)
            if not is_right_response(logger, records, data_dict):
                return False
            data = records.get("data", [])
            if not data:
                break
            if current_size > max_size:
                loop_count += 1
                current_size = 0
            save_to_local_file(
                save_file_tmp_name.format(save_path, loop_count), data
            )
            time.sleep(0.01)
        return True
    except (Json_Exception, ToolException, Exception):
        logger.error(traceback.format_exc())
    finally:
        rest_connection.closeSession()
    return False


def is_support_collect(ret):
    """
    是否支持收集模型日志
    :param ret: 回显
    :return: True：支持，False：不支持
    """
    return "502 bad gateway" not in str(ret).lower()


def is_right_response(logger, record, data_dict):
    """
    校验响应数据是否正确。
    :param logger:
    :param record:
    :param data_dict:
    :return:
    """
    logger.info("work load data: {}".format(record))
    err_info = restUtil.CommonRest.getErrInfo(record)
    if not err_info:
        data_dict["py_detail"] = "No error information."
        return False
    err_code = restUtil.CommonRest.getRecordValue(
        err_info, restData.ErrorInfo.CODE
    )
    # 不支持该命令, 直接返回通过
    if str(err_code) == '-1':
        return True
    # 如果命令执行失败
    if str(err_code) != '0':
        data_dict["py_detail"] = str(err_info)
        return False
    return True


def save_to_local_file(save_path, record):
    """
    每次收集的数据都追加保存到文件中
    :param save_path:
    :param record:
    :return:
    """
    with codecs.open(save_path, 'a', encoding='utf-8') as f:
        f.write("\n".join(record))
