#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@version: Toolkit V200R006C00
@time: 2020/07/27
@file: check_sed_msid_risk.py
@function:
@modify:
"""

from cbb.business.operate.checkitems import check_required_hot_patch
from cbb.frame.cli import cliUtil
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
import traceback

# 用于无解决补丁时需要拦截的场景，补丁版本比较必不通过
NO_FIX_PATCH = "SPH999"

HOT_PATCH_VER_LIST = [
    {
        "product_version": "6.0.0",  # 软件版本
        "required_patch": "SPH7",  # 要求的补丁
        "product_series": "Dorado",  # 型号系列
        "product_models": []
    },
    {
        "product_version": "6.0.1",  # 软件版本
        "required_patch": "SPH1",  # 要求的补丁
        "product_series": "Dorado",  # 型号系列
        "product_models": []
    },
    {
        "product_version": "6.0.RC1",
        "required_patch": NO_FIX_PATCH,
        "product_series": "Dorado",
        "product_models": [],
    },
]


class errCode:
    # 错误信息码
    PASS = 0
    V60RC1_TOO_OLD = 1
    V600_PATCH_NOT_MEET = 2
    V601_PATCH_NOT_MEET = 3


def execute(context, exp_disk_types, cli):
    """加密盘与产品读取MSID算法冲突,导致接盘失败
    @OMRPId bdm_check_sed_msid_risk_product_version

    :param cli:
    :param exp_disk_types:
    :param context:上下文
                需要字段：
                    cli
                    lang
                    logger
    :return:
        ver_Ret：是否通过， 回显， 错误码, 软件版本，补丁版本
    """
    logger = context.get("logger")

    try:
        # 检查扩展盘是否为SED盘
        ret = _is_Exp_SED(exp_disk_types)
        if ret:
            return ret, "", errCode.PASS, "", ""

        # 检查版本是否匹配
        return _check_Ver(context, cli)

    except Exception as ex:
        logger.error(traceback.format_exc())
        raise Exception(ex)


def _is_Exp_SED(exp_disk_types):
    '''
    @summary: 待扩硬盘是否为SED盘
    @return: 否：False，是：True
    '''
    if not exp_disk_types:
        return True
    return not bool(
        filter(lambda disk_type: "SED" in disk_type.upper(), exp_disk_types))


def _check_Ver(context, cli):
    '''
    @summary: 检查版本和补丁情况
    @return: 是否通过， 回显， 错误码, 软件版本，补丁版本
    '''

    # 检查补丁是否符合要求
    flag, dev_version, dev_patch, dev_model, required_patch = \
        check_required_hot_patch.execute(context, HOT_PATCH_VER_LIST,
                                         cli)
    # 补丁版本满足要求，通过

    if not baseUtil.isDoradoDev(dev_model):
        return True, "", errCode.PASS, dev_version, dev_patch

    if flag:
        return True, "", errCode.PASS, dev_version, dev_patch

    # 不通过，返回原因及回显
    # 获取回显
    lang = contextUtil.getLang(context)
    cmd = "show upgrade package"
    (ok, cliRet, errorMsg) = cliUtil.execCmdInCliMode(cli, cmd, True, lang)

    result_dict = {
        "flag": False,
        "err": -1,
    }

    if dev_version.startswith('6.0.RC1'):
        result_dict["err"] = errCode.V60RC1_TOO_OLD
    elif dev_version.startswith('6.0.0'):
        result_dict["err"] = errCode.V600_PATCH_NOT_MEET
    else:
        result_dict["err"] = errCode.V601_PATCH_NOT_MEET

    return result_dict.get("flag"), cliRet, result_dict.get("err"), \
        dev_version, dev_patch
