# -*- coding: UTF-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2012-. All rights reserved.
from cbb.business.operate.checkitems import check_required_hot_patch
from cbb.frame.base import baseUtil
from cbb.frame.base import funcUtils
from cbb.frame.base import product
from cbb.frame.base.config import CTRL_6U_PRODUCTS
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from cbb.frame.base.config import ARM_HIGH_END_NEW
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from cbb.common.query.hardware import controller
from utils import Products

REQUIRED_PATCH_LIST = [
    {
        "product_version": "V500R007C30SPC100",
        "required_patch": "V500R007C30SPH109",
        "product_series": "ConvergedStorage",
        "product_models": []
    }, {
        "product_version": "V300R006C50SPC100",
        "required_patch": "V300R006C50SPH109",
        "product_series": "ConvergedStorage",
        "product_models": []
    }
]

TIME_OUT = 60

TCP_CMD_MAP = {
    "TCP_MEM": "tcpconfig show TCP_MEM",
    "TCP_RMEM": "tcpconfig show TCP_RMEM",
    "TCP_WMEM": "tcpconfig show TCP_WMEM",
}


@funcUtils.fakeProgress(TIME_OUT)
def execute(context, ctrl_id):
    """备份TCP缓存设置

    :param context: python上下文
    :param ctrl_id: 要备份配置的控制器ID
    :return:
    """
    # 检查版本补丁
    flag, dev_version, dev_patch, dev_model, required_patch = \
        check_required_hot_patch.execute(context, REQUIRED_PATCH_LIST)

    if not flag:
        # 检查失败 说明是维护版本 但是不满足补丁要求
        return True

    # 检查成功 有required_patch说明是维护版本
    is_maintenance_release = True if required_patch else False
    if not is_maintenance_release:
        # 需要判断是否融合在研版本或者dorado在研版本
        if not is_later_than_required_version(dev_model, dev_version):
            return True

    flag, tcp_mem_setting = get_tcp_mem_setting(
        context, dev_model, ctrl_id, is_maintenance_release)
    if not flag:
        return False

    # 保存到上下文中
    contextUtil.setItem(context, "tcp_mem_setting", tcp_mem_setting)
    return True


def is_later_than_required_version(dev_model, dev_version):
    """判断版本是否大于等于当前的在研版本

    :param dev_model:
    :param dev_version:
    :return:
    """
    if product.isKunpeng(dev_version):
        # 融合Kunpeng版本
        if dev_version >= "V500R007C70":
            return True
    if "DORADO" not in dev_model.upper():
        # 融合purly版本
        if "V3" in dev_model.upper() and dev_version >= "V300R006C61":
            return True
        if "V5" in dev_model.upper() and dev_version >= "V500R007C61":
            return True
    if any([baseUtil.is_ocean_protect(dev_model), baseUtil.is_new_oceanstor(dev_model),
            baseUtil.is_micro_dev(dev_model), baseUtil.is_computing_dev(dev_model)]):
        return True
    if "DORADO" in dev_model.upper() and product.isDigitalVer(dev_version):
        # Dorado V6数字版本
        if dev_version != "6.0.RC1" and dev_version >= "6.0.1":
            return True
    if Products.compareVersion(dev_version, "V700R001C00") >= 0:
        return True
    return False


def get_tcp_mem_setting(context, dev_model, ctrl_id, is_maintenance_release):
    """获取TCP缓存设置

    :param context: 上下文
    :param dev_model: 设备型号
    :param ctrl_id: 要备份配置的控制器ID
    :param is_maintenance_release: 是否维护版本
    :return:
    """
    tcp_mem_setting = {}
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    is_high_end = dev_model in (CTRL_6U_PRODUCTS + DORADO_DEVS_V6_HIGH +
                                ARM_HIGH_END_NEW)
    cli = cliUtil.get_cli_for_svp(context, logger)
    lang = contextUtil.getLang(context)
    dev_obj = contextUtil.getDevObj(context)

    # 获取目标节点
    flag, target_node = controller.get_node_by_ctrl(is_high_end, ctrl_id)
    logger.logInfo("target node is:{}".format(target_node))
    if not flag:
        return False, tcp_mem_setting

    try:
        # 心跳到目标节点
        heart_cmd = "sshtoremoteExt {}".format(target_node)
        heart_ret = cliUtil.sshToRemoteContr(cli, heart_cmd,
                                             dev_obj.get("pawd"), lang)
        # 心跳失败
        if "You have accessed the system." not in heart_ret[1]:
            logger.logInfo("Failed to heartbeat to node:{}".format(
                target_node))
            return False, tcp_mem_setting
        logger.logInfo("Succeed to heartbeat to node:{}".format(target_node))
        # 在目标节点小系统下执行查询命令
        sh_name = "ifm.sh" if baseUtil.isDoradoV6Dev(dev_model) or baseUtil.is_big_clusters_ai_dev(dev_model) \
            else "eam.sh"

        query_flag = True
        for tcp_cmd in TCP_CMD_MAP:
            # 执行命令
            cmd = "{} {}".format(sh_name, TCP_CMD_MAP.get(tcp_cmd))
            flag, cli_ret, err_msg = cliUtil.excuteCmdInMinisystemModel(
                cli, cmd, lang)
            if flag is not True:
                query_flag = False
                break
            if is_maintenance_release and "usage:" in cli_ret.lower():
                # 维护版本中的紧急补丁不支持查询命令 检查通过
                return True, tcp_mem_setting
            # 解析回显
            flag, ret_str = parse_cli_ret(cli_ret, logger)
            if not flag:
                query_flag = False
                break
            # 解析成功
            tcp_mem_setting[tcp_cmd] = ret_str

        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        # 退出minisystem
        cliUtil.exitMinisystemToCliMode(cli)
        if not query_flag:
            return False, tcp_mem_setting

        return True, tcp_mem_setting
    except Exception as ex:
        logger.logException(ex)
        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        # 退出minisystem
        cliUtil.exitMinisystemToCliMode(cli)
        return False, tcp_mem_setting


def parse_cli_ret(cli_ret, logger):
    """解析回显：在研版本返回3行 维护版本返回1行

    :param cli_ret: 回显字符串
    :param logger: 日志句柄
    :return: flag： 解析出3个数字时返回True 否则返回False
    ret_str: 3个数字组成的字符串
    """
    logger.logInfo("cli_ret is:{}".format(cli_ret))
    cli_ret_lines = cli_ret.splitlines()
    logger.logInfo("cli_ret_lines is:{}".format(cli_ret_lines))
    ret_str = ""
    temp_list = []
    if " : " in cli_ret:
        # 在研版本返回3行
        for line in cli_ret_lines:
            fields = line.split(" : ")
            if len(fields) < 2:
                continue
            temp_list.append(fields[1])
        ret_str = " ".join(temp_list)
    else:
        # 维护版本返回1行
        temp_list = cli_ret_lines[1].split()
    logger.logInfo("temp_list is:{}".format(temp_list))

    if len(temp_list) != 3:
        return False, ret_str

    for temp in temp_list:
        if not temp.isdigit():
            return False, ret_str

    ret_str = " ".join(temp_list)
    logger.logInfo("ret_str is:{}".format(ret_str))
    return True, ret_str
