﻿# -*- coding: UTF-8 -*-
import math
import re

from java.lang import Exception as JException

from cbb.frame.base import baseUtil
from cbb.frame.base.config import DORADO_DEVS_V6
from cbb.frame.rest import restUtil
from cbb.frame.cli import cliUtil
from cbb.frame.rest import restData
from cbb.frame.tlv import tlvData
from cbb.frame.base.config import CTRL_6U_PRODUCTS
from cbb.frame.base.config import DORADO_DEVS_V6_HIGH
from cbb.frame.base.config import ARM_HIGH_END_NEW
from cbb.frame.context import contextUtil
from cbb.common.query.hardware import controller
from cbb.common.query.hardware.controller import get_node_by_ctrl
import config
import resource
import time
import BaseFactory
import FuncFactory
import threading
from cbb.business.operate.fru.common.TlvEnumFactory import TlvEnum
from cbb.business.operate.fru.common.TlvFactory import Tlv
from cbb.business.operate.fru.common.constData import Keys
from utils import Products

# 错误回显信息匹配正则
ERROR_CODE_PATTERN = re.compile('Error Code : (.*)')

# 支持查询指定回显'Error Code : (.*)'的版本
SUPPORT_CHECK_ERROR_CODE_VERSION = ["V300R006C20", "V300R006C50SPC100", "V500R007C10", "V500R007C30SPC100",
                                    "V500R007C60SPC300", "V500R007C61", "V500R007C71SPC100", "6.0.1", "6.1.0", "6.1.2"]

# 黑名单：系统繁忙这个错误码需要特殊处理
ERROR_CODE_IN_BLACKLIST = ["0x4040324E"]


def isSingleModel(rest):
    '''
    @summary: 判读系统是否为单控模式
    @param rest: rest连接
    @return:
        True: 系统未单控模式
        False: 系统不为单控模式
    '''
    record = restUtil.Tlv2Rest.getSysRecord(rest)
    configModel = restUtil.Tlv2Rest.getRecordValue(record, restData.Sys.System.CONFIG_MODEL)
    if configModel == restData.Enum.CofigModelEnum.SINGLE_CTRL:
        return True

    return False


def getCtrlEnclosureModel(rest, logger):
    '''
    @summary: 获取指定框的框model
    '''
    enclosureRecords = restUtil.Tlv2Rest.getEnclosureRecords(rest)
    for record in enclosureRecords:
        logicType = int(str(restUtil.Tlv2Rest.getRecordValue(record, tlvData.ENCLOSURE['logicType'])))
        if logicType == tlvData.ENCLOSURE_TYPE_E['CTRL']:
            model = int(str(restUtil.Tlv2Rest.getRecordValue(record, tlvData.ENCLOSURE['model'])))
            return model
    return None


def getMsg(lang, msg, errMsgArgs="", suggestionArgs="",
           msgDict=resource.MESSAGES_DICT):
    return baseUtil.getMsg(lang, msg, errMsgArgs, suggestionArgs, msgDict)


def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    return baseUtil.getRes(lang, res, args, resDict)


def setUIProgress(context, totalTime, remainTime):
    '''
    @summary: 根据剩余时间设置界面显示进度
    @param context: 上下文对象
    @param totalTime: 总时间
    @param context: 剩余时间
    '''
    remainTime = int(remainTime)
    if remainTime < 0:
        remainTime = 0

    context["curProgressPer"] = int(100.0 * (totalTime - remainTime) / totalTime)
    context["remainTime"] = remainTime
    return


def showUI(context, curProgressPer, remainTime):
    if curProgressPer < 0 or curProgressPer > 100:
        curProgressPer = 100

    if remainTime < 0:
        remainTime = 0

    context["curProgressPer"] = curProgressPer
    context["remainTime"] = remainTime
    return


def setWaitedProcess(context, waitTime):
    '''
    @summary: 设置等待进度
    @param context: 上下文对象
    @param waitTime: 等待时间
    '''
    usedTime = 0
    startTime = time.time()
    while (time.time() - startTime < waitTime):
        usedTime = time.time() - startTime
        remainTime = int(waitTime - usedTime)
        setUIProgress(context, waitTime, remainTime)
        restUtil.CommonRest.safeSleep(config.CHECK_DEFAULT_PERIOD)
    return


def checkDiskEncConsistent(encList, redEncList):
    """
    @summary: 检查当前环路的硬盘框是否在冗余环路存在
    @param encList: 当前环路的硬盘框列表
    @param redEncList: 冗余环路硬盘框列表
    @return: False 冗余链路正常
             True: 冗余链路连接的框比当前环路少  
    """
    notLinkedEncList = []
    # 比较更换链路的硬盘框是否都在冗余链路上。如果不在记录下来
    for enc in set(encList):
        if enc not in set(redEncList):
            notLinkedEncList.append(enc)

    if len(notLinkedEncList) > 0:
        return (True, notLinkedEncList)
    else:
        return (False, notLinkedEncList)


def getManuallyRecoverPolicy(data):
    """判断系统中存在的远程复制是否存在恢复策略为手动的项目，无则返回通过
    此处将所有恢复策略为手动的项都获取出来，更换后是否启同步，由到时的状态决定

    :param data:
    :return:
    """
    recoverPolicyRepList = []
    for record in data:
        recoverPolicy = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.ReplicationPair.RECOVERY_POLICY)
        if recoverPolicy == restData.Enum.RecoveryPolicyEnum.MANUALLY:
            recoverPolicyRepList.append(record)

    return recoverPolicyRepList


def getRepRemoteDevices(repSrvRecs):
    """获取受更换影响远程复制ID。

    :param repSrvRecs: 需要手动恢复的增值业务信息集合
    :return:
    """

    deviceRepIdsDict = {}
    for repRec in repSrvRecs:
        repId = restUtil.CommonRest.getRecordValue(repRec, restData.RestCfg.ReplicationPair.ID)
        remoteDeviceID = restUtil.CommonRest.getRecordValue(repRec, restData.RestCfg.ReplicationPair.REMOTE_DEVICE_ID)

        repList = []  # 当前远端设备上的增值业务列表
        if remoteDeviceID in deviceRepIdsDict.keys():
            repList = deviceRepIdsDict[remoteDeviceID]
        repList.append(repId)
        deviceRepIdsDict[remoteDeviceID] = repList

    return deviceRepIdsDict


def getRemoteDevEffectPortsList(ports, remoteDeviceIds):
    '''
    
    @param ports: 增值业务的端口
    @param deviceIds: 远端设备ID
    '''
    portsOnDeviceDict = {}
    for record in ports:
        devId = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.PubRemoteDevPortAttr.DEVICE_ID)
        if devId in remoteDeviceIds:
            localPortId = restUtil.CommonRest.getRecordValue(record,
                                                             restData.RestCfg.PubRemoteDevPortAttr.LOCAL_PORT_ID)
            portsOnDeviceDict[localPortId] = devId
    return portsOnDeviceDict


def getPortIdDict(ports):
    '''
    @param ports: 端口列表
    @return: 返回ID
    '''

    portIdDict = {}
    if ports is not None:
        for port in ports:
            portIdDict[restUtil.CommonRest.getRecordValue(port, restData.RestCfg.PublicAttributes.ID)] = port
    return portIdDict


def getEffectPortIdDict(allEffectPortsDeviceDict, frontPorts):
    '''
    @param allEffectPortsDeviceDict: 全部受影响的端口和设备对应的字典
    @param frontPorts: 需要过滤的前端端口
    @return: 返回过滤后的受影响的前端端口Id
    '''
    effectPortIdDict = {}
    allEffectPortsIds = allEffectPortsDeviceDict.keys()
    frontPortIdDict = getPortIdDict(frontPorts)
    for effectPortId in allEffectPortsIds:
        if effectPortId in frontPortIdDict.keys():
            effectPortIdDict[effectPortId] = frontPortIdDict[effectPortId]
    return effectPortIdDict


def getEffectReplicationIds(remoteDevicesRepIdDict, effectFrontPortsList, allEffectPortsDeviceDict):
    '''
    @summary: 通过受影响的前端端口ID -> 找到远端设备ID -> 再找到远程复制项目ID
    @param remoteDevicesRepIdDict: 远端设备-远程复制项目ID
    @param effectFrontPortsList: 受影响的前端端口
    @param allEffectPortsDeviceDict: 前端端口-远端设备ID
    @return: 返回受影响的远程复制ID列表
    '''
    effectRepIdList = []
    for port in effectFrontPortsList:
        devId = allEffectPortsDeviceDict.get(port)
        effectRepIdList.extend(remoteDevicesRepIdDict.get(devId))

    return list(set(effectRepIdList))


def getUnLinkedEffectFrontPortLocationList(effectFrontPortDict, frontPortsOnCtrl):
    '''
    @summary: 获取未连接的受影响的前端端口
    @param effectFrontPortId: 前端端口ID
    @param frontPortsOnCtrl: 当前连接的前端端口ID
    '''
    unLinkedPortLocationList = []
    if effectFrontPortDict is None or len(effectFrontPortDict) == 0:
        return unLinkedPortLocationList

    currentFrontPortsIdDict = getPortIdDict(frontPortsOnCtrl)
    currentFrontPortsIdList = currentFrontPortsIdDict.keys()
    beforeFrontPortsIdList = effectFrontPortDict.keys()
    for beforePortId in beforeFrontPortsIdList:
        if beforePortId not in currentFrontPortsIdList:
            unLinkedPortLocationList.append(restUtil.CommonRest.getRecordValue(effectFrontPortDict[beforePortId],
                                                                               restData.RestCfg.PublicAttributes.LOCATION))
    return unLinkedPortLocationList


def getEnclosureList(rest):
    '''
    @summary:获取框字典列表
    @param rest: rest连接
    '''
    encList = []
    records = restUtil.Tlv2Rest.getEnclosureRecords(rest)
    for record in records:
        encDict = {}
        encId = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.ID)
        name = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.NAME)
        model = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.MODEL)
        location = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.LOCATION)
        encDict = {
            'id': encId,
            'name': name,
            'model': model,
            'location': location,
        }
        encList.append(encDict)
    return encList


def initFruInfo(context, rest):
    '''
    @summary: 初始化fru工具需要使用的信息
    @param context: 上下文
    @param rest: rest连接
    '''
    ENCLOSURE_LIST = "ENCLOSURE_LIST"
    SYS_NET_MODEL = "SYS_NET_MODEL"

    logger = baseUtil.getLogger(context.get("logger"), __file__)

    enclosureList = getEnclosureList(rest)
    logger.logInfo("get enclosure list:%s" % str(enclosureList))
    contextUtil.setItem(context, ENCLOSURE_LIST, enclosureList)
    return


################################
#         标准Rest             #
################################
def getFrontPortValueFilterList(rest, ctrlId, port_obj_list=config.FRONT_END_PORT_OBJ):
    """
    @summary: 获取冗余链路上的所有前端端口过滤值（目前只支持盘控一体的设备）
    """
    allPorts = []
    valueFilterList = []
    for port_obj in port_obj_list:
        ports = getPortsByType(rest, port_obj)
        if ports:
            allPorts.extend(ports)

    portRecords = getLinkedPortsOnCtrl(rest, allPorts, ctrlId)

    for port in portRecords:
        location = restUtil.CommonRest.getRecordValue(port, restData.RestCfg.PublicAttributes.LOCATION)
        valueList = location.split(".")
        # CTE0.A.P0
        if len(valueList) == 3:
            intfId = valueList[1]
        # CTE0.A.IOM0.P0
        if len(valueList) == 4:
            intfId = "%s.%s" % (valueList[1], valueList[2])
        enclosure = valueList[0].replace("CTE", "")
        if [enclosure, intfId] not in valueFilterList:
            valueFilterList.append([enclosure, intfId])
    return valueFilterList


def getPortsByType(rest, objType):
    """
    @summary: 获取端口信息
    """
    paramDict = restUtil.CommonRest.getUriParamDict(objType)
    data = restUtil.CommonRest.execCmd(rest, paramDict, None, restData.RestCfg.RestMethod.GET)
    return restUtil.CommonRest.getData(data)


def getLinkedPortsOnCtrl(rest, portRecords, ctrlId):
    '''
    @summary: 获取link up的端口信息
    '''
    linkedPorts = []
    portsRecords = getPortsOnCtrl(rest, portRecords, ctrlId)
    for record in portsRecords:
        runningStatus = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.RUNNING_STATUS)
        if runningStatus == restData.RestCfg.RunningStatusEnum.LINK_UP:
            linkedPorts.append(record)

    return linkedPorts


def getPortsOnCtrl(rest, portRecords, ctrlId):
    """
    @summary: 从portRecords中过滤出ctrlId控制器下的端口信息
              1.端口父ID是接口卡，2.端口父ID是控制器
    @param rest: rest连接
            portRecords: 需要过滤的端口
            ctrlId: 控制器ID。
    @return: ctrlId控制器下的端口
    """
    # 获取所有的SAS端口
    portList = []
    if portRecords is None:
        return portList

    for record in portRecords:
        parantType = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.PARENT_TYPE)
        parantId = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.PublicAttributes.PARENT_ID)

        if parantType == restData.RestCfg.ParentTypeEnum.CONTROLLER and ctrlId == parantId:
            portList.append(record)
            continue

        if parantType == restData.RestCfg.ParentTypeEnum.INTF_MODULE:
            interfaceRecord = restUtil.CommonRest.getInterfaceById(rest, parantId)
            parantType = restUtil.CommonRest.getRecordValue(interfaceRecord, restData.RestCfg.IntfModule.PARENT_TYPE)
            parantId = restUtil.CommonRest.getRecordValue(interfaceRecord, restData.RestCfg.PublicAttributes.PARENT_ID)
            if parantType == restData.RestCfg.ParentTypeEnum.CONTROLLER and ctrlId == parantId:
                portList.append(record)
                continue

    return portList


def checkHostMultPathAlarm(rest, alarmRuleItems, valueFilterList=[],
                           precheck=False):
    """
    @summary: 检查是否存在主机多路径告警
    """
    startTime = time.time()
    while (time.time() - startTime) < config.HOST_MULT_PATH_ALARM_CHECK_TIME_LIMIT:
        isExisted, resAlarmIdList = restUtil.CommonRest.checkAlarm(rest, alarmRuleItems, valueFilterList)
        # 判断系统告警中是否存在多路径告警
        if isExisted:
            if precheck:
                break
            else:
                restUtil.CommonRest.safeSleep(
                    config.HOST_MULT_PATH_ALARM_CHECK_TIME_INTEVAL)
                continue

        # 处理返回结果
        if not isExisted:
            return False, resAlarmIdList

    return True, resAlarmIdList


def getUnSyncReplicationList(rest, effectRepIdList):
    '''
    @summary: 根据记录的需要手动恢复的远程复制业务ID，检查是否拉起。
    @param rest:
    @param effectRepIdList: 需要手动恢复的远程复制业务ID列表
    @return: 返回未同步的远程复制信息
    '''
    failSyncRepIds = []
    for repId in effectRepIdList:
        # 查询增值项运行状态
        try:
            repRec = restUtil.CommonRest.getRelicationPairById(rest, repId)
            repRunStat = restUtil.CommonRest.getRecordValue(repRec, restData.RestCfg.ReplicationPair.RUNNING_STATUS)

            # 若该项运行状态：正常或分裂或同步中，跳过处理；待恢复(链路恢复后，链路断开状态将变为该状态)或故障或其他，启同步
            if restData.RestCfg.RunningStatusEnum.NORMAL == repRunStat or restData.RestCfg.RunningStatusEnum.SPLIT == repRunStat or restData.RestCfg.RunningStatusEnum.SYNCHRONIZING == repRunStat:
                continue
            elif restData.RestCfg.RunningStatusEnum.TO_BE_RECOVERD == repRunStat:
                uriParamDict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.SYNC_REPLICATION_PAIR)
                paramDict = {restData.RestCfg.ReplicationPair.ID["key"]: repId}
                restUtil.CommonRest.execCmd(rest, uriParamDict, paramDict, restData.RestCfg.RestMethod.PUT)
            else:
                failSyncRepIds.append(repId)
        except Exception:
            failSyncRepIds.append(repId)
    return failSyncRepIds


def safeSleep(seconds, logger=None):
    '''
    @summary: 安全睡眠时间
    @param param: seconds为睡眠时间，单位：秒；数据类型：整数或小数
    '''
    try:
        if type(seconds) not in [int, long, float] or seconds <= 0:
            return

        startTime = time.clock()
        while True:
            if (time.clock() - startTime) >= seconds:
                return

            # 睡眠一下，避免长时间占用cpu，该时间设置过长会影响睡眠时间精度
            try:
                time.sleep(0.1)
            except Exception as ex:
                if logger:
                    logger.error(ex)
    except Exception as ex:
        if logger:
            logger.error(ex)

    return


def setExpansionProgress(context, showRemain=False):
    '''
    @summary: 根据剩余时间设置扩容进度
    @param context: 上下文对象
    '''
    currentRemainTime = context["curRemainTime"] if context["curRemainTime"] > 0 else 0
    if showRemain == True:
        context["remainTime"] = currentRemainTime
    totalReaminTime = contextUtil.getItem(context, "totalReaminTime")
    if currentRemainTime is None or totalReaminTime is None:
        return

    context["curProgressPer"] = int(100.0 * (totalReaminTime - currentRemainTime) / totalReaminTime)
    return


def isDorado18000(productModel):
    """判断是否是Dorado18000 V3产品型号.

    :param productModel:产品型号
    :return:
    """
    if "DORADO18000" in productModel.upper():
        return True
    return False


def isDorado5000NVMe(productModel, encModel):
    """判断是否是Dorado5000 NVMe V3产品型号.

    :param productModel:产品型号
    :return:
    """
    if "DORADO5000" in productModel.upper() and isNVMeDev(encModel):
        return True
    return False


def isNVMeDev(encModel):
    if encModel == restData.Enum.EnclosureModelEnum.CTRL_NVME_2U_25:
        return True
    return False


def isDoradoNAS(productModel):
    """
    判断是否为Doardo NAS
    :param productModel:
    :return:
    """
    if "DORADO NAS" in productModel.upper():
        return True
    return False


# 刷新进度公共方法
def threadUpProcess(context, totalTime, intervalTime=config.INTERVAL_TIME):
    try:
        inProcess(context)
        t = threading.Thread(target=updateProcess, args=(context, totalTime, intervalTime))
        t.start()
        context["thread"] = t
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to Update process and remainTime: %s" % ex)


# 8s优化提取添加的完成进度的公共方法
def finishProcess(context):
    context["checkState"] = config.PROCESS_UPGRADE_FINISHED
    # 主线程等待刷进度的线程将剩余时间置于0
    threadJoin(context)
    return


def threadJoin(context):
    try:
        if context.get("thread") != None:
            context["thread"].join()
    except:
        if "logger" in context:
            context.get("logger").info("threadJoin error")


# 扩容8s优化提取添加的更新进度的公共方法
def updateProcess(context, totalTime, interval):
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    # 剩余时间总数
    totalReaminTime = totalTime
    showUI(context, 0, totalReaminTime)

    # 进度平滑处理（解决停留在100%不动的问题）
    # 0~50 interval/50~75 2*interval/75~90 4*interval
    # /90~100 8*interval /99停留直到100
    tmpInterval = interval

    # isSet[0] range(50,75)  isSet[1] range(75,90)  isSet[2] range(90,100)
    isSet = [False, False, False]

    while context["checkState"] == config.PROCESS_STATE_CHECKING:
        # 更新进度条
        totalReaminTime -= tmpInterval
        if totalReaminTime <= 0:
            # 最后的剩余时间保持为上一次显示的剩余时间
            totalReaminTime += tmpInterval
            showUI(context, 99, totalReaminTime)
            safeSleep(interval, logger)
            continue

        currentPro = int(100 * (totalTime - totalReaminTime) / totalTime)
        if tmpInterval != 1:
            if not isSet[0] and currentPro in range(50, 75):
                tmpInterval = 1 if tmpInterval / 2 == 0 else tmpInterval / 2
                isSet[0] = True
            elif not isSet[1] and currentPro in range(75, 90):
                tmpInterval = 1 if tmpInterval / 2 == 0 else tmpInterval / 2
                isSet[1] = True
            elif not isSet[2] and currentPro in range(90, 100):
                tmpInterval = 1 if tmpInterval / 2 == 0 else tmpInterval / 2
                isSet[2] = True

        showUI(context, currentPro, totalReaminTime)

        safeSleep(tmpInterval, logger)

    if context["checkState"] == config.PROCESS_UPGRADE_FINISHED:
        showUI(context, 100, 0)
    return


# 扩容8s优化提取添加的正在刷进度的公共方法
def inProcess(context):
    context["checkState"] = config.PROCESS_STATE_CHECKING
    return


class ExpBordObjForLowAndMidEnd:
    """
    级联板对象封装(针对中低端)
    DAE000.A : DAE+引擎号+框号（xx）+ 控制器 eg:DAE000.A
    """

    ctrlLocFormatter = "CTE%s.%s"

    def __init__(self, context, location):
        self.location = location
        self.context = context
        self.logger = baseUtil.getLogger(self.context.get("logger"), __file__)
        self.engId = self._getEngId()
        self.ctrlName = self._getCtrlName()
        self.ctrId = self.engId + self.ctrlName
        self.ctrlLoc = self.ctrlLocFormatter % (self.engId, self.ctrlName)

    def _getEngId(self):
        filter_exp_location = self.location
        exp_location = re.compile("DAE\w+\.\w+").findall(self.location)
        if exp_location:
            filter_exp_location = exp_location[0]
        expBords = FuncFactory.getFruListInfo(self.context, restData.Enum.ObjEnum.EXPBOARD)
        condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.LOCATION,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    filter_exp_location)
        expBord = restUtil.Tlv2Rest.filter(expBords, restUtil.Tlv2Rest.getConditionList(condition0))
        self.logger.logInfo("filter exp location:{}, expBord:{}".format(filter_exp_location, expBord))
        if not expBord:
            return ""
        encId = restUtil.Tlv2Rest.getRecordValue(expBord, restData.Hardware.Expboard.PARENT_ID)
        enclosure = FuncFactory.getFruInfo(self.context, restData.Enum.ObjEnum.ENCLOSURE, encId)
        eng = restUtil.Tlv2Rest.getRecordValue(enclosure, restData.Hardware.Enclosure.VISIBLE_AREA_ID)
        self.logger.logInfo("filter exp location eng:{}".format(eng))
        return str(eng)

    def _getCtrlName(self):
        return "" if not self.location else self.location[7]

    def getCtrlId(self):
        return self.ctrId

    def getEngId(self):
        return self.engId

    def getCtrlLoc(self):
        return self.ctrlLoc


def is_fru_eva_scene(context):
    """是否为备件评估场景化

    :param context:
    :return:
    """
    cur_scene = contextUtil.getSceneArgStrValue(context, config.SCENE_KEY)
    return cur_scene == config.SCENE_FRU_EVALUATION


def set_enc_height_from_tlv_info(context, fru_info):
    """
    从fru_info中获取控制框高度，并保存到上下文中
    适用V3Rx几个业务包通过tlv获取fru_info的设备
    :param context: 上下文对象
    :param fru_info 备件父框信息
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)

    enc_height = "--"
    if fru_info is None:
        return

    height = Tlv.Record.getIntValue(
        fru_info, TlvEnum.Hardware.ENCLOSURE["height"])

    logic_type = Tlv.Record.getIntValue(
        fru_info, TlvEnum.Hardware.ENCLOSURE["logicType"])

    if logic_type == TlvEnum.Enum.ENCLOSURE_TYPE_E['CTRL']:
        enc_height = str(height)

    contextUtil.setItem(
        context, Keys.CONTROLLER_ENC_HEIGHT, enc_height)
    logger.info("save context value %s=%s success" % (
        Keys.CONTROLLER_ENC_HEIGHT, enc_height))


def set_enc_height_from_rest_info(context, fru_info):
    """
    从fru_info中获取控制框高度，并保存到上下文中
    适用DoradoV3等通过rest方式获取到的fru_info的设备
    :param context: 上下文对象
    :param fru_info 备件父框信息
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)

    enc_height = "--"
    if fru_info is None:
        return

    height = restUtil.Tlv2Rest.getRecordValue(
        fru_info, restData.Hardware.Enclosure.HEIGHT)

    logic_type = restUtil.Tlv2Rest.getRecordValue(
        fru_info, restData.Hardware.Enclosure.LOGIC_TYPE)

    if logic_type == restData.Enum.EncLogicTypeEnum.CTRL:
        enc_height = str(height)

    contextUtil.setItem(context, Keys.CONTROLLER_ENC_HEIGHT, enc_height)
    logger.info("save context value %s=%s success" % (
        Keys.CONTROLLER_ENC_HEIGHT, enc_height))


def is_standard_sas_link(start_port, peer_port, standard_link):
    """判断线缆是否符合标准组网

    :param start_port: 始端端口
    :param peer_port: 末端端口
    :param standard_link: 标准组网dict
    :return: 在标准组网中返回True 否则返回False
    """
    # 标准组网dict始末端反转 因为A-B和B-A是同一根线缆
    for (key, value) in standard_link.items():
        standard_link.setdefault(value, key)
    # 检查线缆是否符合标准组网
    for (key, value) in standard_link.items():
        if (start_port, peer_port) == (key, value):
            return True
    return False


def count_precopy_time(disk_capacity):
    """计算预拷贝消耗最大时间

    :param disk_capacity: 硬盘容量（单位：MB）
    :return: 预拷贝时间（单位：秒）
    """
    # pool预拷贝带宽在最大业务量下固定为88Mb/s
    LOW_PRECOPY_BANDWIDTH = 88
    # 预留10%的时间
    RESERVED_PERCENT = 0.1

    return int(math.ceil(disk_capacity / LOW_PRECOPY_BANDWIDTH
                         * (1 + RESERVED_PERCENT)))


def is_smart_io_container_card(context):
    """
    是否是 SmartIo容器卡
    :param context:
    :return:
    """
    selected_record = BaseFactory.json.toDict(
        context.get("input_selectfru_frontEndIntftab"))
    return (int(selected_record.get("originModel", '0')) in
            restData.CONTAINER_CARD_MODE + restData.CONTAINER_CARD)


def check_node_switch_status(context, ctrl_id):
    """
    检查node业务是否回切
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    lang = contextUtil.getLang(context)
    cli = contextUtil.getCli(context)
    product_model = contextUtil.getProductModel(context)
    logger.logInfo("product_model: %s." % product_model)
    product_version = contextUtil.getCurVersion(context)
    logger.logInfo("product_version: %s." % product_version)
    if product_model not in DORADO_DEVS_V6:
        return True
    if Products.compareVersion(product_version, "6.1.0") < 0:
        return True
    is_high_end = baseUtil.isDoradoV6HighEnd(product_model)
    # 获取控制器的node_id
    flag, node_id = get_node_by_ctrl(is_high_end, ctrl_id)
    if not flag:
        return False
    logger.logInfo("node_id: %s." % node_id)

    cmd = "sys.sh showvnode"
    flag, cli_ret, err_msg = cliUtil.excuteCmdInMinisystemModel(cli, cmd, lang)
    cliUtil.enterCliModeFromSomeModel(cli, lang)
    if flag is not True:
        return False
    data = cliUtil.getHorizontalCliRet(cli_ret)
    logger.logInfo("data: %s." % data)

    # 判断node业务是否回切
    for item in data:
        home_node = item.get("HomeNode")
        work_node = item.get("WorkNode")
        pair_status = item.get("PairStatus")
        switch_status = item.get("switchstatus")
        if node_id != home_node:
            continue
        if not all([home_node == work_node,
                    pair_status == "double",
                    switch_status == "normal"]):
            return False
    return True


def check_login_controller_error_code(context, ctrl_id):
    """
    检查待更换控制器是否有'Error Code : (.*)'回显

    :param context: 上下文对象
    :param ctrl_id: 待登录控制器
    :return: True: 有特定回显，需要结束死等，提示控制器上电失败； False：没有
    """

    dev_model = contextUtil.getProductModel(context)
    is_high_end = dev_model in (CTRL_6U_PRODUCTS + DORADO_DEVS_V6_HIGH + ARM_HIGH_END_NEW)
    logger = baseUtil.getLogger(context.get("logger"), __file__)

    # 获取目标节点
    flag, target_node = controller.get_node_by_ctrl(is_high_end, ctrl_id)
    logger.logInfo("target node is:{}".format(target_node))
    heart_cmd = "sshtoremoteExt {}".format(target_node)
    flag, error_code, heart_ret = get_error_info(context, heart_cmd, target_node, logger)
    # 排除黑名单里的错误码
    if error_code in ERROR_CODE_IN_BLACKLIST:
        return False, '', ''
    return flag, error_code, heart_ret


def get_error_info(context, heart_cmd, target_node, logger):
    """
    解析心跳回显，获取错误码

    :param context: 上下文对象
    :param heart_cmd: 心跳命令
    :param target_node: 目标节点
    :param logger: logger
    :return: True：同一错误码连报五次； False：情况一：心跳失败；情况二：未获取到错误码；情况三：异常情况
    """
    cli = contextUtil.getCli(context)
    dev_obj = contextUtil.getDevObj(context)
    lang = contextUtil.getLang(context)
    from com.huawei.ism.tool.obase.exception import ToolException
    try:
        logger.logInfo("the heart_cmd is:{}".format(heart_cmd))
        heart_flag, heart_ret, heart_err_msg = cliUtil.ssh_remote_ctrl_even_mini_sys(cli, heart_cmd,
                                                                                     dev_obj.get("pawd"), lang)
        # 心跳失败
        if not heart_flag:
            logger.logInfo("Failed to heartbeat to node:{}".format(target_node))
            return False, '', heart_ret
        # 心跳成功，解析错误码
        logger.logInfo("the heart_ret is:{}".format(heart_ret))
        logger.logInfo("Succeed to heartbeat to node:{}".format(target_node))
        # 退出心跳
        cliUtil.exitHeartbeatCli(cli, lang)
        for line in heart_ret.splitlines():
            match_obj = ERROR_CODE_PATTERN.match(line)
            if match_obj:
                return True, match_obj.group(1), heart_ret
        return False, '', heart_ret

    except (ToolException, Exception) as ex:
        logger.logException(ex)
        return False, '', ''

    finally:
        # 退出minisystem
        cliUtil.enterCliModeFromSomeModel(cli, lang)


def show_err_ui(context, heart_ret, occur_error_code, lang, logger):
    """
    错误信息提示

    :param context: context
    :param heart_ret: 回显
    :param occur_error_code: 错误码
    :param lang: lang
    :param logger: logger
    """
    result_dict = {"flag": True, "errMsg": "", "suggestion": ""}
    heart_ret = handle_heart_ret(heart_ret, occur_error_code)
    result_dict["flag"] = False
    result_dict["errMsg"], result_dict["suggestion"] = getMsg(lang,
                                                              "controller.health.status.error.code",
                                                              (occur_error_code, heart_ret))
    contextUtil.handleFailure(context, result_dict)
    logger.logNoPass("[check_error_code] the error code is: {}".format(occur_error_code))


def check_error_code(context, ctrl_id):
    """
    对于特定产品型号做检查错误码的判断

    :param context: 上下文对象
    :param ctrl_id: 控制器id
    :return: 是否支持检查，True：有特定错误码；False: 没查到或产品型号不支持
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    dev_version = contextUtil.getCurVersion(context)
    logger.logInfo("the dev_version is:{}".format(dev_version))
    for dev_ver in SUPPORT_CHECK_ERROR_CODE_VERSION:
        if dev_version.startswith(dev_ver):
            return check_login_controller_error_code(context, ctrl_id)
    # Dorado V6及其衍生型号，6.1.7及以后支持检查错误码的判断
    if is_risk_version_and_model(context):
        return check_login_controller_error_code(context, ctrl_id)
    return False, '', ''


def is_risk_version_and_model(context):
    """
    检查是否为风险版本和型号（支持备件最后一公里测试：Dorado V6及其衍生型号，6.1.7及以后支持进入小系统检查错误码，提前拦截控制器上电失败）
    @param context: 上下文
    @return: True：风险版本，False：非风险版本
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    dev_version = contextUtil.getCurVersion(context)
    product_model = contextUtil.getProductModel(context)
    logger.logInfo("product_model:{}, dev_version is:{}".format(product_model, dev_version))
    # 二级存储1.5.0及以后支持检查错误码的判断
    if baseUtil.is_ocean_protect(product_model) and Products.compareVersion(dev_version, "1.3.0") > 0:
        return True
    # 微存储1.2.0及以后支持检查错误码的判断
    if baseUtil.is_micro_dev(product_model) and Products.compareVersion(dev_version, "1.1.0") > 0:
        return True
    # Dorado V6和新融合6.1.7及以后支持检查错误码的判断
    if baseUtil.isDoradoV6Dev(product_model) and Products.compareVersion(dev_version, "6.1.6") > 0:
        return True
    return False


def handle_heart_ret(heart_ret, occur_error_code):
    """
    截取回显信息并返回显示

    :param heart_ret: 回显
    :param occur_error_code: 错误码
    :return: 截取后的原始信息
    """
    temp_list = []
    for line in heart_ret.splitlines():
        temp_list.append(line)
    index = -1
    for i, val in enumerate(temp_list):
        if val.endswith(occur_error_code):
            index = i
            break

    return "\n".join(str(line) for line in temp_list[index: -2])


def check_repeat_error_code(context, queue, lang, logger, ctrl_id):
    """
    检查错误码，检查标准：连续5次出现同一错误码
    使用场景：针对特定版本，在更换控制器后的健康检查项中检查如果上电识别，通过错误码提前识别，以节省用户等待时间

    :param context: context
    :param error_code_list: 错误码列表
    :param lang: lang
    :param logger: logger
    :param ctrl_id: 控制器id
    :return: True: 连续5次出现相同错误码；False：其他情况
    """
    try:
        flag, error_code, heart_ret = check_error_code(context, ctrl_id)
        if not flag:
            return False
        queue.append(error_code)
        if len(queue) == queue.maxlen and len(set(queue)) == 1:
            show_err_ui(context, heart_ret, error_code, lang, logger)
            return True
        return False

    except (JException, Exception) as ex:
        logger.logException(ex)
        return False
    finally:
        time.sleep(10)
