﻿# -*- coding: UTF-8 -*-
import re
import time

from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import FuncFactory
from cbb.business.operate.fru.common import resource
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil
from cbb.frame.cli import cliUtil
from cbb.frame.base.config import NEW_DORADO
from cbb.frame.rest.restUtil import Tlv2Rest
from cbb.business.operate.fru.common import constData
from cbb.frame.rest.restDataConstants import OperateLogObj

from com.huawei.ism.tlv.lang import UnsignedInt32
from java.lang import Exception as JException

# 接口卡上的端口类型列表
PORT_TYPE_OF_INTF_LIST = [restData.Enum.ObjEnum.ETH_PORT,
                          restData.Enum.ObjEnum.FC_PORT,
                          restData.Enum.ObjEnum.SAS_PORT,
                          ]


def getPortsOfIntf(context, intfId, portType=None):
    """
    @summary: 获取指定接口卡上的端口信息
    """
    portsOfIntf = None

    if portType is not None:
        ports = FuncFactory.getFruListInfo(context, portType)
        condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                    restData.Enum.ConditionTypeEnum.EQ, intfId)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        portsOfIntf = restUtil.Tlv2Rest.filter(ports, conditionList)
        return portsOfIntf

    for portType in PORT_TYPE_OF_INTF_LIST:
        ports = FuncFactory.getFruListInfo(context, portType)
        condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                    restData.Enum.ConditionTypeEnum.EQ, intfId)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        portsOfIntf = restUtil.Tlv2Rest.filter(ports, conditionList)

        if len(portsOfIntf) > 0:
            break

    return portsOfIntf


def getLinkedPortsOfIntf(context, intfId, portType=None):
    """
    @summary: 获取指定接口卡上已连接的端口
    """
    ports = getPortsOfIntf(context, intfId, portType)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.LINK_UP)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    linkedPorts = restUtil.Tlv2Rest.filter(ports, conditionList)
    return linkedPorts


def getLang(context):
    """
    @summary: 获取环境语言
    """
    return context.get("lan")


def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    '''
    @summary: 资源国际化
    @param lang: 语言lang
    @param res: 资源
    @param args: 资源对应的参数
    @param resDict: 资源字典
    @return: 经过国际化处理后的资源
    '''
    try:
        key = "%s_%s" % (res, lang)
        if not resDict.has_key(key):
            return "--"

        context = resDict.get(key)
        if "%s" in context or "%i" in context:
            context = context % args

        return context

    except:
        return "--"


def isIntfPowerOnSucc(context, intfId, location, overtime):
    """检查接口卡上电是否成功

    :param context: 环境变量
    :param intfId: 备件ID
    :param location: 备件location
    :param overtime: 超时时间
    :return:
    """
    # 等待开始上电
    time.sleep(25)

    runningStatus = getIntfRunningStatus(context, intfId)
    BaseFactory.log.info(context, "the first runningStatus of %s is %s" % (intfId, runningStatus))

    # 接口卡运行状态为UNKNOWN或UPGRADING、POWER_OFF则需要轮查
    startTime = time.time()
    while isIntfUpgrade(runningStatus) or isIntfPowerOff(runningStatus):
        time.sleep(30)
        # 接口卡上电超时
        if (time.time() - startTime) > overtime:
            BaseFactory.log.info(context, "power on time out, and the finally runningStatus of %s is %s" % (
            intfId, runningStatus))
            return False

        # 检查特定告警 如果有 说明硬件不兼容 直接返回失败
        if FuncFactory.exist_not_allowed_fru_alarm(context, location):
            return False

        runningStatus = getIntfRunningStatus(context, intfId)
        if isIntfRunning(runningStatus):
            return True
        if isIntfPowerOnFailed(context, intfId, runningStatus):
            return False

    if isIntfRunning(runningStatus):
        return True
    if isIntfPowerOnFailed(context, intfId, runningStatus):
        return False
    BaseFactory.log.info(context, "power on failed, and the final runningStatus of %s is %s" % (intfId, runningStatus))
    return False


def poweron_intf(context, intf_id, location, overtime):
    """
     使用普通接口检查接口卡上电是否成功
    :param context:
    :param intf_id:
    :param location:
    :param overtime:
    :return:
    """

    # 等待开始上电
    time.sleep(25)
    intf_rec = FuncFactory.getFruInfoByBatch(context,
                                             restData.Enum.ObjEnum.INTF_MODULE,
                                             intf_id)
    running_status = restUtil.Tlv2Rest.getRecordValue(intf_rec,
                                                      restData.Hardware.
                                                     IntfModule.RUNNING_STATUS)
    BaseFactory.log.info(context, "the first runningStatus of %s is %s"
                          % (intf_id, running_status))

    # 接口卡运行状态为UNKNOWN、POWER_OFF则需要轮查
    ABNORMAL_STATUS = [restData.Enum.RunningStatusEnum.UNKNOWN,
                       restData.Enum.RunningStatusEnum.POWER_OFF]
    start_time = time.time()
    while running_status in ABNORMAL_STATUS:
        time.sleep(30)
        # 接口卡上电超时
        if (time.time() - start_time) > overtime:
            BaseFactory.log.info(context,
                                 "power on time out, and the finally "
                                 "runningStatus of %s is %s"
                                 % (intf_id, running_status))
            return False

        # 检查特定告警 如果有 说明硬件不兼容 直接返回失败
        if FuncFactory.exist_not_allowed_fru_alarm(context, location):
            return False

        intf_rec = \
            FuncFactory.getFruInfoByBatch(context,
                                          restData.Enum.ObjEnum.INTF_MODULE,
                                          intf_id)
        running_status = \
            restUtil.Tlv2Rest.getRecordValue(intf_rec,
                                             restData.Hardware.
                                             IntfModule.RUNNING_STATUS)
        if isIntfRunning(running_status):
            return True
        if isIntfPowerOnFailed(context, intf_id, running_status):
            return False

    if isIntfRunning(running_status):
        return True
    if isIntfPowerOnFailed(context, intf_id, running_status):
        return False
    BaseFactory.log.info(context,
                         "power on failed, "
                         "and the final runningStatus of %s is %s"
                         % (intf_id, running_status))
    return False


def getIntfRunningStatus(context, intfId):
    """
    @summary: 获取接口卡运行状态
    """
    rec = getIntfInfo(context, intfId)
    if rec == None or len(rec) == 0:
        return None
    else:
        return restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.RUNNING_STATUS)


def isIntfRunning(runningStatus):
    """
    @summary: 检查接口卡运行状态是否为running
    """
    if runningStatus == restData.Enum.RunningStatusEnum.RUNNING:
        return True
    else:
        return False


def isIntfPowerOnFailed(context, intfId, runningStatus):
    """
    @summary: 检查接口卡运行状态是否为POWER_ON_FAILED
    """
    BaseFactory.log.info(context, "the running status of card:%s is %s." % (intfId, runningStatus))
    if runningStatus == restData.Enum.RunningStatusEnum.POWER_ON_ERROR:

        param1 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.INTF_MODULE)
        param2 = (restData.PublicAttributes.ID, intfId)
        paramList = restUtil.Tlv2Rest.getParamList(param1, param2)
        try:
            rest = contextUtil.getRest(context)
            rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.INTF_MODULE_POWER_OFF, paramList)
            BaseFactory.log.info(context, "The rec of Poweroff is %s" % str(rec))
        except Exception as ex:
            BaseFactory.log.error(context,
                                  "When the status is POWER_ON_FAILED,execute poweroff commond failed.The excaption :%s" % ex)
        return True

    return False


# 接口卡处在变化中的状态（一般是在升级中出现）
INTF_RUN_STATU_UPGRADE_LIST = [restData.Enum.RunningStatusEnum.UNKNOWN,
                               restData.Enum.RunningStatusEnum.UPGRADING,
                               ]


def isIntfUpgrade(runningStatus):
    """
    @summary: 检查接口卡是否正在升级
    """
    if runningStatus in INTF_RUN_STATU_UPGRADE_LIST:
        return True
    else:
        return False


def isIntfPowerOff(runningStatus):
    """
    @summary: 检查接口卡是否正在power off状态
    """
    if runningStatus == restData.Enum.RunningStatusEnum.POWER_OFF:
        return True
    else:
        return False


def getIntfInfo(context, intfId):
    """
    @summary: 
    """
    param0 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.INTF_MODULE)
    param1 = (restData.PublicAttributes.ID, intfId)
    paramlist = restUtil.Tlv2Rest.getParamList(param0, param1)
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_INTF_MODULE_STATUS, paramlist, False)
    return rec


def getLinkUpPortByLocation(portList, location):
    """
    @summary: 获取指定位置上的端口信息
    @param alarmList: 端口列表
    @param location: 一个代表位置信息的正则表达式
    """
    reg = re.compile(location)

    filterPortList = []

    for index in range(0, len(portList)):
        port = portList[index]
        # 运行状态信息在TLV中的位置为8
        runningStatus = restUtil.Tlv2Rest.getRecordValue(port, restData.PublicAttributes.RUNNING_STATUS)
        if runningStatus == restData.Enum.RunningStatusEnum.LINK_UP:
            # 端口位置信息在TLV中的位置为6
            portLocation = restUtil.Tlv2Rest.getRecordValue(port, restData.PublicAttributes.LOCATION)
            if reg.search(portLocation) is not None:
                filterPortList.append(port)

    return filterPortList


def getRedExpCard(context, expCardId):
    """
    获取所有的冗余级联板
    """

    # 获取冗余级联板
    expLocations = []
    redunDantExpBoard = getRedundantExpBoard(context, expCardId)
    if redunDantExpBoard is None:
        return (False, FuncFactory.LangKey.EXPBOARD_EXISTED_NONE_REDT_SAS_PATH, expLocations)

    expId = restUtil.Tlv2Rest.getRecordValue(redunDantExpBoard, restData.PublicAttributes.ID)

    BaseFactory.log.info(context, u"redunDantExpBoard = [%s], id = [%s]" % (str(redunDantExpBoard), expId))

    # 冗余级联板的运行的sasPort
    sasPort = getSasPortsInExpBoard(context, expId)
    BaseFactory.log.info(context, u"The running sasPort = [%s]" % str(sasPort))

    if sasPort is None:
        return (False, FuncFactory.LangKey.EXPBOARD_EXISTED_NONE_REDT_SAS_PATH, expLocations)

    BaseFactory.log.info(context, "redunDant SAS Port WWN :[%s]"
                         % restUtil.Tlv2Rest.getRecordValue(sasPort, restData.Hardware.SasPort.WWN))

    # 冗余环路所有的SAS端口enclosureWWNList
    redundantSasPorts = getAllSasPortsByWWN(context,
                                            restUtil.Tlv2Rest.getRecordValue(sasPort, restData.Hardware.SasPort.WWN),
                                            restUtil.Tlv2Rest.getRecordValue(redunDantExpBoard,
                                                                             restData.PublicAttributes.ID))

    BaseFactory.log.info(context, u"The all redundantSasPorts= [%s]" % str(redundantSasPorts))

    # (异常场景)如果WWN不存在，当前SAS端口确定是连到了其他环境中并且和当前环境共享此框，意思就是A板可能在当前环境中，B板连到了其他的环境中。（不处理）
    # 走到这个分支里面的意思是：获取冗余级联板上的SAS端口在接口卡上端口SAS PORT的 wwnList中不存在，表示可能记录的wwnList是待更换级联板的WWN.
    if redundantSasPorts is None:
        allRedunDantExpBoards = getRedundantByExpId(context, expCardId)
    else:
        # 取得所有的冗余级联板ID
        expIds = list(set(
            [restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.PARENT_ID) for rec in redundantSasPorts]))
        BaseFactory.log.info(context, u"the all expBoards= [%s]" % str(expIds))

        # 所有冗余级联板记录
        allRedunDantExpBoards = getExpBoardsBySasPorts(context, expIds)
        BaseFactory.log.info(context, u"get all expBoards records = [%s]" % str(allRedunDantExpBoards))

    # 所有级联板location
    expLocations = list(set(
        [restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.LOCATION) for rec in allRedunDantExpBoards]))
    BaseFactory.log.info(context, u"the redundant expBoards location = [%s]" % str(expLocations))
    return (True, "", expLocations)


def checkRedundantByAlarms(context, expCardId):
    """
    检查所有冗余级联板是否存在单链路告警或SAS线缆连接错误的告警
    返回结果 (flag, errorKey)->(是否存在告警True/False，错误信息/None)。
    """

    # 获取所有单链路和SAS错误的告警
    allSasAlarmDict = getAllSasAlarm(context)
    if allSasAlarmDict is not None:
        # 获取冗余级联板
        redunDantExpBoard = getRedundantExpBoard(context, expCardId)
        if redunDantExpBoard is None:
            return (True, FuncFactory.LangKey.EXPBOARD_EXISTED_NONE_REDT_SAS_PATH)

        BaseFactory.log.info(context, u"冗余级联板 redunDantExpBoard = [%s], id = [%s]" % (
        str(redunDantExpBoard), restUtil.Tlv2Rest.getRecordValue(redunDantExpBoard, restData.PublicAttributes.ID)))

        # 冗余级联板的运行的sasPort
        sasPort = getSasPortsInExpBoard(context, restUtil.Tlv2Rest.getRecordValue(redunDantExpBoard,
                                                                                  restData.PublicAttributes.ID))
        BaseFactory.log.info(context, u"冗余级联板的运行的 sasPort = [%s]" % str(sasPort))

        if sasPort is None:
            return (True, FuncFactory.LangKey.EXPBOARD_EXISTED_NONE_REDT_SAS_PATH)
        BaseFactory.log.info(context, "redunDant SAS Port WWN :[%s]"
                             % restUtil.Tlv2Rest.getRecordValue(sasPort, restData.Hardware.SasPort.WWN))

        # 冗余环路所有的SAS端口enclosureWWNList
        redundantSasPorts = getAllSasPortsByWWN(context, restUtil.Tlv2Rest.getRecordValue(sasPort,
                                                                                          restData.Hardware.SasPort.WWN),
                                                restUtil.Tlv2Rest.getRecordValue(redunDantExpBoard,
                                                                                 restData.PublicAttributes.ID))
        BaseFactory.log.info(context, u"冗余环路所有的SAS端口  redundantSasPorts= [%s]" % str(redundantSasPorts))

        # (异常场景)如果WWN不存在，当前SAS端口确定是连到了其他环境中并且和当前环境共享此框，意思就是A板可能在当前环境中，B板连到了其他的环境中。（不处理）
        # 走到这个分支里面的意思是：获取冗余级联板上的SAS端口在接口卡上端口SAS PORT的 wwnList中不存在，表示可能记录的wwnList是待更换级联板的WWN.
        if redundantSasPorts is None:
            allRedunDantExpBoards = getRedundantByExpId(context, expCardId)
        else:
            # 取得所有的冗余级联板ID
            expIds = list(set([restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.PARENT_ID) for rec in
                               redundantSasPorts]))
            BaseFactory.log.info(context, u"所有的冗余级联板ID expIds= [%s]" % str(expIds))

            # 所有冗余级联板记录
            allRedunDantExpBoards = getExpBoardsBySasPorts(context, expIds)
            BaseFactory.log.info(context, u"所有冗余级联板记录allRedunDantExpBoards = [%s]" % str(allRedunDantExpBoards))

        # 所有级联板location
        expLocations = list(set([restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.LOCATION) for rec in
                                 allRedunDantExpBoards]))
        BaseFactory.log.info(context, u"所有冗余级联板location expLocations = [%s]" % str(expLocations))

        # 检查是否存在告警在冗余级联板上
        return checkSasAlarm(context, allSasAlarmDict, expLocations)

    else:
        return (False, None)


def getRedundantByExpId(context, expCardId):
    replaceSasPort = getSasPortsInExpBoard(context, expCardId)
    replaceSasPort = getAllSasPortsByWWN(context, restUtil.Tlv2Rest.getRecordValue(replaceSasPort,
                                                                                   restData.Hardware.SasPort.WWN),
                                         expCardId)

    # 获取待更换级联板（A或B）板所在环路的所有（A或B）级联板。
    expIds = list(
        set([restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.PARENT_ID) for rec in replaceSasPort]))
    BaseFactory.log.info(context, u"getRedundantByExpId 所有的待更换的级联板所在链路的级联板ID expIds= [%s]" % str(expIds))

    allExpBoards = getExpBoardsBySasPorts(context, expIds)

    # 获取链路中所有的框：
    diskEncIds = list(
        set([restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.PARENT_ID) for rec in allExpBoards]))
    BaseFactory.log.info(context, u"getRedundantByExpId 所有的待更换的级联板所在链路的硬盘框ID diskEncIds= [%s]" % str(diskEncIds))

    allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.EXPBOARD)

    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.ID,
                                                restData.Enum.ConditionTypeEnum.NEAND, expIds)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQOR, diskEncIds)
    condition2 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.RUNNING)

    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1, condition2)

    return restUtil.Tlv2Rest.filter(allBoardRecs, conditionList)


def getAllSasAlarm(context):
    """
    Single单链路：#define ALM_DEV_FRAME_SINGLE_LINKED 0x000F00CE002A
    SAS线缆连接错误： #define ALM_DEV_EXP_PORT_CABLE_LINK_ERROR 0x000F00D00016
    """
    # SAS告警列表
    singleLinked = "0x000F00CE002A"
    errorLinked = "0x000F00D00016"
    sasAlarmDic = {
        str(int(singleLinked, 16)): FuncFactory.LangKey.ALM_DEV_FRAME_SINGLE_LINKED,
        str(int(errorLinked, 16)): FuncFactory.LangKey.ALM_DEV_EXP_PORT_CABLE_LINK_ERROR
    }
    currentAlarmDict = {}
    sasAlarmList = [singleLinked, errorLinked]
    # 将告警ID转换为10进制的字典
    sasAlarmIdDict = {}
    for almId in sasAlarmList:
        sasAlarmIdDict[str(int(almId, 16))] = 1
    # 获取系统中的所有告警
    retVal = FuncFactory.getSysAlarmsInfo(context)
    if not retVal["succ"]:
        BaseFactory.result.setResultFailByKeys(context, FuncFactory.LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                               FuncFactory.LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return

    # 判断系统告警中是否存在多路径告警
    sysAlarmList = retVal["info"]
    if sysAlarmList is not None:
        almNum = len(sysAlarmList)
        for index in range(0, almNum):
            curAlm = sysAlarmList[index]
            almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
            if str(almId) in sasAlarmIdDict:
                currentAlarmDict[str(almId)] = curAlm
                BaseFactory.log.error(context, "found SAS alarm. decId=" + str(almId)
                                      + "hexId=" + str(hex(almId)))
        BaseFactory.log.info(context, u"sasAlarmDic=【%s】检查的告警是：%s，currentAlarmDict.keys= %s" % (
        sasAlarmDic, str(currentAlarmDict), str(len(currentAlarmDict.keys()) > 0)))
        if len(currentAlarmDict.keys()) > 0:
            return currentAlarmDict

    return None


def checkSasAlarm(context, currentAlarmDict, expLocations):
    # 判断是否是更换级联板的告警
    # SAS告警列表
    singleLinked = "0x000F00CE002A"
    errorLinked = "0x000F00D00016"
    sasAlarmDic = {
        str(int(singleLinked, 16)): FuncFactory.LangKey.ALM_DEV_FRAME_SINGLE_LINKED,
        str(int(errorLinked, 16)): FuncFactory.LangKey.ALM_DEV_EXP_PORT_CABLE_LINK_ERROR
    }
    for key in currentAlarmDict.keys():
        alarmLocation = None
        alarmParamStr = None
        try:
            alarmParamStr = restUtil.Tlv2Rest.getRecordValue(currentAlarmDict[key],
                                                             restData.Sys.Alarm.CMO_ALARM_PARMETER)
            pamList = str(alarmParamStr).split(",")
            if key == str(int(singleLinked, 16)):
                # 单链路告警：8(STRING)=1,DAE100,A,210235980610E9000200
                # [FrameType]{0:控制框;1:硬盘框;2:引擎}
                # （ID [FrameID]，控制器 [CtrlID]{0:A;1:B}，
                # 框SN [sn]）单链路。

                alarmLocation = "%s.%s" % (pamList[1], pamList[2])

            elif key == str(int(errorLinked, 16)):
                # SAS线缆连接错误告警：1,DAE000,0,B,EXP,210235980810F4000130
                # 级联模块（[FrameType]{0:控制框;1:硬盘框;2:引擎}
                # [FrameID]，[PhyType]{0:SAS;1:FC}
                # 级联模块 [CtrlID]{0:A;1:B}，
                # 端口 [port-id]）线缆连接错误。
                alarmLocation = "%s.%s" % (pamList[1], pamList[3])
        except Exception as e:
            BaseFactory.log.error(context, u"[解析alarmParamStr参数错误]，告警字典：%s, exception info: %s" % (
                str(currentAlarmDict), str(e)))

        BaseFactory.log.info(context,
                             u"结果 ：alarmLocation=%s, expLocations=%s" % (str(alarmLocation), str(expLocations)))
        # 更换的级联板存在告警，直接返回了。
        if alarmLocation is not None and alarmLocation in expLocations:
            BaseFactory.log.info(context, u"检查的告警是：%s, 且存在级联板的告警：%s" % (str(alarmParamStr), alarmLocation))
            return (True, sasAlarmDic[key])

    # 不存在告警
    return (False, None)


def getExpBoardsBySasPorts(context, expIds):
    """
    获取所有级联板By id.
    """
    allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.EXPBOARD)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.ID,
                                                restData.Enum.ConditionTypeEnum.EQOR, expIds)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.RUNNING)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    return restUtil.Tlv2Rest.filter(allBoardRecs, conditionList)


def getSasPortsInExpBoard(context, expCardId):
    """
    根据级联板ID获取其上的linkUp的SAS端口
    """
    allBoardRecs = FuncFactory.get_fru_list_info_with_not_support_type(context, restData.Enum.ObjEnum.SAS_PORT)
    if allBoardRecs:
        allPCIePortRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.PCIE_PORT)
        if allPCIePortRecs:
            allBoardRecs.extend(allPCIePortRecs)
    else:
        allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.PCIE_PORT)

    allEthPort = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.ETH_PORT)
    allBoardRecs.extend(allEthPort)

    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, expCardId)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.LINK_UP)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allBoardRecs, conditionList)
    restUtil.Tlv2Rest.logRecList(context, tmpRecs, "the normal redundant expboard of %s:" % expCardId)
    if len(tmpRecs) == 0:
        BaseFactory.log.error(context, "no redundant and normal expboard of "
                              + expCardId + " is existed, check failed.")
        return None
    else:
        return tmpRecs[0]


def getPCIePortsInExpBoard(context, expCardId):
    """
    根据级联板ID获取其上的linkUp的SAS端口
    """
    allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.PCIE_PORT)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.SasPort.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, expCardId)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.SasPort.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.LINK_UP)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allBoardRecs, conditionList)
    restUtil.Tlv2Rest.logRecList(context, tmpRecs, "the normal redundant pcie expboard of %s:" % expCardId)
    if len(tmpRecs) == 0:
        BaseFactory.log.error(context, "no redundant and normal pcie expboard of "
                              + expCardId + " is existed, check failed.")
        return None
    else:
        return tmpRecs[0]


def getAllSasPortsByWWN(context, wwn, redExpId):
    """
    获取所有enclosureWWNList和目标enclosureWWNList相同的SAS端口
    """
    allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.SAS_PORT)

    enclosureWWNList = ''
    for record in allBoardRecs:
        wwnList = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.SasPort.ENCLOSURE_WWNLIST)
        BaseFactory.log.info(context,
                             "getAllSasPortsByWWN:wwn=%s, wwnList=%s, len(wwnList) > len(enclosureWWNList)=%s" % (
                                 wwn, wwnList, str(len(wwnList) > len(enclosureWWNList))))
        if wwn in wwnList and len(wwnList) > len(enclosureWWNList):
            BaseFactory.log.info(context, "enclosureWWNList=%s" % wwnList)
            enclosureWWNList = wwnList

    BaseFactory.log.info(context, "getAllSasPortsByWWN:enclosureWWNList=%s" % enclosureWWNList)

    sasPortRecords = []
    for record in allBoardRecs:
        wwn = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.SasPort.WWN)
        runningStatus = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.SasPort.RUNNING_STATUS)
        if enclosureWWNList != '' and wwn in enclosureWWNList and runningStatus == restData.Enum.RunningStatusEnum.LINK_UP:
            sasPortRecords.append(record)

    if len(sasPortRecords) == 0:
        BaseFactory.log.error(context, "no redundant and normal SAS_PORT of 【%s】 is existed, check failed." % str(
            enclosureWWNList))
        return None
    else:
        return sasPortRecords


def getRedundantExpBoard(context, expCardId):
    """
    根据级联板id获取同框下的健康运行中的冗余级联板
    """
    # 若框下expCardId的冗余级联板，冗余不存在或状态不正常，则检查不通过
    # 根据级联板id获取框
    expEncId = FuncFactory.getFruParentEncId(context, restData.Enum.ObjEnum.EXPBOARD, expCardId)

    # 所有级联板父节点是expEncId的，且不为expCardId健康运行中的级联板即为冗余级联板。
    allBoardRecs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.EXPBOARD)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                expEncId)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.ID,
                                                restData.Enum.ConditionTypeEnum.NE,
                                                expCardId)
    condition2 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    condition3 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.RUNNING)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1, condition2, condition3)
    tmpRecs = restUtil.Tlv2Rest.filter(allBoardRecs, conditionList)
    restUtil.Tlv2Rest.logRecList(context, tmpRecs, "the normal redundant expboard of %s:" % expCardId)
    if len(tmpRecs) == 0:
        BaseFactory.log.error(context, "no redundant and normal expboard of "
                              + expCardId + " is existed, check failed.")
        return None
    else:
        return tmpRecs[0]


def get_redundant_exp_board_by_status(context, exp_board_id):
    """根据级联板id获取同框下的健康运行中的冗余级联板，区分不在位和状态不正常

    :param context: 上下文
    :param exp_board_id: 级联板ID
    :return: @param0: 冗余级联板是否在位, True-在位 False-不在位
             @param1: 在位的冗余级联板状态（健康状态和运行状态）是否正常, True-正常 False-不正常
             @param2: 在位且正常的冗余级联板信息列表
    """
    # 根据级联板id获取框
    exp_enc_id = FuncFactory.getFruParentEncId(context, restData.Enum.ObjEnum.EXPBOARD, exp_board_id)

    # 先判断冗余级联板是否在位。
    all_board_recs = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.EXPBOARD)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                exp_enc_id)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.ID,
                                                restData.Enum.ConditionTypeEnum.NE,
                                                exp_board_id)
    condition_list0 = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    # 在位的冗余级联板
    red_board_in = restUtil.Tlv2Rest.filter(all_board_recs, condition_list0)
    if not red_board_in:
        BaseFactory.log.error(context, "no redundant expboard of %s is existed, check failed." % exp_board_id)
        return False, False, None

    # 再判断在位的冗余级联板状态
    condition2 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    condition3 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Expboard.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.RUNNING)
    condition_list1 = restUtil.Tlv2Rest.getConditionList(condition2, condition3)
    # 在位且状态正常的冗余级联板
    red_board_in_normal = restUtil.Tlv2Rest.filter(red_board_in, condition_list1)
    restUtil.Tlv2Rest.logRecList(context, red_board_in_normal, "the normal redundant expboard of %s:" % exp_board_id)
    if not red_board_in_normal:
        BaseFactory.log.error(context, "no redundant and normal expboard of %s is existed, check failed." % exp_board_id)
        return True, False, red_board_in[0]
    else:
        return True, True, red_board_in_normal[0]


def judgePcieConnect(context, portsRecs):
    '''
    @deprecated: 判断PCIE端口连接正确性(暂适用于3UDorado设备)
    @param portsRec:需要判断的端口信息
    @return: True or False
    '''
    wrongConnectList = []

    for port in portsRecs:
        location = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.PciePort.LOCATION)
        dswId = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.PciePort.DSW_ID)
        current_peer_port_id = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.PciePort.CURRENT_PEER_PORT_ID)
        suggest_peer_port_id = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.PciePort.SUGGEST_PEER_PORT_ID)
        portMark = location[-1]

        if portMark != str(dswId) or current_peer_port_id != suggest_peer_port_id:
            BaseFactory.log.info(context, "The port(ID:%s) connect wrong" % location)
            wrongConnectList.append(location)

    return wrongConnectList


def judgeMgmtAlarmForDorado(context):
    """
    Attention管理网口链路连接错误：#0xF00060007
    """
    errorLinked = "0xF00060007"
    mgmtAlarmList = [errorLinked]

    # 将告警ID转换为10进制的字典
    mgmtAlarmIdDict = {}
    for almId in mgmtAlarmList:
        mgmtAlarmIdDict[str(int(almId, 16))] = 1

    # 获取系统中的所有告警
    retVal = FuncFactory.getSysAlarmsInfo(context)
    if not retVal["succ"]:
        BaseFactory.result.setResultFailByKeys(context, FuncFactory.LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                               FuncFactory.LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return True

    # 判断系统告警中是否存在管理网口链路连接错误告警
    sysAlarmList = retVal["info"]
    if sysAlarmList is not None:
        almNum = len(sysAlarmList)
        for index in range(0, almNum):
            curAlm = sysAlarmList[index]
            almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
            if str(almId) not in mgmtAlarmIdDict:
                continue

            BaseFactory.log.error(context, "found Mgmt alarm. decId=" + str(almId)
                                  + "hexId=" + str(hex(almId)))
            BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.CABMGMT_EXISTED_CABMGMTALARM)
            return True

    return False


def getCtrlEncHight(context):
    # get controller enclosure list
    enclosureList = FuncFactory.getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Enclosure.LOGIC_TYPE,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.EnclosureTypeEnum.CTRL)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    ctrlEnclosureList = restUtil.Tlv2Rest.filter(enclosureList, conditionList)
    for ctrlEnclosure in ctrlEnclosureList:
        encHeight = restUtil.Tlv2Rest.getRecordValue(ctrlEnclosure, restData.Hardware.Enclosure.HEIGHT)
        return encHeight


def getProductVersion(context):
    spcVersion = "--"
    retryTimes = 3
    exception = None
    for t in range(retryTimes):
        try:
            BaseFactory.log.info(context, "[getProductVersion]get system version time:%s." % str(t))
            rest = contextUtil.getRest(context)
            resc = restUtil.Tlv2Rest.getFullProduct(rest)
            spcVersion = restUtil.Tlv2Rest.getRecordValue(resc, restData.Upgrade.LstVer.CMO_VER_CUR_VERSION)
            return spcVersion
        except Exception as e:
            exception = e
            try:
                time.sleep(10)
            except Exception as e1:
                BaseFactory.log.error(context, "[getProductVersion]sleep error:%s." % (e1))
    raise Exception('Get product version exception.')


def is_scm_pool_disk(smart_cache_pool_id, disk_type):
    """
    1. 判断是否是SCM盘，
    2. 是否用作smart cache pool成员盘
    :param smart_cache_pool_id: smart cache pool id
    :param disk_type: 硬盘类型 SCM : 17, NVME SSD: 14
    :return: True 是，False 不是SCM盘，不是smart cache pool成员盘
    """
    return all(
        [
            smart_cache_pool_id,
            disk_type,
            "SCM SSD" in disk_type,
            str(smart_cache_pool_id).isdigit(),
            FuncFactory.FRUCONST.INVALID_PORT_ID != smart_cache_pool_id
        ]
    )


def is_selected_scm_pool_disk(context, select_id):
    """
    1. 判断是否是SCM盘，
    2. 是否用作smart cache pool成员盘
    :param context: 上下文
    :param select_id: 硬盘常量
    :return: True 是，False 不是SCM盘，不是smart cache pool成员盘
    """
    # 新融合不支持smart cache pool
    if context.get("dev").get("type") in NEW_DORADO:
        return False
    disk = BaseFactory.json.toDict(context.get(select_id))
    smart_cache_pool_id = disk.get("smartCachePoolId", '')
    disk_type = disk.get("diskType")
    BaseFactory.log.info(context, "check is scm disk!")
    return is_scm_pool_disk(smart_cache_pool_id, disk_type)


def get_disk_info_in_cli_mode(context):
    """
    通过CLI获取disk信息
    :param context:
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    cmd = "show disk general"
    _, ret, _ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    return cliUtil.getHorizontalCliRet(ret)


def get_disk_role(disk_info_list, disk_location):
    """
    获取硬盘角色
    :param disk_info_list:
    :param disk_location:
    :return:
    """
    for disk_info in disk_info_list:
        if disk_info.get("ID") == disk_location:
            return disk_info.get("Role")
    return ''


def record_ctrl_enc_fru_replace_log(context, obj_type, sn_key, obj_key,
                                    row_key):
    """
    更换控制框操作日志
    :param context: 上下文
    :param obj_type: 类型 0:电源;1:框;2:BBU;3:控制器;4:级联模块;5:接口模块;6:辅助散热模块
    :param sn_key: 不同对象的 sn 的index不同
    :param obj_key: 批量取值的对象enum
    :param row_key: 选中的对象key
    """
    _record_fru_replace_log(context, obj_type, True, sn_key, obj_key, row_key)


def record_fru_replace_log(context, obj_type, sn_key, obj_key, row_key):
    """
    更换其他备件的操作日志
    :param context: 上下文
    :param obj_type: 类型 0:电源;1:框;2:BBU;3:控制器;4:级联模块;5:接口模块;6:辅助散热模块
    :param sn_key: 不同对象的 sn 的index不同
    :param obj_key: 批量取值的对象enum
    :param row_key: 选中的对象key
    """
    _record_fru_replace_log(context, obj_type, False, sn_key, obj_key, row_key)


def _record_fru_replace_log(context, obj_type, is_ctrl_enc,
                           sn_key, obj_key, row_key):
    """
    记录FRU更换日志
    :param context: 上下文
    :param obj_type: 类型 0:电源;1:框;2:BBU;3:控制器;4:级联模块;5:接口模块;6:辅助散热模块
    :param is_ctrl_enc: 是否控制框，控制框需要特殊处理参数
    :param sn_key: 不同对象的 sn 的index不同
    :param obj_key: 批量取值的对象enum
    :param row_key: 选中的对象key
    :return:
    """
    if not is_need_record_operate_log(context):
        BaseFactory.log.info(
            context, "current version is no need record operate log!"
        )
        return
    recorded_flag_key = "HAS_RECORD_OPERATE_LOG"
    sel_row = BaseFactory.json.toDict(context.get(row_key))
    if isinstance(sel_row, list):
        sel_row = sel_row[0]
    sn = sel_row.get(constData.Keys.SN)
    if context.get(recorded_flag_key):
        # 已记录日志不再重复记录
        BaseFactory.log.info(context, "has recorded the operate log!")
        return
    location = sel_row.get(constData.Keys.LOCATION)
    # 部分备件无location，只有name属性
    obj_name = sel_row.get(constData.Keys.NAME)
    if location == '--':
        location = obj_name
    recs = FuncFactory.getFruListInfo(context, obj_key)
    if recs is None:
        return  # 失败时将自动设置错误信息，此处直接返回
    sn_new = '--'
    for rec in recs:
        if restUtil.Tlv2Rest.getRecordValue(
            rec, restData.PublicAttributes.LOCATION
        ) == location or (
            sel_row[constData.Keys.LOCATION] == '--'
            and restUtil.Tlv2Rest.getRecordValue(
                rec, restData.PublicAttributes.NAME
            )
            == obj_name
        ):
            sn_new = restUtil.Tlv2Rest.getRecordValue(rec, sn_key)
    # 换框时，记录name
    if obj_type == OperateLogObj.ENCLOSURE:
        location = obj_name
    try:
        rest = contextUtil.getRest(context)
        result = Tlv2Rest.execCmdJlist(
            rest, restData.TlvCmd.RECORD_FRU_OPERATION_LOG,
            get_operate_params(obj_type, location, sn, sn_new, is_ctrl_enc)
        )
        if result is None:
            # 两种异常，一种返回值为None,另一种通过异常信息捕获。
            BaseFactory.log.error(context, "send operate log failed!")
    except JException as e:
        BaseFactory.log.error(
            context, "send operate log failed! {}".format(e))
    else:
        context.put(recorded_flag_key, True)
        BaseFactory.log.info(context, "send operate log successfully.")


def get_operate_params(obj_type, location, sn, sn_new, is_ctrl_enc):
    """
    获取参数
    :param obj_type:
    :param location:
    :param sn:
    :param sn_new:
    :param is_ctrl_enc:
    :return:
    """
    # 生命周期事件ID
    cmo_alarm_type_sys_life_cycle = 7
    # 一般硬件事件记录ID
    cmo_operate_event_id = 0x400F00CF0003
    # 控制框事件记录ID
    cmo_operate_event_id_ctrl_enc = 0x400F00CF0004
    sn = sn if sn else '--'
    sn_new = sn_new if sn_new else '--'
    ret = "{},{},{},{}".format(obj_type, location, sn, sn_new)
    param0 = (restData.Upgrade.FruOperationLog.CMO_OPERATE_EVENT_ID,
              cmo_operate_event_id)
    if is_ctrl_enc:
        param0 = (restData.Upgrade.FruOperationLog.CMO_OPERATE_EVENT_ID,
                  cmo_operate_event_id_ctrl_enc)
        ret = "{},{}".format(location, sn)
    param1 = (restData.Upgrade.FruOperationLog.CMO_OPERATE_EVENT_TYPE,
              cmo_alarm_type_sys_life_cycle)
    param3 = (restData.Upgrade.FruOperationLog.CMO_OPERATE_EVENT_PARAM, ret)
    return [param0, param1, param3]


def is_need_record_operate_log(context):
    from utils import Products
    p_version = context.get("dev").get("version")
    return any([p_version.startswith("6.") and
                Products.compareVersion(p_version, "6.1.3RC1") >= 0,
                p_version.startswith("1.") and
                Products.compareVersion(p_version, "1.1.RC1") >= 0])
