# -*- coding: UTF-8 -*-

""" rest接口和TLV接口适配器 """

import os
import defusedxml.ElementTree as ET

from cbb.frame.adapter.resouceSelector import ResourceSelector, ResourceType
from cbb.frame.adapter.resourceAdapter import ResourceAdapter
from cbb.frame.adapter.restService import RestService
from cbb.frame.adapter.tlvService import TlvService
from cbb.frame.context import contextUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv.tlvEnumFactory import TlvEnum


class ResourceXml(object):
    """ 解析xml的类 """

    def __init__(self):
        pass

    @staticmethod
    def parse():
        """解析xml文件

        :return: 解析后的xml文件信息（一个嵌套的字典)
                allResourceDict={"getHost":{
                    "selector":"commonSelector", "restImpl":{"className":"", },
                    "tlvImpl":{"className":"", "adapter":""}}}
        """
        scriptPath = os.path.split(os.path.realpath(__file__))[0]
        configFile = scriptPath + os.sep + "resources.xml"
        xmlElementTree = ET.parse(configFile)
        resourceElements = xmlElementTree.getiterator("resource")
        allResourceDict = dict()
        for resourceElement in resourceElements:
            resourceName = resourceElement.attrib.get("id", "").strip()
            resourceSelector = resourceElement.attrib.get("selector",
                                                          "").strip()
            resourceDict = dict(selector=resourceSelector)
            implElements = resourceElement.getchildren()
            for implElement in implElements:
                implType = implElement.tag
                implClass = implElement.attrib.get("className")
                implAdapter = implElement.attrib.get("adapter")
                implBean = implElement.attrib.get("bean")
                resourceDict[implType] = dict(
                    className=implClass, adapter=implAdapter, bean=implBean)
            allResourceDict[resourceName] = resourceDict
        return allResourceDict


class ResrouceFactory:
    def __init__(self, context):
        """ 解析配置文件，得到资源字典"""
        self.context = context
        if context.get("ALL_RESOURCES") is None:
            allResources = ResourceXml.parse()
            context["ALL_RESOURCES"] = allResources
        self.allResourceDict = context.get("ALL_RESOURCES")
        self.logger = contextUtil.getLogger(context)
        self.logger.info("allResourceDict=%s" % self.allResourceDict)
        self.resourceSelector = ResourceSelector(context)

    def executeInterface(self, name, *params):
        """ 执行selector，查找对应的实现 """
        currentResourceInfo = self.allResourceDict.get(name)
        selecotrName = currentResourceInfo.get("selector")
        self.logger.info("executeInterface name=%s, *params=%s"
                         % (name, params))

        selectorValue = ""
        if selecotrName:
            selectorFun = getattr(self.resourceSelector, selecotrName)
            if selectorFun:
                selectorValue = selectorFun()
        if selectorValue == "":
            if ResourceType.REST in currentResourceInfo.keys():
                selectorValue = ResourceType.REST
            if ResourceType.TLV in currentResourceInfo.keys():
                selectorValue = ResourceType.TLV

        implDict = currentResourceInfo.get(selectorValue)
        implInstance = self._getServiceImpl(implDict.get("className"))
        implFun = getattr(implInstance, name, None)
        if implFun is None:
            return
        if params:
            results = implFun(*params)
        else:
            results = implFun()

        # 执行adapter对结果进行处理
        adapterName = implDict.get("adapter")
        beanName = implDict.get("bean")
        self.logger.info("adapterName=%s, beanName=%s"
                         % (adapterName, beanName))
        if adapterName and beanName:
            bean = self._getBeanByName(beanName)
            self.logger.info("bean=%s" % bean)
            adapterFun = getattr(ResourceAdapter(), adapterName)
            if adapterFun:
                results = adapterFun(results, bean)
        return results

    def _getServiceImpl(self, className):
        if className == "restService":
            return RestService(self.context)
        elif className == "tlvService":
            return TlvService(self.context)
        return None

    @staticmethod
    def _getBeanByName(beanName):
        temps = beanName.split(".")
        obj = dict()
        if temps[0] == "TlvEnum":
            obj = TlvEnum
        elif temps[0] == "tlvData":
            obj = tlvData
        for i in range(1, len(temps)):
            obj = getattr(obj, temps[i])
        return obj
