# -*- coding: UTF-8 -*-
import traceback
from cbb.frame.dsl.dsl import Dsl, DslContext
from cbb.frame.faulttree.tree_node import CheckNode
from cbb.frame.dsl import adapter
from cbb.frame.faulttree.echo_adapter import EchoAdapter
from cbb.frame.dsl.fault_mode import ReturnIfHandler


class CheckStatus:
    PASS = "0"
    NOTPASS = "1"
    NOCHECK = "2"
    NO_SUPPORT = "3"
    WARNING = "4"
    UNKNOWN = "5"

    @staticmethod
    def merge_all_ctrl_result(lang, all_ret):
        all_status = []
        all_msgs = []
        for key in all_ret.keys():
            status, ctr_msg = all_ret.get(key)
            if status is not None:
                all_status.append(status)
            if ctr_msg:
                all_msgs.append(
                    adapter.get_msg(lang, "Dsl.contrller.checkitem.detail",
                                    (key, ctr_msg)))

        err_msg = "\n".join(all_msgs)
        if not all_status:
            return CheckStatus.NOCHECK

        if CheckStatus.NOTPASS in all_status:
            return CheckStatus.NOTPASS, err_msg
        if CheckStatus.NOCHECK in all_status:
            return CheckStatus.NOCHECK, err_msg
        if CheckStatus.NO_SUPPORT in all_status:
            return CheckStatus.NO_SUPPORT, err_msg
        if CheckStatus.WARNING in all_status:
            return CheckStatus.WARNING, err_msg
        if CheckStatus.UNKNOWN in all_status:
            return CheckStatus.UNKNOWN, err_msg
        return CheckStatus.PASS, ""


class BaseCheckItem(object):
    def __init__(self, context):
        self.logger = adapter.get_logger(context)
        self.lang = adapter.get_lang(context)
        self.context = context
        self.sn = adapter.get_sn(context)
        self.dsl_context = None
        self.echo_adapter = EchoAdapter(context)

        # 用于存放提示信息中用到的变量值
        self.data = {}

        # 最后一次执行的dsl对象
        self.last_dsl = None

        # 最后一个dsl执行的结果
        self.last_data = {}

        # 存放每个节点信息
        self.check_node = CheckNode(context, "", {})

    def check(self):
        try:
            return self.execute()
        except Exception:
            self.logger.error("execute check error{}".
                              format(traceback.format_exc()))
            return CheckStatus.NOCHECK, self.get_msg("query.result.abnormal")

    def execute(self):
        raise NotImplementedError

    def dsl(self, *args, **kwargs):
        if self.dsl_context:
            self.last_dsl = Dsl(self.dsl_context)
        else:
            dsl_context = DslContext(self.context)
            self.last_dsl = Dsl(dsl_context)

        try:
            ret = self.last_dsl.dsl(*args, **kwargs)
        except Exception as ex:
            ret = ReturnIfHandler(kwargs.get("return_if")).process_exception(ex)
        finally:
            if self.last_dsl.origin_info:
                self.check_node.origin_info.extend(self.last_dsl.origin_info)
        return ret

    def get_msg(self, key, *args):
        return adapter.get_msg(self.lang, key, *args)

    def create_check_node(self, node_id):
        """
        每个节点生成一个node实例
        :param node_id:
        :return:
        """
        self.check_node = CheckNode(self.context, node_id, self.data)
        self.logger.info("create check node={}".format(str(self.check_node)))
        return self.check_node

    def set_check_node(self, check_node):
        self.check_node = check_node

    def add_origin_info(self, info):
        self.check_node.origin_info.append(info)

    def get_origin_info(self):
        return "\n".join(self.check_node.origin_info)
