# -*- coding: UTF-8 -*-

#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import java.lang.Exception as jException
from cbb.frame.base.exception import *
from cbb.frame.base.funcUtils import wrap_all_exception_logged


class CliConnection(object):
    CLI_CAN_NOT_CONNECT_CODE = "1073949185"  # 与设备通信异常，请检查网络连接或设备状态是否正常
    RE_CONNECTION_TIMES = 3
    _instance = {}

    def __new__(cls, *args, **kwargs):
        ip = args[1]
        if ip not in cls._instance:
            cls._instance[ip] = super(CliConnection, cls).__new__(
                cls, *args, **kwargs
            )
        return cls._instance.get(ip)

    def __init__(self, connectorFactory, ip, sshPort, sshConnector=None, logger=None):
        self._cli = None
        self.connectorFactory = connectorFactory
        self.ip = ip
        self.sshPort = sshPort
        self.sshConnector = sshConnector
        self.logger = logger

    def create(self, user, pawd):
        reConnectionTimes = 0
        while self._cli is None and reConnectionTimes < CliConnection.RE_CONNECTION_TIMES:
            self.close()
            if self.sshConnector is None:
                self.sshConnector = self.connectorFactory.createSshConnector(self.ip, self.sshPort, user, pawd)
            self._cli = self.sshConnector.getConnectionNoException()
            reConnectionTimes += 1
        del pawd
        return self._cli

    def createForwardConnetor(self, dev):
        reConnectionTimes = 0
        while self._cli is None \
                and reConnectionTimes < CliConnection.RE_CONNECTION_TIMES:
            if self.sshConnector is None:
                self.sshConnector = \
                    self.connectorFactory.createSshForwardConnector(dev)
            self._cli = self.sshConnector.getConnectionNoException()
            reConnectionTimes += 1
        return self._cli

    @wrap_all_exception_logged(logger=None)
    def close(self):
        if self._cli is not None:
            try:
                self.exitHeartbeatCli()
                self._cli.close()
            finally:
                self._cli = None
        return

    def exitHeartbeatCli(self):
        try:
            for _ in range(3):
                cliRet = self._cli.execCmdNoLogTimout("exit", 5)

                while "(y/n)" in cliRet:
                    cliRet = self._cli.execCmdNoLogTimout("y", 5)

                # Compatible for debug version.
                if "Storage:~ #" in cliRet:
                    self._cli.execCmdNoLogTimout("exit", 5)
        except:
            if self.logger:
                self.logger.error("Heart beat except.")

    @staticmethod
    def check_connect(conn, logger, allow_not_admin=False):
        if conn is None:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_CONNECTION_EXCEPTION))

        try:
            if not conn.isConnected():
                if logger:
                    logger.info("Ssh connection not connected, "
                                "reconnecting...")
                conn.reConnect()

            stdOut = conn.getLoginMessage()
            if "Ctrl+c to exit or Input minisystem to minisystem mode" in str(stdOut):
                conn.execCmd("minisystem")
            if "Local node is offline, Press any key to exit" in str(stdOut):
                conn.execCmd("\n")
        except (Exception, jException):
            if logger:
                logger.error("Ssh connection except.")

        cliRet = conn.execCmd("show cli configuration")

        if "-bash:" in cliRet or "Storage:~" in cliRet:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_CONNECTION_EXCEPTION))
        if allow_not_admin:
            return True
        if "minisystem>" in cliRet:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_IN_MINISYSTEM_MODE))

        if "Try 'help'" in cliRet or "does not exist or has an incorrect format" in cliRet:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_CONNECTION_EXCEPTION))

        if "More enabled" not in cliRet:
            raise CliCmdException(ErrorCode(ErrorCodeSet.CLI_CONNECTION_EXCEPTION))

        return True

    def getCli(self, dev):
        try:
            if self._cli is None:
                self.createForwardConnetor(dev)
            self.check_connect(self._cli, self.logger)
        except (jException, Exception):
            self._cli.close()
            self._cli.reConnect()

        return self._cli

    @staticmethod
    def get_available_con(con, logger, allow_not_admin=False):
        """
        获取链接并检查有效性
        :param con: 原链接
        :param logger: logger
        :param allow_not_admin: 是否允许获取到的连接是非admin模式的，如果不允许则非admin模式时需要重连接以回到admin模式
        :return:
        """
        try:
            CliConnection.check_connect(con, logger, allow_not_admin)
        except (Exception, jException):
            con.reConnect()
        return con
