﻿# coding:utf-8
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.common.query.hardware.enclosure import get_high_density_enclosure
from cbb.frame.rest import restDataConstants as restData
from cbb.frame.rest import restUtil
from cbb.frame.tlv import tlvData
from cbb.frame.devGraph import devGraphCfg


def drawGraph(context, isReloadGraph=False, **propDict):
    """绘制设备图

    :param context:上下文对象
    :param propDict: 支持如下key:
            hardwareTypeList: 可选，硬件类型集合（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"），默认支持
                                所有的硬件设备图绘制
            isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，默认值为False
    :param isReloadGraph: 是否刷新设备图
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        defaultHardwareList = ["BAY", "ENCLOSURE", "CONTROLLER", "FAN_CTRL_MODULE", "FAN", "INTF_MODULE", "DISK",
                               "BACKUP_POWER", "POWER", "EXPBOARD", "SAS_PORT", "ETH_PORT", "PCIE_PORT", "FCoE_PORT",
                               "FC_PORT", "IB_PORT", "SFP_OPTICAL_TRANSCEIVER", ]
        hardwareTypeList = propDict.get("hardwareTypeList", defaultHardwareList)
        isBack = propDict.get("isBack", False)

        hardwareList = contextUtil.getItem(context, "hardwareList")
        logger.logInfo("context hardwareList:%s" % hardwareList)

        if hardwareList is None or isReloadGraph:
            hardwareList = queryArrayHardwareListByRest(context, hardwareTypeList)
            logger.logInfo("rest hardwareList:%s" % hardwareList)

        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": devGraphCfg.DEV_GRAPH_CFG_FILE, "deviceImgCfgPath": devGraphCfg.DEV_GRAPH_IMG_PATH}
        devGraphUtils.buildGraph(cfgMap, hardwareList, isBack, False)
        contextUtil.setItem(context, "hardwareList", hardwareList)

    except (JException, Exception) as exception:
        logger.logException(exception)
        pass

    return


def queryArrayHardwareListByRest(context, hardwareTypeList):
    """通过阵列 rest 连接查询阵列硬件信息.

    :param context: 上下文
    :param hardwareTypeList: 要查询的硬件类型列表.
    :return:
    """
    # 获取阵列的相关器件信息
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    hardwareList = []

    for hardwareType in hardwareTypeList:
        try:
            logger.logInfo("begin to get hardware[%s] info." % hardwareType)
            hardwareList.extend(getHardwareGraphListByRest(context, hardwareType))
            logger.logInfo("get hardware[%s] info successfully." % hardwareType)
        except (JException, Exception) as ex:
            logger.logException(ex)
            logger.logInfo("get hardware[%s] info failed." % hardwareType)

    return hardwareList


def getHardwareGraphListByRest(context, hardwareType):
    """ 获取绘制硬件图的信息集合

      所有硬件对象需要包含如下字段：typeName,id,location,modelName,parentId,parentType,healthStatus
                    框对象需要包含如下字段：name,SAS_DAE_START_U,heightU,logicTypeName
                    端口对象需要包含如下字段：runningStatus,currentPeerId,suggestPeerId

    :param context:
    :param hardwareType:硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        rest = contextUtil.getRest(context)
    except (JException, Exception) as e:
        logger.logException(e)
        logger.logNoPass(
            'Query hardware failed because getting rest connection failed and hardware list empty in context.')
        return []

    ctrlEncHeight = contextUtil.getItem(context, 'ctrlEnclosureHeight', None)
    if ctrlEncHeight is None:
        ctrlEncHeight = queryCtrlEnclosureHeight(context, rest)
    # 高密框ID列表
    hd_enclosure_ids = contextUtil.getItem(context, "hd_enclosure_ids")
    if not hd_enclosure_ids:
        hd_enclosure_ids = get_high_density_enclosure(context)
        contextUtil.setItem(context, "hd_enclosure_ids", hd_enclosure_ids)

    hardwareGraphList = []
    intStartU = devGraphCfg.DEV_CTRL_DEFAULT_START_U
    hdwareEnumType = tlvData.OM_OBJ_E[hardwareType]
    records = restUtil.Tlv2Rest.getHardwareRecords(rest, hdwareEnumType)
    for record in records:
        infoDict = dict()
        infoDict["typeName"] = hardwareType
        infoDict["id"] = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["id"])
        infoDict["name"] = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["name"])
        location = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["location"])
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        parent_id = restUtil.Tlv2Rest.getRecordValue(
            record, restData.PUB_ATTR.get("parentID"))
        infoDict["modelName"] = getModelName(record, hardwareType, ctrlEncHeight)
        infoDict["parentId"] = parent_id
        infoDict["parentType"] = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["parentType"])
        infoDict["healthStatus"] = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["healthStatus"])
        infoDict["runningStatus"] = restUtil.Tlv2Rest.getRecordValue(record, restData.PUB_ATTR["runningStatus"])
        infoDict["startU"] = ""
        infoDict["heightU"] = ""
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = ""
        infoDict["suggestPeerId"] = ""

        if hardwareType == "ENCLOSURE":
            logicType = restUtil.Tlv2Rest.getRecordValue(record, restData.ENCLOSURE["logicType"])
            logicTypeName = restData.ENCLOSURE_TYPE.get(logicType, "")

            height = restUtil.Tlv2Rest.getRecordValue(record, restData.ENCLOSURE["height"])
            # 框逻辑类型或框高获取不到时，跳过，否则无法绘制设备图
            if not logicTypeName or not height:
                continue
            if location in ['', '--']:
                startU = str(intStartU)
                intStartU += height
                location = 'SMB0.%sU' % startU
            else:
                startU = location.split(".")[-1].replace("U", "")

            infoDict["location"] = location
            infoDict["startU"] = startU
            infoDict["heightU"] = height
            infoDict["logicTypeName"] = logicTypeName

        elif hardwareType == "SAS_PORT":
            infoDict["currentPeerId"] = restUtil.Tlv2Rest.getRecordValue(record,
                                                                         restData.SAS_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = restUtil.Tlv2Rest.getRecordValue(record,
                                                                         restData.SAS_PORT["suggest_peer_port_id"])

        elif hardwareType == "PCIE_PORT":
            infoDict["currentPeerId"] = restUtil.Tlv2Rest.getRecordValue(record,
                                                                         restData.PCIE_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = restUtil.Tlv2Rest.getRecordValue(record,
                                                                         restData.PCIE_PORT["suggest_peer_port_id"])
        # 高密框级联板硬件图与普通框的不同
        elif hardwareType == "EXPBOARD" \
                and hd_enclosure_ids and parent_id in hd_enclosure_ids:
            infoDict["modelName"] = "HIGH_DENSITY_SAS"

        for key in infoDict.keys():
            infoDict[key] = str(infoDict[key])

        hardwareGraphList.append(infoDict.copy())

    return hardwareGraphList


def queryCtrlEnclosureHeight(context, rest):
    """通过rest查询阵列控制框高度

    :param context:
    :param rest:
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    hdwareEnumType = tlvData.OM_OBJ_E['ENCLOSURE']
    records = restUtil.Tlv2Rest.getHardwareRecords(rest, hdwareEnumType)
    logger.logInfo("get enclosure records:%s" % records)
    for record in records:
        logicType = restUtil.Tlv2Rest.getRecordValue(record, restData.ENCLOSURE['logicType'])
        if logicType == restData.ENCLOSURE_TYPE_E["CTRL"]:
            enclosureHeight = restUtil.Tlv2Rest.getRecordValue(record, restData.ENCLOSURE['height'])
            logger.logInfo("get controller enclosure height success:%s" % enclosureHeight)
            break
    else:
        logger.logInfo("get controller enclosure height failed")
        enclosureHeight = None
    contextUtil.setItem(context, 'ctrlEnclosureHeight', enclosureHeight)
    return enclosureHeight


def getModelName(record, hardwareType, enclosureHeight):
    """获取设备图显示需要的modelName

    :param record:tlv记录
    :param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    :param enclosureHeight:框高（2U对应2,3U对应3,6U对应6）
    :return:
    """
    modelName = ""
    if hardwareType == "ENCLOSURE":
        model = restUtil.Tlv2Rest.getRecordValue(record, restData.ENCLOSURE["model"])
        modelName = restData.ENCLOSURE_MODEL.get(model, "")
    elif hardwareType == "INTF_MODULE":
        suffixDict = {2: "_CTRL_2U", 3: "_CTRL_3U", 4: "_CTRL_4U", 6: ""}  # 6U取默认值
        suffix = suffixDict.get(enclosureHeight, "")
        model = restUtil.Tlv2Rest.getRecordValue(record, restData.INTF_MODULE["model"])
        modelName = "%s%s" % (restData.INTF_MODEL.get(model, ""), suffix)
    elif hardwareType == "DISK":
        # 6U取默认值
        suffixDict = {2: "_CTRL_2U", 3: "_CTRL_3U", 4: "_CTRL_4U", 6: ""}
        suffix = suffixDict.get(enclosureHeight, "")
        model = restUtil.Tlv2Rest.getRecordValue(
            record, restData.DISK["disk_type"]
        )
        modelName = "{}{}".format(
            restData.DISK_TYPE_E_DICT.get(str(model), ""), suffix
        )
    elif hardwareType == "EXPBOARD":
        model = restUtil.Tlv2Rest.getRecordValue(record, restData.EXPBOARD["model"])
        modelName = restData.EXPBOARD_TYPE.get(model, "")
    return modelName
