# -*- coding: UTF-8 -*-
import datetime
import os
import re
import sys
import time
import traceback
import ast


class const():
    """工具总体常量定义类

    """
    RET_ERROR = -1
    FRU_PERSIST = "persist_fru_info"
    DATA_PERSIST = "persist_tool_data_dict"
    INVALID_SAS_PORTID = "4294967295"  # 此处必须为字符串
    INVALID_PCIE_PORTID = "4294967295"  # 此处必须为字符串
    INVALID_IPSCALEOUT_DSWID = "4294967295"  # 此处必须为字符串

    HELP_DESC_ZH = u"请联系技术支持工程师协助处理。"
    HELP_DESC_EN = "Contact technical support engineers for help."

    # 错误状态
    ERROR_STATUS = -1

    # 系统产品型号
    SYSMODEL = {"ERROR": ERROR_STATUS,  # 错误
                "S2600T": 0,  # 26T控制框
                "S5500T": 1,  # 55T控制框
                "T_ENGINE": 2,  # T独立机头
                "V3_3U_ENGINE": 3,  # 3U独立机头
                "V3_6U_ENGINE": 4,  # 6U独立机头
                "5X00V3": 5,  # 53V3/55V3控制框
                "2600V3": 6,  # 26V3控制框
                "V3_18000": 7,  # V3 18000控制框
                "2800V3": 8  # 2800V3控制框
                }

    EXPENCMODEL = {"ERROR": ERROR_STATUS,  # 错误
                   "T_2U": 0,  # T-2U硬盘框
                   "T_4U": 1,  # T-4U硬盘框
                   "T_4U75": 2,  # T-4U 75盘 硬盘框
                   "V3_2U": 3,  # V3-2U硬盘框
                   "V3_4U": 4  # V3-4U硬盘框
                   }


class returnVal():
    """通用返回值类：实现统一的字典返回格式

    """

    @staticmethod
    def dict2(succ, info):
        """构造两个返回值的字典

        :param succ: 执行结果是否成功
        :param info: 需要回传的信息
        :return: 两个值的字典
        """
        return {"succ": succ, "info": info}

    @staticmethod
    def dict3(succ, reason, suggestion):
        """构造三个返回值的字典

        :param succ: 执行结果是否成功
        :param reason: 原因
        :param suggestion: 建议
        :return:
        """
        return {"succ": succ, "reason": reason, "suggestion": suggestion}


class globalVar():
    """全局变量类：用于定义和存放所有脚本中均可访问的全局变量

    """
    # 工具上下文
    context = None

    @staticmethod
    def getContext():
        """获取全局可访问的工具上下文

        :return:
        """
        if None is globalVar.context:
            raise exception.newToolException(
                u"全局工具上下文已失效或未设置，工具无法运行。",
                const.HELP_DESC_ZH,
                "The global context of the tool has become invalid "
                "or has not been configured. The tool cannot work.",
                const.HELP_DESC_EN)
        else:
            return globalVar.context

    @staticmethod
    def setContext(context):
        """设置全局可访问的工具上下文

        :param context:
        :return:
        """
        if None is context:
            raise exception.newToolException(
                u"无效的工具上下文，工具无法运行。",
                const.HELP_DESC_ZH,
                "Invalid tool context. The tool cannot work.",
                const.HELP_DESC_EN)
        else:
            globalVar.context = context


class exception():
    """异常处理类：用于定义脚本框架中能识别的异常和处理机制

    """
    # 异常信息字段索引，可通过args[id]获取各字段描述信息
    argsId = {
        "FLAG": 0,  # 自定义异常标识
        "MSG_ZH": 1,  # 中文错误信息
        "SUG_ZH": 2,  # 中文处理建议
        "MSG_EN": 3,  # 英文错误信息
        "SUG_EN": 4,  # 英文处理建议
        "ERRID": 5  # 错误ID
    }

    # 默认工具异常名称：该类异常由工具脚本生成，包含中英文的异常描述和处理建议
    ALL_LANG_EXCEPTION = "ALL_LANG_EXCEPTION"

    @staticmethod
    def newToolException(errMsg_Zh, suggestion_Zh,
                         errMsg_En, suggestion_En, errCode=-1):
        """生成新的工具异常对象

        :param errMsg_Zh: 中文异常描述
        :param suggestion_Zh: 中文处理建议
        :param errMsg_En: 英文异常描述
        :param suggestion_En: 英文处理建议
        :param errCode: 工具框架异常将设置为相应的异常编号
        :return: 工具通用异常对象
        """
        return Exception(exception.ALL_LANG_EXCEPTION,
                         errMsg_Zh, suggestion_Zh, errMsg_En, suggestion_En,
                         str(errCode))

    @staticmethod
    def newToolExceptionByKeys(context, errKey, sugKey,
                               errParamTuple=(), sugParamTuple=(), errCode=-1):
        """生成新的工具异常对象

        :param context: 工具上下文
        :param errKey: 错误描述Key
        :param sugKey: 建议描述Key
        :param errParamTuple: 错误信息参数列表、元组 或单个字符串
        :param sugParamTuple: 建议信息参数列表、元组 或单个字符串
        :param errCode: 错误码
        :return: 工具通用异常对象
        """
        (errMsg, sugMsg) = lang.getErrAndSugDesc(
            context, errKey, sugKey, errParamTuple, sugParamTuple)
        if lang.isChinese(context):
            return exception.newToolException(errMsg, sugMsg, "", "", errCode)
        else:
            return exception.newToolException("", "", errMsg, sugMsg, errCode)

    @staticmethod
    def newUnexpectedToolException(strErrInfo_En=""):
        """生成新的工具异常对象

        :param strErrInfo_En: 英文错误信息
        :return: 包含信息的工具异常
        """
        return exception.newToolException(
            u"未知工具异常。" + strErrInfo_En,
            const.HELP_DESC_ZH,
            "Unknown tool exception." + strErrInfo_En,
            const.HELP_DESC_EN)

    @staticmethod
    def getToolExceptionByErrId(ex):
        """通过错误编号生成新的工具异常对象

        :param ex:
        :return:
        """
        # 容错：保证输入参数为整数
        errorId = 0
        try:
            errorId = int(str(ex.getErrorId()))
        except Exception:
            return exception.newUnexpectedToolException(
                "non-integer Framework Exception errorId="
                + str(errorId))

        # 据错误编号设置异常信息
        if 1 == errorId:  # 鉴权错误（如：密码错误）
            errMsg_Zh = u"用户名或密码错误。"
            suggestion_Zh = u"请更正用户名和密码。"
            errMsg_En = "Incorrect username or password."
            suggestion_En = "Enter correct username and password."
        elif 2 == errorId:  # 连接异常（如：超时异常,IP 不通）
            errMsg_Zh = u"设备连接异常。"
            suggestion_Zh = u"请确保设备和网络状态正常。"
            errMsg_En = "Abnormal device connection."
            suggestion_En = "Ensure that the status of the device " \
                            "and network is normal."
        elif 3 == errorId:  # 用户状态异常（只适用于TLV连接，如：未初始化）
            errMsg_Zh = ex.getErrMsg()
            suggestion_Zh = ""
            errMsg_En = ex.getErrMsg()
            suggestion_En = ""
        elif 4 == errorId:  # 设备状态异常（只适用于TLV连接，如：开工失败）
            errMsg_Zh = u"设备状态异常。"
            suggestion_Zh = u"请确保设备状态正常。"
            errMsg_En = "Abnormal device status."
            suggestion_En = "Ensure that the device status is normal."
        elif 9 == errorId:  # 工具框架返回的其他异常类型
            errMsg_Zh = ex.getErrMsg()
            suggestion_Zh = ""
            errMsg_En = ex.getErrMsg()
            suggestion_En = ""
        elif 1077949002 == errorId:
            errMsg_Zh = u"获取系统信息失败。"
            suggestion_Zh = const.HELP_DESC_ZH
            errMsg_En = "Cannot get the system infomation."
            suggestion_En = const.HELP_DESC_EN
        else:  # 其他异常
            errMsg_Zh = u"工具框架运行异常。"
            suggestion_Zh = const.HELP_DESC_ZH
            errMsg_En = "The tool works abnormally."
            suggestion_En = const.HELP_DESC_EN

        # 生成工具异常
        return exception.newToolException(
            errMsg_Zh, suggestion_Zh, errMsg_En, suggestion_En, errorId)

    FRAMEWORK_EXCEPTION = "FRAMEWORK_EXCEPTION"

    @staticmethod
    def newFrameworkException(errMsg, suggestion):
        """生成新的框架异常对象

        :param errMsg: 异常描述
        :param suggestion: 处理建议
        :return: 工具通用异常对象
        """
        return Exception(exception.FRAMEWORK_EXCEPTION, errMsg, suggestion)

    @staticmethod
    def handler(context, ex):
        """工具通用异常处理器

        :param context: 工具上下文
        :param ex: 工具通用异常
        :return:
        """
        # 参数检测，若入口参数无效，则抛出异常
        # 若工具上下文无效，则抛出新异常，让工具终止运行
        if None is context:
            raise Exception(u"无效的工具上下文。", const.HELP_DESC_ZH,
                            "Invalid tool context.", const.HELP_DESC_EN)

        # 先行记录异常的函数调用路径信息
        log.error(context,
                  "***********GENERAL EXCEPTION HANDLER BEGIN***************")
        log.error(context,
                  "the call trace of exception as follows:\n"
                  + unicode(traceback.format_exc()))

        # 若ex不为有效异常，则报错
        if type(ex) != Exception or len(ex.args) == 0:
            log.error(context,
                      "the param ex is not a invalid exception. ex="
                      + unicode(ex))
            result.setResultFail(context,
                                 u"工具运行错误。未知异常："
                                 + unicode(ex), const.HELP_DESC_ZH,
                                 "Tool running error. Unexpected exception: "
                                 + unicode(ex),
                                 const.HELP_DESC_EN)
            return False

        # 按异常的类型设置context
        if exception.ALL_LANG_EXCEPTION == ex.args[0]:  # 标准工具异常
            result.setResultFail(context, ex.args[1], ex.args[2],
                                 ex.args[3], ex.args[4])
            isSucc = True

        elif exception.FRAMEWORK_EXCEPTION == ex.args[0]:  # 封装的框架异常
            result.setResultFailByDesc(context, ex.args[1], ex.args[2])
            isSucc = True

        else:  # 未定义的异常
            log.error(context, "the param ex is not a standard exception. ex="
                      + unicode(ex))
            result.setResultFail(context,
                                 u"工具运行错误。非标准的异常："
                                 + unicode(ex), const.HELP_DESC_ZH,
                                 "Tool running error. Non-standard exception: "
                                 + unicode(ex),
                                 const.HELP_DESC_EN)
            isSucc = False

        # 记录错误信息
        log.error(context, result.getResultInfo(context))
        log.error(context,
                  "*************GENERAL EXCEPTION HANDLER END****************")
        return isSucc


class log():
    """日志记录类：提供各种级别的日志记录入口，并打印日志记录函数信息

    """

    @staticmethod
    def getCallerInfo(MAX_CALLER_LEVEL=5, skipLastLevel=True):
        """获取调用函数和调用行号

        :param MAX_CALLER_LEVEL: 最大调用关系层数
        :param skipLastLevel:
        :return:
        """
        # 从堆栈中获取调用函数和行号

        # 初始化参数
        funcBack = sys._getframe().f_back
        if skipLastLevel:  # 忽略最近的调用关系
            funcBack = funcBack.f_back
            MAX_CALLER_LEVEL -= 1

        # 生成函数调用关系
        callerInfo = ""
        for i in range(0, MAX_CALLER_LEVEL):

            # 获取该级调用函数和行号
            if hasattr(funcBack, "f_code") and hasattr(funcBack, "f_lineno"):
                funcName = funcBack.f_code.co_name
                lineNumber = funcBack.f_lineno
                callerInfo = " [" + str(funcName) + ":" + str(
                    lineNumber) + "]" + callerInfo
            else:
                break

            # 刷新Back函数
            if hasattr(funcBack, "f_back"):
                funcBack = funcBack.f_back
            else:
                break

        # 返回函数调用关系
        return callerInfo

    @staticmethod
    def debug(context, info):
        """记录调试信息info到工具日志中

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").debug('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)

    @staticmethod
    def error(context, info):
        """记录err级别日志

        :param context: 工具上下文
        :param info: 要记录的信息
        :return:
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").error('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)

    @staticmethod
    def info(context, info):
        """记录info级别日志

        :param context:
        :param info:
        :return:
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").info('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)

    @staticmethod
    def warn(context, info):
        """记录警告级别日志

        :param context:
        :param info:
        :return:
        """
        logInfo = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").warn('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)

    @staticmethod
    def test(info):
        """记录测试日志

        :param info:
        :return:
        """
        context = globalVar.getContext()
        logInfo = info + log.getCallerInfo()
        if "logger" in context:
            context.get("logger").info('[ToolLog]:' + logInfo)
            return True
        else:
            raise Exception("[failed]: logger is inexisted. info=" + logInfo)


class json():
    """JSON字符串相关函数类

    """

    @staticmethod
    def decodeUtf8ToUnicode(dictObj):
        if type(dictObj) == dict:
            # 逐一将字典的key和value转换为unicode
            newDict = {}
            for (key, value) in dictObj.items():
                newKey = unicode(key, "utf-8")
                newValue = unicode(value, "utf-8")
                newDict[newKey] = newValue
            return newDict
        else:
            return None

    @staticmethod
    def dictToStr(dictObj):
        """将字典转换为JSON字符串

        :param dictObj:
        :return:
        """
        if type(dictObj) == dict:
            strDict = "{"
            metFirst = False
            for (key, value) in dictObj.items():
                # 转换为unicode格式的str
                strKey = format.getUnicodeStr(key)
                strValue = format.getUnicodeStr(value)
                # 构建字典字符串
                if metFirst:
                    strDict = strDict + ', \"' \
                              + strKey + '\":\"' \
                              + strValue + '\"'
                else:
                    strDict = strDict + '\"' \
                              + strKey + '\":\"' \
                              + strValue + '\"'
                    metFirst = True
            strDict = strDict + '}'
            return strDict
        else:
            return None

    @staticmethod
    def dictToStr_TEST(context, dictObj):
        log.info(context, "dictObj=" + str(dictObj))
        if type(dictObj) == dict:
            strDict = "{"
            metFirst = False
            for (key, value) in dictObj.items():
                if metFirst:
                    log.info(context, "key=" + key + ", value=" + value)
                    strDict = strDict + ', \"' + key + '\":\"' + value + '\"'
                else:
                    log.info(context, "key=" + key + ", value=" + value)
                    strDict = strDict + '\"' + key + '\":\"' + value + '\"'
                    metFirst = True
            strDict = strDict + '}'
            return strDict
        else:
            return None

    @staticmethod
    def toStr(pyObj):
        """生成显示需要的字符串时使用

        :param pyObj: 字典或字典列表
        :return:
        """
        # 若输入为字典
        if type(pyObj) == dict:
            return json.dictToStr(pyObj)
        # 若输入为字典列表
        elif type(pyObj) == list:
            strObj = "["
            metFirst = False
            for index in range(0, len(pyObj)):
                strDict = json.dictToStr(pyObj[index])
                if strDict is not None:
                    if metFirst:
                        strObj = strObj + ', ' + strDict
                    else:
                        strObj = strObj + strDict
                        metFirst = True
                else:
                    return "Err: invalid sub dict: index=" + str(
                        index) + ", dict=" + str(pyObj[index])
            strObj = strObj + ']'
            return strObj
        else:  # 其他情况
            return "Err: invalid pyobj=" + str(pyObj)

    @staticmethod
    def toDict(strJson):
        """将JSON字符串转换为字典

        :param strJson:
        :return:
        """
        return ast.literal_eval(strJson)


class result():
    """处理结果设置类

    """

    @staticmethod
    def test(context):

        # 单Key语言设置接口测试
        result.setResultFailByKey(context, "EXEC_TLV_CMD_FAIL")
        log.error(context, "1_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        result.setResultFailByKey(context, "EXEC_TLV_CMD_FAIL", None, None)
        log.error(context, "2_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 双Key语言设置接口测试
        # 无参
        result.setResultFailByKeys(context, "EXEC_TLV_CMD_FAIL", "KEY_TEST")
        log.error(context, "3_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 字符串
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST", "[err]",
                                   "[sug]")
        log.error(context, "4_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 中英文列表，半空
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ["[err]", u"[参数]"], None)
        log.error(context, "5_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 半空，英中文列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST", None,
                                   [u"[错误]", "[param]"])
        log.error(context, "6_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 元组
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ("[err]", "[param]"), None)
        log.error(context, "7_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 元组
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST", None,
                                   ("[err]", "[param]"))
        log.error(context, "8_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 空元组，空列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST", (), ())
        log.error(context, "9_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        # 过多参数元组&列表
        result.setResultFailByKeys(context, "KEY_TEST", "KEY_TEST",
                                   ("[err]", "[param]", "123"),
                                   ["[err]", "[param]", "123"])
        log.error(context, "10_succ=" + str(context["succ"]))
        log.error(context, "errMsg=" + context["errMsg"])
        log.error(context, "suggestion=" + context["suggestion"])

        return

    @staticmethod
    def isResultFail(context):
        """判断结果是否失败

        :param context:
        :return:
        """
        if not context.get("succ", False):
            return True
        else:
            return False

    @staticmethod
    def setResult(context, succ, errMsg_zh, suggestion_zh, errMsg_en,
                  suggestion_en):
        """设置错误信心

        :param context: 上下文
        :param succ: 是否成功
        :param errMsg_zh: 中文错误信息
        :param suggestion_zh: 中文修复建议
        :param errMsg_en: 英文错误信息
        :param suggestion_en: 英文修复建议
        :return:
        """
        context["succ"] = succ
        lan = context.get("lan")
        if lan == "zh":
            context["errMsg"] = errMsg_zh
            context["suggestion"] = suggestion_zh
        else:
            context["errMsg"] = errMsg_en
            context["suggestion"] = suggestion_en
        return

    @staticmethod
    def setResultFail(context, errMsg_zh, suggestion_zh, errMsg_en,
                      suggestion_en):
        """设置失败结果接修复建议

        :param context:
        :param errMsg_zh:
        :param suggestion_zh:
        :param errMsg_en:
        :param suggestion_en:
        :return:
        """
        result.setResult(context, False, errMsg_zh, suggestion_zh, errMsg_en,
                         suggestion_en)
        log.error(context, "set result to fail.")
        return

    @staticmethod
    def setResultPass(context):
        """设置结果通过

        :param context:
        :return:
        """
        result.setResult(context, True, "", "", "", "")
        log.info(context, "set result to pass.")
        return

    @staticmethod
    def setResultFailByKey(context, msgKey, errParamTuple=(),
                           sugParamTuple=()):
        """使用相同的key设置错误和建议语言内容

        :param context:
        :param msgKey:
        :param errParamTuple:
        :param sugParamTuple:
        :return:
        """
        result.setResultFailByKeys(context, msgKey, msgKey, errParamTuple,
                                   sugParamTuple)
        return

    @staticmethod
    def setResultFailByKeys(context, errKey, sugKey, errParamTuple=(),
                            sugParamTuple=()):
        """直接填写错误和建议的语言描述key，内部自动从lang类获取语言并填充参数

        :param context: 工具上下文
        :param errKey: 错误描述Key
        :param sugKey: 建议描述Key
        :param errParamTuple: 错误信息参数列表、元组 或单个字符串
        :param sugParamTuple: 建议信息参数列表、元组 或单个字符串
        :return:
        """
        # 设置结果为不通过
        context["succ"] = False

        # 获取并设置错误和建议提示信息
        (errMsg, sugMsg) = lang.getErrAndSugDesc(context, errKey, sugKey,
                                                 errParamTuple, sugParamTuple)
        context["errMsg"] = errMsg
        context["suggestion"] = sugMsg

        log.error(context, "set result to fail.")
        return

    @staticmethod
    def setResultFailByDesc(context, errMsg, suggestion):
        """直接填写语言描述，用于和lang类配合使用(初级)

        :param context:
        :param errMsg:
        :param suggestion:
        :return:
        """
        context["succ"] = False
        context["errMsg"] = errMsg
        context["suggestion"] = suggestion
        return

    @staticmethod
    def getResultInfo(context):
        """获取context中的返回值描述信息

        :param context:
        :return:
        """
        rstInfo = "CONTEXT_RESULT_INFO: "

        errMsg = unicode(context["errMsg"], "utf-8")
        sugMsg = unicode(context["suggestion"], "utf-8")
        if context["succ"]:  # 执行成功
            rstInfo = rstInfo + "succ=True"
        elif not context["succ"]:  # 执行失败
            rstInfo = rstInfo + "succ=False, errMsg=" + errMsg \
                      + ", suggestion=" + sugMsg
        else:  # 未定义状态
            rstInfo = rstInfo + "succ=Undefined Value:[" \
                      + str(context["succ"]) \
                      + "], errMsg=" + errMsg + ", suggestion=" + sugMsg

        return rstInfo


class lang():
    """实现工具中自定义语言描述的自动获取

    """
    # 语言文件存放目录
    LANG_FILE_PATH = os.path.dirname(
        __file__) + "\\..\\..\\res\\language\\language.txt"

    # 信息保持项ID定义
    # DESC模块
    PERSIST_MODULE_DESC = "TOOL_DESC_MODULE"
    # 英文语言项列表
    PERSIST_ITEM_ENLIST = "DESC_LIST_ENGLISH"
    # 中文语言项列表
    PERSIST_ITEM_ZHLIST = "DESC_LIST_CHINESE"
    # 语言文件的修改时间。当语言文件的修改时间晚于context中记录的时间时，自动刷新context中的语言描述
    PERSIST_ITEM_UPDATE_TIME = "LANGUAGE_UPDATE_TIME"

    # 语言描述Key头
    HEAD_ERR = "ERR@"
    HEAD_SUG = "SUG@"

    @staticmethod
    def isChinese(context):
        """语言是否为中文

        :param context:
        :return:
        """
        return "zh" == context["lan"]

    @staticmethod
    def isEnglish(context):
        """语言是否为英文

        :param context:
        :return:
        """
        return "en" == context["lan"]

    @staticmethod
    def getErrAndSugDesc(context, errKey, sugKey, errParamTuple=(),
                         sugParamTuple=()):
        """据语言KEY获取错误和建议描述

        :param context: 工具上下文
        :param errKey: 错误描述Key
        :param sugKey: 建议描述Key
        :param errParamTuple: 错误信息参数列表、元组 或单个字符串
        :param sugParamTuple: 建议信息参数列表、元组 或单个字符串
        :return:
        """

        # 设置错误提示信息
        errMsg = ""  # 默认为空
        if errKey is not None and "" != errKey:
            if errParamTuple is None:  # 传入参数为None
                errMsg = lang.getDesc(context, lang.HEAD_ERR + errKey)
            elif tuple == type(errParamTuple) or list == type(
                    errParamTuple):  # 传入参数为元组或列表
                errMsg = lang.getDesc(context, lang.HEAD_ERR + errKey,
                                      *errParamTuple)
            else:  # 传入参数为单个字符串或其他类型
                errMsg = lang.getDesc(context, lang.HEAD_ERR + errKey,
                                      format.getUnicodeStr(errParamTuple))

        # 设置错误建议信息
        sugMsg = ""  # 默认为空
        if sugKey is not None and "" != sugKey:
            if sugParamTuple is None:  # 传入参数为None
                sugMsg = lang.getDesc(context, lang.HEAD_SUG + sugKey)
            elif tuple == type(sugParamTuple) or list == type(
                    sugParamTuple):  # 传入参数为元组或列表
                sugMsg = lang.getDesc(context, lang.HEAD_SUG + sugKey,
                                      *sugParamTuple)
            else:  # 传入参数为单个字符串
                sugMsg = lang.getDesc(context, lang.HEAD_SUG + sugKey,
                                      format.getUnicodeStr(sugParamTuple))

        return (errMsg, sugMsg)

    @staticmethod
    def getDesc(context, descKey, *params):
        """据Context中的当前语言类型获取描述

        :param context:
        :param descKey:
        :param params:
        :return:
        """
        # params为由参数序列生成的元组（tuple）,若需再次以参数序列传递，需在传递时，在其前面加*
        # 另外，字典的参数序列传递使用**进行。
        return lang.getDescByLang(context, context["lan"], descKey, *params)

    @staticmethod
    def getDescZh(context, descKey, *params):
        """获取中文语言描述

        :param context:
        :param descKey:
        :param params:
        :return:
        """
        return lang.getDescByLang(context, "zh", descKey, *params)

    @staticmethod
    def getDescEn(context, descKey, *params):
        """获取英文语言描述

        :param context:
        :param descKey:
        :param params:
        :return:
        """
        return lang.getDescByLang(context, "en", descKey, *params)

    @staticmethod
    def getDescByLang(context, langType, descKey, *params):
        """获取语言描述

        :param context: 工具上下文
        :param langType: 语言类型
        :param descKey: 语言描述Key
        :param params: 语言描述参数
        :return:
        """
        # 获取语言描述模块
        descModule = lang.getDescModule(context)
        if None is descModule:
            log.error(context, "fail to get the desc of key " + descKey
                      + " since desc module is inexisted.")
            return "???"

        # 获取语言描述列表
        if langType == "zh":  # 中文
            descList = descModule[lang.PERSIST_ITEM_ZHLIST]
        else:  # 英文
            descList = descModule[lang.PERSIST_ITEM_ENLIST]
        if descList is None:
            log.error(context, "fail to get the desc of key " + descKey
                      + " since desc list is inexisted.")
            return "???"

        # 获取并返回语言描述信息
        if descKey in descList:
            strDesc = descList[descKey]
        else:
            log.error(context,
                      "the key " + descKey + " is not existed in desc list.")
            return "???"

        # 替换所有参数并返回结果
        # 注意：描述行中的参数使用{index}进行标记，其中index为参数的位置，从0开始
        paramNum = len(params)
        for index in range(0, paramNum):
            strDesc = strDesc.replace("{" + str(index) + "}", params[index])
        return strDesc

    @staticmethod
    def initDescList(context):
        """从语言文件生成语言描述列表，并保存在context中

        :param context: 工具上下文
        :return:
        """
        # 判断语言文件是否存在
        if not os.path.isfile(lang.LANG_FILE_PATH):
            log.error(context,
                      "the lang file " + lang.LANG_FILE_PATH
                      + " is not existed.")
            raise exception.newToolException(
                u"工具语言文件不存在。",
                const.HELP_DESC_ZH,
                "Non-existent tool language file.",
                const.HELP_DESC_EN)

        # 读取语言文件内容
        descFile = None
        try:
            # 打开并读回所有描述文件信息
            descFile = open(lang.LANG_FILE_PATH)
            allDescInfo = descFile.read()
        except Exception:
            log.error(context, "fail to read lang file's contents.")
            allDescInfo = None
        finally:
            if descFile is not None:
                descFile.close()
        if None is allDescInfo:
            raise exception.newToolException(
                u"读取工具语言文件失败。",
                const.HELP_DESC_ZH,
                "Failed to read the tool language file.",
                const.HELP_DESC_EN)

        # 语言文件格式标识定义
        zhLinePrefix = "ZH>"  # 中文语言行描述前缀
        enLinePrefix = "EN>"  # 英文语言行描述前缀
        secSeperater = "="  # KEY和语言描述间的分隔符

        # 解析描述文件信息到中英文语言字典
        zhDescDict = {}
        enDescDict = {}
        allDescInfo = allDescInfo.splitlines()
        lineNum = len(allDescInfo)
        for index in range(0, lineNum):
            descLine = allDescInfo[index]
            descLine = descLine.lstrip()  # 去除左边的空格

            if "" == descLine.strip(" "):
                # 跳过空行
                continue
            elif descLine.startswith("#"):
                # 跳过注释行
                continue
            elif descLine.startswith(zhLinePrefix):
                # 为中文语言描述行
                # 去除语言描述前缀
                descLine = descLine[len(zhLinePrefix):].lstrip()
                # 获取语言key和内容
                eqIndex = descLine.index(secSeperater)
                descKey = descLine[:eqIndex].rstrip()
                descValue = descLine[eqIndex + 1:].decode("utf-8")
                # 设置到语言字典
                zhDescDict[descKey] = descValue
                continue
            elif descLine.startswith(enLinePrefix):
                # 为英文语言描述行
                # 去除语言描述前缀
                descLine = descLine[len(enLinePrefix):].lstrip()
                # 获取语言key和内容
                eqIndex = descLine.index(secSeperater)
                descKey = descLine[:eqIndex].rstrip()
                descValue = descLine[eqIndex + 1:].decode("utf-8")
                # 设置到语言字典
                enDescDict[descKey] = descValue
                continue
            else:
                # 记录错误并记录无效行
                log.error(context,
                          "invalid line(" + str(index + 1) + "): " + descLine)
                continue

        # 保存语言字典到context中，并返回初始化成功
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_ZHLIST, zhDescDict)
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_ENLIST, enDescDict)
        # 初始化语言文件修改时间
        curUpdateTime = time.ctime(os.stat(lang.LANG_FILE_PATH).st_mtime)
        log.info(context, "curUpdateTime=" + str(curUpdateTime))
        persist.setObject(context, lang.PERSIST_MODULE_DESC,
                          lang.PERSIST_ITEM_UPDATE_TIME, curUpdateTime)
        log.info(context, "succeed to init the language desc module.")
        return True

    @staticmethod
    def getDescModule(context):
        """获取语言描述模块

        :param context: 工具上下文
        :return:
        """
        # 从context中获取已保存的语言模块，若存在且未修改，则直接返回
        descModule = persist.getModule(context, lang.PERSIST_MODULE_DESC)
        if descModule is not None:
            lastUpdateTime = persist.getObject(context,
                                               lang.PERSIST_MODULE_DESC,
                                               lang.PERSIST_ITEM_UPDATE_TIME)
            curUpdateTime = time.ctime(
                os.stat(lang.LANG_FILE_PATH).st_mtime)  # 语言文件的当前修改时间
            # 如果初始化时记录的文件修改时间和当前查出的文件修改时间相等，则认为未更新过。否则删除已有语言描述，重新初始化
            if curUpdateTime == lastUpdateTime:
                return descModule
            else:
                persist.delModule(context, lang.PERSIST_MODULE_DESC)

        # 不存在时，则从语言文件创建语言模块并保存在context中
        if not lang.initDescList(context):
            log.error(context, "fail to init lang desc list.")
            return None

        # 再次获取已保存的语言模块
        return persist.getModule(context, lang.PERSIST_MODULE_DESC)


class ttime():
    """工具时间类（tool time）：用于实现工具时间的相关转换

    """

    @staticmethod
    def getTimeStrNow(format='%Y-%m-%d %H:%M:%S'):
        """获取当前时间的字符串

        :param format:
        :return:
        """
        return time.strftime(format)

    @staticmethod
    def getTimeStrFromObj(objTime, format='%Y-%m-%d %H:%M:%S'):
        """从时间对象生成时间字符串

        :param objTime:
        :param format:
        :return:
        """
        return datetime.datetime.strftime(objTime, format)

    @staticmethod
    def getTimeStrByTimeStamp(intStmpValue, format='%Y-%m-%d %H:%M:%S'):
        """从时间戳（数值）生成时间字符串

        :param intStmpValue:
        :param format:
        :return:
        """
        # 转换为时间结构体。传入的时间戳必须为整形
        tmpValue = time.localtime(intStmpValue)
        # 转为正常日期格式
        strTime = time.strftime(format, tmpValue)
        return strTime

    @staticmethod
    def getTimeStampByTimeStr(strTime, format='%Y-%m-%d %H:%M:%S'):
        """从时间字符串生成对应的时间戳

        :param strTime:
        :param format:
        :return:
        """
        stmp = time.mktime(time.strptime(strTime, format))
        return int(stmp)

    @staticmethod
    def getTimeObjNow():
        """获取当前时间对象

        :return:
        """
        return datetime.datetime.now()

    @staticmethod
    def getTimeObjFromStr(strTime, format='%Y-%m-%d %H:%M:%S'):
        """从时间字符串生成时间对象

        :param strTime:
        :param format:
        :return:
        """
        return datetime.datetime.strptime(strTime, format)

    @staticmethod
    def getTimeObjBeforeSeconds(objTime, longSeconds):
        """获取指定时间之前longSeconds的时间对象

        :param objTime:
        :param longSeconds:
        :return:
        """
        return objTime - datetime.timedelta(seconds=longSeconds)

    @staticmethod
    def getTimeObjAfterSeconds(objTime, longSeconds):
        """获取指定时间之后longSeconds的时间对象

        :param objTime:
        :param longSeconds:
        :return:
        """
        return objTime + datetime.timedelta(seconds=longSeconds)


class format():
    """数据格式类：将信息保存在上下文中，方便无调用关系的脚本中分享信息

    """

    @staticmethod
    def isInteger(strObj):
        """是否为整数格式

        :param strObj:
        :return:
        """
        return re.search("^[1-9][0-9]*$", str(strObj)) is not None

    @staticmethod
    def getUnicodeStr(obj):
        """将obj转换为字符串

        :param obj: 字符串
        :return:
        """
        # 常规字符串或unicode字符串，直接返回
        if isinstance(obj, str) or \
                isinstance(obj, unicode):
            return obj
        # 正数、字典、元组、列表
        elif type(obj) == int or \
                type(obj) == float or \
                type(obj) == bool or \
                type(obj) == dict or \
                type(obj) == tuple or \
                type(obj) == list:
            return str(obj)
        elif str(type(obj)) == "<type 'instance'>":
            return str(obj)
        else:  # 含有非英文字符
            return obj.decode("utf-8")

    @staticmethod
    def getUnicodeStrItemList(objList):
        """将objList中的所有元素转换为字符串，并重新构造为列表

        :param objList:
        :return:
        """
        if type(objList) not in [list, tuple]:
            return None

        strList = []
        for obj in objList:
            strList.append(format.getUnicodeStr(obj))

        return strList


class persist():
    """持久化信息类：将信息保存在上下文中，方便无调用关系的脚本中分享信息

    """
    # 持久化信息在context中的工具数据对象KEY
    ROOT = "tool_persist_data"

    @staticmethod
    def isObjectExisted(context, mdlName, objName):
        """指定模块中的指定对象是否存在

        :param context:
        :param mdlName:
        :param objName:
        :return:
        """
        try:
            obj = persist.getObject(context, mdlName, objName)
            return obj is not None
        except Exception:
            return False

    @staticmethod
    def getObject(context, mdlName, objName):
        """获取指定模块中的指定对象值

        :param context:
        :param mdlName:
        :param objName:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName or not objName:
            return None
        # 获取context中的工具数据字典
        dataDict = {}
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            return None
        # 获取工具数据中相应module的字典
        mdlDict = {}
        if mdlName in dataDict:
            mdlDict = dataDict[mdlName]
        else:
            return None
        # 获取模块数据中相应对象的信息
        if objName in mdlDict:
            return mdlDict[objName]
        else:
            return None

    @staticmethod
    def setObject(context, mdlName, objName, objValue):
        """设置指定模块中的指定对象值

        :param context:
        :param mdlName:
        :param objName:
        :param objValue:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName or not objName:
            log.info(context,
                     "failed to set tool object data of context. mdlName="
                     + mdlName + ", objName=" + objName)
            return False
        # 获取context中的工具数据字典，不存在则创建
        dataDict = {}
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            context[persist.ROOT] = dataDict
        # 获取工具数据中相应module的字典，不存在则创建
        mdlDict = {}
        if mdlName in dataDict:
            mdlDict = dataDict[mdlName]
        else:
            dataDict[mdlName] = mdlDict
        # 设置模块数据中相应对象的信息
        mdlDict[objName] = objValue
        # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
        context[persist.ROOT] = dataDict
        # 返回设置结果
        log.info(context,
                 "succeed to set tool object data of context. mdlName="
                 + mdlName + ", objName=" + objName)
        return True

    @staticmethod
    def delObject(context, mdlName, objName):
        """删除指定模块下的指定对象数据

        :param context:
        :param mdlName:
        :param objName:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName or not objName:
            log.info(context,
                     "failed to delete tool object data of context. mdlName="
                     + mdlName + ", objName=" + objName)
            return False
        # 获取context中的工具数据字典，不存在则创建
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            return True
        # 获取工具数据中相应module的字典，不存在则创建
        if mdlName in dataDict:
            mdlDict = dataDict[mdlName]
        else:
            return True
        # 删除模块数据中相应对象的信息
        if objName in mdlDict:
            del mdlDict[objName]
            # 返回删除结果
        log.info(context,
                 "succeed to delete tool object data of context. mdlName="
                 + mdlName + ", objName=" + objName)
        return True

    @staticmethod
    def isModuleExisted(context, mdlName):
        """指定模块是否存在

        :param context:
        :param mdlName:
        :return:
        """
        try:
            mdl = persist.getModule(context, mdlName)
            return mdl is not None
        except Exception:
            return False

    @staticmethod
    def getModule(context, mdlName):
        """获取指定的模块对象

        :param context:
        :param mdlName:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName:
            return None
        # 获取context中的工具数据字典
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            return None
        # 获取工具数据中相应module的字典
        mdlDict = None
        if mdlName in dataDict:
            mdlDict = dataDict[mdlName]
        return mdlDict

    @staticmethod
    def setModule(context, mdlName, mdlValue):
        """设置指定的模块

        :param context:
        :param mdlName:
        :param mdlValue:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName:
            log.info(context,
                     "failed to set tool module data of context. mdlName="
                     + mdlName)
            return False
        # 获取context中的工具数据字典，不存在则创建
        dataDict = {}
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            context[persist.ROOT] = dataDict
        # 设置工具数据中相应module的字典
        dataDict[mdlName] = mdlValue
        # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
        context[persist.ROOT] = dataDict
        # 返回设置结果
        log.info(context,
                 "succeed to set tool module data of context. mdlName="
                 + mdlName)
        return True

    @staticmethod
    def delModule(context, mdlName):
        """删除指定的模块数据

        :param context:
        :param mdlName:
        :return:
        """
        # 入口参数检查
        if not context or not mdlName:
            log.info(context,
                     "failed to set tool module data of context. mdlName="
                     + mdlName)
            return False
        # 获取context中的工具数据字典，不存在则创建
        if persist.ROOT in context:
            dataDict = context[persist.ROOT]
        else:
            return True
        # 删除工具数据中相应module的字典
        if mdlName in dataDict:
            del dataDict[mdlName]
        # 刷新dataDict对象：保险措施，防备dataDict对象值传不到context中
        context[persist.ROOT] = dataDict
        # 返回删除结果
        log.info(context,
                 "succeed to delete tool module data of context. mdlName="
                 + mdlName)
        return True

    @staticmethod
    def delAll(context):
        """删除所有工具的持久化数据

        :param context:
        :return:
        """
        # 入口参数检查
        if context is None:
            log.info(context, "failed to delete all tool data of context.")
            return False
        # 删除context中的工具数据字典
        if persist.ROOT in context:
            del context[persist.ROOT]
        # 返回删除结果
        log.info(context, "succeed to delete all tool data of context.")
        return True
