/*
 * Decompiled with CFR 0.152.
 */
package com.huawei.us.common.crypto;

import com.huawei.us.common.codec.USCodec;
import com.huawei.us.common.crypto.USCryptoConfig;
import com.huawei.us.common.crypto.USKeyUtils;
import com.huawei.us.common.crypto.USRandomizer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import org.apache.commons.lang3.StringUtils;
import org.owasp.esapi.crypto.CipherSpec;
import org.owasp.esapi.crypto.CipherText;
import org.owasp.esapi.crypto.PlainText;
import org.owasp.esapi.errors.EncryptionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class USCryptoBase {
    private static final Logger logger = LoggerFactory.getLogger(USCryptoBase.class);
    private static final int TAG_LENGTH = 128;
    private static final Set<String> ALLOWED_MODES = Stream.of(USCryptoConfig.ALLOWED_MODES.split(",")).collect(Collectors.toSet());

    private USCryptoBase() {
    }

    public static CipherText encrypt(SecretKey key, PlainText plainText, String xform) throws EncryptionException {
        if (Objects.isNull(key)) {
            throw new IllegalArgumentException("(Master) encryption key arg may not be null.");
        }
        if (Objects.isNull(plainText)) {
            throw new IllegalArgumentException("PlainText may arg not be null");
        }
        byte[] plainTextInByte = plainText.asBytes();
        boolean success = false;
        int keySize = key.getEncoded().length * 8;
        try {
            Cipher encrypter = Cipher.getInstance(xform);
            byte[] ivBytes = USRandomizer.getRandomBytes(encrypter.getBlockSize());
            CipherSpec cipherSpec = new CipherSpec(encrypter, keySize);
            cipherSpec.setIV(ivBytes);
            AlgorithmParameterSpec ivSpec = USCryptoBase.getAlgorithmParameterSpec(ivBytes, cipherSpec.getCipherMode());
            encrypter.init(1, (Key)key, ivSpec);
            byte[] raw = encrypter.doFinal(plainTextInByte);
            CipherText cipherText = new CipherText(cipherSpec, raw);
            success = true;
            CipherText cipherText2 = cipherText;
            return cipherText2;
        }
        catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
            throw new EncryptionException("Encryption failure (unavailable cipher requested)", "Encryption problem: specified algorithm in cipher xform " + xform + " not available: " + e.getMessage(), (Throwable)e);
        }
        catch (InvalidKeyException e) {
            throw new EncryptionException("Encryption failure: Invalid key exception.", "Requested key size: " + keySize + "bits greater than 128 bits. Must install unlimited strength crypto extension from Sun: " + e.getMessage(), (Throwable)e);
        }
        catch (InvalidAlgorithmParameterException e) {
            throw new EncryptionException("Encryption failure (invalid IV)", "Encryption problem: Invalid IV spec: " + e.getMessage(), (Throwable)e);
        }
        catch (BadPaddingException e) {
            throw new EncryptionException("Encryption failure", "[Note: Should NEVER happen in encryption mode.] Encryption problem: " + e.getMessage(), (Throwable)e);
        }
        catch (IllegalBlockSizeException e) {
            throw new EncryptionException("Encryption failure (no padding used; invalid input size)", "Encryption problem: Invalid input size without padding (" + xform + "). " + e.getMessage(), (Throwable)e);
        }
        finally {
            if (success) {
                plainText.overwrite();
            }
        }
    }

    private static AlgorithmParameterSpec getAlgorithmParameterSpec(byte[] ivBytes, String cipherMode) {
        if (!USCryptoBase.isLegalCipherMode(cipherMode)) {
            throw new IllegalArgumentException("illegal cipher mode:" + cipherMode);
        }
        if ("GCM".equalsIgnoreCase(cipherMode)) {
            return new GCMParameterSpec(128, ivBytes, 0, ivBytes.length);
        }
        return new IvParameterSpec(ivBytes);
    }

    private static boolean isLegalCipherMode(String cipherMode) {
        return ALLOWED_MODES.contains(cipherMode);
    }

    public static PlainText decrypt(SecretKey key, CipherText cipherText) throws EncryptionException {
        long start = System.nanoTime();
        boolean caughtException = false;
        if (Objects.isNull(key)) {
            throw new IllegalArgumentException("(Master) encryption key arg may not be null.");
        }
        if (Objects.isNull(cipherText)) {
            throw new IllegalArgumentException("CipherText may arg not be null");
        }
        try {
            PlainText plainText = USCryptoBase.doDecryption(key, cipherText);
            return plainText;
        }
        catch (EncryptionException e) {
            caughtException = true;
            logger.error("decryption failed, {}", (Object)e.getMessage());
            throw e;
        }
        finally {
            USCryptoBase.defenseForTimingAttack(caughtException, start);
        }
    }

    public static PlainText decryptForPlainText(SecretKey secretKey, String rawCipher, String xform) throws EncryptionException {
        if (StringUtils.isEmpty((CharSequence)rawCipher)) {
            throw new IllegalArgumentException("rawCipher arg may not be null.");
        }
        if (StringUtils.isEmpty((CharSequence)xform)) {
            throw new IllegalArgumentException("xform arg may not be null.");
        }
        try {
            byte[] cipherBytes = USCodec.decodeForHex(rawCipher);
            byte[] iv = USKeyUtils.getIvFromCipher(cipherBytes);
            byte[] content = USKeyUtils.getContentFromCipher(cipherBytes);
            int keySize = secretKey.getEncoded().length * 8;
            CipherSpec cipherSpec = new CipherSpec(xform, keySize, iv);
            CipherText cipherText = new CipherText(cipherSpec);
            cipherText.setCiphertext(content);
            return USCryptoBase.decrypt(secretKey, cipherText);
        }
        catch (NumberFormatException e) {
            throw new EncryptionException("invalid hex encoded input string", "invalid hex encoded input string");
        }
    }

    private static void defenseForTimingAttack(boolean caughtException, long start) {
        if (caughtException) {
            long now = System.nanoTime();
            long elapsed = now - start;
            int minDelaySec = 2;
            long nanoSecsInSec = 1000000000L;
            long nSecs = 2000000000L;
            if (elapsed < 2000000000L) {
                long extraSleep = 2000000000L - elapsed;
                long millis = extraSleep / 1000000L;
                long nanos = extraSleep - millis * 1000000L;
                assert (nanos >= 0L && nanos <= Integer.MAX_VALUE) : "Nanosecs out of bounds; nanos = " + nanos;
                try {
                    Thread.sleep(millis, (int)nanos);
                }
                catch (InterruptedException e) {
                    logger.error("fatal error in decryption, {}", (Object)e.getMessage());
                }
            }
        }
    }

    private static PlainText doDecryption(SecretKey key, CipherText cipherText) throws EncryptionException {
        try {
            Cipher decrypter = Cipher.getInstance(cipherText.getCipherTransformation());
            AlgorithmParameterSpec iv = USCryptoBase.getAlgorithmParameterSpec(cipherText.getIV(), cipherText.getCipherMode());
            decrypter.init(2, (Key)key, iv);
            byte[] output = decrypter.doFinal(cipherText.getRawCipherText());
            return new PlainText(output);
        }
        catch (NoSuchAlgorithmException e) {
            throw new EncryptionException("Decryption failed; see logs for details.", "Invalid algorithm for available JCE providers - " + cipherText.getCipherTransformation() + ": " + e.getMessage(), (Throwable)e);
        }
        catch (InvalidKeyException e) {
            throw new EncryptionException("Decryption failed; see logs for details.", "Must install JCE Unlimited Strength Jurisdiction Policy Files from Sun", (Throwable)e);
        }
        catch (InvalidAlgorithmParameterException | IllegalBlockSizeException e) {
            throw new EncryptionException("Decryption failed; see logs for details.", "Decryption problem: " + e.getMessage(), (Throwable)e);
        }
        catch (NoSuchPaddingException e) {
            throw new EncryptionException("Decryption failed; see logs for details.", "Invalid padding scheme (" + cipherText.getPaddingScheme() + ") for cipher transformation " + cipherText.getCipherTransformation() + ": " + e.getMessage(), (Throwable)e);
        }
        catch (BadPaddingException e) {
            throw new EncryptionException("Decryption failed; see logs for details.", "Decryption problem: WARNING: Adversary may have tampered with CipherText object orCipherText object mangled in transit: " + e.getMessage(), (Throwable)e);
        }
    }
}

