/*
 * Decompiled with CFR 0.152.
 */
package com.huawei.us.common.file;

import com.huawei.us.common.file.UsFileLiteUtils;
import com.huawei.us.common.os.UsOSSafeUtils;
import com.huawei.us.common.resource.SystemConfigUtil;
import com.huawei.us.common.string.UsStringUtils;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclEntryPermission;
import java.nio.file.attribute.AclEntryType;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.nio.file.attribute.UserPrincipal;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.text.MessageFormat;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class UsFileSafeUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(UsFileSafeUtils.class);
    private static final int DEFAULT_SYMLINK_DEPTH = 5;
    private static final List<Integer> SPECIAL_PERMISSIONS = Collections.unmodifiableList(Arrays.asList(1, 2, 3));
    private static final String STAT_CMD = "stat -c %a";
    private static final StandardOpenOption[] DEFAULT_INPUT_OPTIONS = new StandardOpenOption[]{StandardOpenOption.READ};
    private static final StandardOpenOption[] DEFAULT_OUTPUT_OPTIONS = new StandardOpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE};
    private static final StandardOpenOption[] DEFAULT_APPENDABLE_OUTPUT_OPTIONS = new StandardOpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.APPEND};
    private static final Set<String> SUPPORTED_FILE_ATTRIBUTE_VIEWS = Collections.unmodifiableSet(FileSystems.getDefault().supportedFileAttributeViews());
    private static final UserPrincipalLookupService LOOKUP_SERVICE = FileSystems.getDefault().getUserPrincipalLookupService();
    private static final String CURRENT_USER_NAME = System.getProperty("user.name");
    private static UserPrincipal posixRoot;
    private static UserPrincipal posixUser;
    private static UserPrincipal aclSystem;
    private static UserPrincipal aclAdministrators;
    private static UserPrincipal aclUser;
    private static FileAttribute<?> posixFilePermissionsAttribute;
    private static FileAttribute<?> posixDirectoryPermissionsAttribute;
    private static FileAttribute<?> aclFilePermissionsAttribute;

    public static InputStream getSafeInputStream(File file) throws IOException {
        Objects.requireNonNull(file, "Input file is null!");
        File safeFile = UsFileSafeUtils.newFile(file.getCanonicalPath(), new String[0]);
        if (!UsFileSafeUtils.isInSecureDir(safeFile)) {
            throw new IllegalArgumentException("File is not in secure directory!");
        }
        WrapAcl wrapAcl = UsFileSafeUtils.getWrapAcl(safeFile.toPath(), false, DEFAULT_INPUT_OPTIONS);
        Path inputPath = wrapAcl.getPath();
        if (UsFileSafeUtils.isRegularFile(inputPath)) {
            return Files.newInputStream(inputPath, wrapAcl.getOptions().toArray(new OpenOption[0]));
        }
        throw new IllegalArgumentException("Can't read from a not regular file " + file.getName());
    }

    public static OutputStream getSafeOutputStream(File file, boolean append) throws IOException {
        UsFileSafeUtils.createCanonicalFile(file, append);
        File safeFile = UsFileSafeUtils.newFile(file.getCanonicalPath(), new String[0]);
        WrapAcl wrapAcl = UsFileSafeUtils.getWrapAcl(safeFile.toPath(), false, append ? DEFAULT_APPENDABLE_OUTPUT_OPTIONS : DEFAULT_OUTPUT_OPTIONS);
        Path safePath = wrapAcl.getPath();
        if (UsFileSafeUtils.isRegularFile(safePath)) {
            return Files.newOutputStream(safePath, wrapAcl.getOptions().toArray(new OpenOption[0]));
        }
        throw new IllegalArgumentException("Can't output in not regular file " + file.getName());
    }

    public static File newFile(String filePath, String ... mores) {
        Objects.requireNonNull(filePath, "Input filePath is null!");
        String finalPath = Normalizer.normalize(filePath, Normalizer.Form.NFC);
        String childPath = Stream.of(mores).filter(Objects::nonNull).map(path -> Normalizer.normalize(path, Normalizer.Form.NFC)).collect(Collectors.joining(File.separator));
        if (StringUtils.isNotEmpty((CharSequence)childPath)) {
            finalPath = filePath.concat(File.separator).concat(childPath);
        }
        return Paths.get(UsFileSafeUtils.getSafeFilePath(finalPath), new String[0]).toFile();
    }

    public static String getSafeFilePath(String filePath) {
        Objects.requireNonNull(filePath, "Input filePath is null!");
        if (UsFileLiteUtils.isSecurityFileName(filePath) && filePath.indexOf(0) < 0) {
            return filePath;
        }
        throw new IllegalArgumentException("Not safe file path of " + filePath);
    }

    public static File createCanonicalFile(String filePath) {
        File createFile = UsFileSafeUtils.newFile(filePath, new String[0]);
        UsFileSafeUtils.createCanonicalFile(createFile, false);
        return createFile;
    }

    public static void createCanonicalFile(File file) {
        Objects.requireNonNull(file, "Input file is null!");
        UsFileSafeUtils.createCanonicalFile(UsFileSafeUtils.newFile(file.getPath(), new String[0]), false);
    }

    public static void createCanonicalFile(File file, boolean append) {
        Objects.requireNonNull(file, "Input file is null!");
        UsFileSafeUtils.createCanonicalFile(UsFileSafeUtils.newFile(file.getPath(), new String[0]).toPath(), append);
    }

    public static void createCanonicalFile(Path path, boolean append) {
        if (path == null || Files.exists(path, new LinkOption[0])) {
            return;
        }
        Path parentPath = path.getParent();
        if (parentPath != null && Files.exists(parentPath, new LinkOption[0]) && !UsFileSafeUtils.isSecureDir(parentPath)) {
            LOGGER.error("File {} is in not secure directory {}!", (Object)path.getFileName(), (Object)parentPath.getFileName());
            throw new IllegalArgumentException("File is in not secure directory!");
        }
        if (parentPath != null && !Files.exists(parentPath, new LinkOption[0])) {
            UsFileSafeUtils.createCanonicalDir(parentPath);
        }
        try {
            WrapAcl wrapAcl = UsFileSafeUtils.getWrapAcl(path, false, append ? DEFAULT_OUTPUT_OPTIONS : DEFAULT_APPENDABLE_OUTPUT_OPTIONS);
            Files.createFile(wrapAcl.getPath(), wrapAcl.getAttr());
        }
        catch (IOException e) {
            throw UsFileSafeUtils.wrapper("Failed to create canonical file!", e);
        }
    }

    public static File createCanonicalDir(String directory) {
        File createFile = UsFileSafeUtils.newFile(directory, new String[0]);
        UsFileSafeUtils.createCanonicalDir(createFile);
        return createFile;
    }

    public static void createCanonicalDir(File directory) {
        Objects.requireNonNull(directory, "Input directory is null!");
        UsFileSafeUtils.createCanonicalDir(UsFileSafeUtils.newFile(directory.getPath(), new String[0]).toPath());
    }

    public static void createCanonicalDir(Path path) {
        Path rootPath = path.toAbsolutePath().getRoot();
        if (rootPath == null) {
            return;
        }
        for (int index = 1; index <= path.getNameCount(); ++index) {
            Path subPath = path.subpath(0, index);
            Path partialPath = Paths.get(rootPath.toString(), subPath.toString());
            Path parentPath = partialPath.getParent();
            if (Files.exists(partialPath, new LinkOption[0]) || !UsFileSafeUtils.isSecureParent(parentPath)) continue;
            UsFileSafeUtils.createDirectory(partialPath);
        }
    }

    private static void createDirectory(Path partialPath) {
        try {
            WrapAcl wrapAcl = UsFileSafeUtils.getWrapAcl(partialPath, true, DEFAULT_OUTPUT_OPTIONS);
            Files.createDirectory(partialPath, wrapAcl.getAttr());
        }
        catch (IOException ex) {
            throw UsFileSafeUtils.wrapper("Failed to create directory " + partialPath.getFileName(), ex);
        }
    }

    private static boolean isSecureParent(Path parentPath) {
        return parentPath == null || Files.exists(parentPath, new LinkOption[0]) && UsFileSafeUtils.isSecureDir(parentPath);
    }

    public static boolean isRegularFile(File file) throws IOException {
        if (file != null) {
            String filePath = file.getCanonicalPath();
            String safeFilePath = UsFileSafeUtils.getSafeFilePath(filePath);
            return UsFileSafeUtils.isRegularFile(Paths.get(safeFilePath, new String[0]));
        }
        return false;
    }

    public static boolean isRegularFile(Path path) throws IOException {
        if (path != null) {
            BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
            return attr.isRegularFile();
        }
        return false;
    }

    public static boolean isPosix() {
        return SUPPORTED_FILE_ATTRIBUTE_VIEWS.contains("posix");
    }

    public static boolean isAcl() {
        return SUPPORTED_FILE_ATTRIBUTE_VIEWS.contains("acl");
    }

    public static boolean isInSecureDir(String filePath) {
        return UsFileSafeUtils.isInSecureDir(UsFileSafeUtils.newFile(filePath, new String[0]));
    }

    public static boolean isInSecureDir(File file) {
        if (file == null) {
            return false;
        }
        try {
            Path path = Paths.get(file.getCanonicalPath(), new String[0]);
            Path parent = path.getParent();
            return parent == null || UsFileSafeUtils.isSecureDir(parent, null);
        }
        catch (IOException ex) {
            throw UsFileSafeUtils.wrapper("Failed to get canonical path!", ex);
        }
    }

    public static boolean isSecureDir(String directory) {
        return UsFileSafeUtils.isSecureDir(UsFileSafeUtils.newFile(directory, new String[0]));
    }

    public static boolean isSecureDir(File directory) {
        if (directory == null) {
            return false;
        }
        try {
            Path path = Paths.get(directory.getCanonicalPath(), new String[0]);
            return Files.isDirectory(path, new LinkOption[0]) && UsFileSafeUtils.isSecureDir(path);
        }
        catch (IOException ex) {
            throw UsFileSafeUtils.wrapper("Failed to get canonical path!", ex);
        }
    }

    public static boolean isSecureDir(Path path) {
        return UsFileSafeUtils.isSecureDir(path, null);
    }

    public static boolean isSecureDir(Path path, UserPrincipal user) {
        return UsFileSafeUtils.isSecureDir(path, user, 5);
    }

    private static boolean isSecureDir(Path path, UserPrincipal user, int symlinkDepth) {
        boolean isSecureDir = true;
        Path absolutePath = path.toAbsolutePath();
        if (UsFileSafeUtils.isInWhiteList(path)) {
            LOGGER.info("Directory {} or parent directory is in {}!", (Object)absolutePath, (Object)"file.permission.check.white.list");
            return true;
        }
        if (UsFileSafeUtils.isPosix()) {
            isSecureDir = UsFileSafeUtils.isSecureDirForUnix(absolutePath, user, symlinkDepth);
        } else if (UsFileSafeUtils.isAcl()) {
            isSecureDir = UsFileSafeUtils.isSecureDirForWindows(absolutePath, user);
        }
        if (!isSecureDir) {
            LOGGER.error("Directory {} is not secure!", (Object)absolutePath);
        }
        return isSecureDir;
    }

    private static boolean isInWhiteList(Path path) {
        String whiteList = SystemConfigUtil.getStringValueByName((String)"file.permission.check.white.list", (String)"");
        if (StringUtils.isNotEmpty((CharSequence)whiteList)) {
            String[] whitePathList = UsStringUtils.strToArray((String)whiteList);
            return StringUtils.startsWithAny((CharSequence)(path.toString() + File.separator), (CharSequence[])((CharSequence[])Arrays.stream(whitePathList).map(whitePath -> whitePath + File.separator).toArray(String[]::new)));
        }
        return false;
    }

    private static boolean isSecureDirForWindows(Path path, UserPrincipal user) {
        Path rootPath = path.getRoot();
        if (rootPath == null) {
            return true;
        }
        if (user == null) {
            user = aclUser;
        }
        if (aclAdministrators == null || aclSystem == null || user == null) {
            LOGGER.error("administrators is null or system is null or user is null!");
            return false;
        }
        for (int i = 1; i <= path.getNameCount(); ++i) {
            Path subPath = path.subpath(0, i);
            Path partialPath = Paths.get(rootPath.toString(), subPath.toString());
            try {
                UserPrincipal owner = Files.getOwner(partialPath, new LinkOption[0]);
                if (user.equals(owner) || owner.equals(aclAdministrators) || owner.equals(aclSystem)) continue;
                LOGGER.error("Owner {} is not administrator {} or system {} or user {}!", new Object[]{owner.getName(), aclAdministrators.getName(), aclSystem.getName(), user.getName()});
                return false;
            }
            catch (IOException ex) {
                LOGGER.error("Failed to check user principal cause by {}!", (Object)ExceptionUtils.getRootCauseMessage((Throwable)ex));
                return false;
            }
        }
        return true;
    }

    private static boolean isSecureDirForUnix(Path path, UserPrincipal user, int symlinkDepth) {
        if (symlinkDepth <= 0) {
            LOGGER.error("symlinkDepth <= 0 ");
            return false;
        }
        Path rootPath = path.getRoot();
        if (rootPath == null) {
            return true;
        }
        if (user == null) {
            user = posixUser;
        }
        if (posixRoot == null || user == null) {
            LOGGER.error("root is null or user is null!");
            return false;
        }
        for (int index = 1; index <= path.getNameCount(); ++index) {
            Path subPath = path.subpath(0, index);
            Path partialPath = Paths.get(rootPath.toString(), subPath.toString());
            try {
                if (UsFileSafeUtils.isPartialPathSecure(partialPath, posixRoot, user, symlinkDepth)) continue;
                return false;
            }
            catch (IOException ex) {
                LOGGER.error("Failed to check user principal cause by {}!", (Object)ExceptionUtils.getRootCauseMessage((Throwable)ex));
                return false;
            }
        }
        return true;
    }

    private static boolean isPartialPathSecure(Path partialPath, UserPrincipal root, UserPrincipal user, int symlinkDepth) throws IOException {
        if (Files.isSymbolicLink(partialPath)) {
            return UsFileSafeUtils.isSecureDirForUnix(Files.readSymbolicLink(partialPath), user, symlinkDepth - 1);
        }
        UserPrincipal owner = Files.getOwner(partialPath, new LinkOption[0]);
        if (!user.equals(owner) && !root.equals(owner)) {
            LOGGER.error("Owner {} is not root {} or user {}!", new Object[]{owner.getName(), root.getName(), user.getName()});
            return false;
        }
        PosixFileAttributes attr = Files.readAttributes(partialPath, PosixFileAttributes.class, new LinkOption[0]);
        Set<PosixFilePermission> perms = attr.permissions();
        if (perms.contains((Object)PosixFilePermission.GROUP_WRITE) || perms.contains((Object)PosixFilePermission.OTHERS_WRITE)) {
            if (UsFileSafeUtils.isSpecialPermissionDirectory(partialPath)) {
                return true;
            }
            LOGGER.error("Permission contains group write or others write!");
            return false;
        }
        return true;
    }

    private static boolean isSpecialPermissionDirectory(Path path) throws IOException {
        String canonicalPath = Normalizer.normalize(path.toFile().getCanonicalPath(), Normalizer.Form.NFC);
        String perm = UsOSSafeUtils.runtimeTrimExec((String[])new String[]{STAT_CMD, canonicalPath});
        if (perm.matches("\\d+")) {
            int specialPermission = Integer.parseInt(perm) / 1000;
            Path fileName = path.getFileName();
            String dirName = Normalizer.normalize(Objects.toString(fileName, "null"), Normalizer.Form.NFC);
            String normalizePerm = Normalizer.normalize(perm, Normalizer.Form.NFC);
            if (SPECIAL_PERMISSIONS.contains(specialPermission)) {
                LOGGER.info("Detected valid special directory [{}] with permission {}!", (Object)dirName, (Object)normalizePerm);
                return true;
            }
            LOGGER.error("Detected invalid special directory [{}] with permission {}!", (Object)dirName, (Object)normalizePerm);
        }
        return false;
    }

    private static WrapAcl getWrapAcl(Path path, boolean isDirectory, StandardOpenOption ... standardOpenOptions) {
        Set options = Stream.of(standardOpenOptions).collect(Collectors.toSet());
        if (UsFileSafeUtils.isPosix()) {
            return new WrapAcl(path, options, isDirectory ? posixDirectoryPermissionsAttribute : posixFilePermissionsAttribute);
        }
        if (UsFileSafeUtils.isAcl()) {
            return new WrapAcl(path, options, aclFilePermissionsAttribute);
        }
        LOGGER.error("Failed to get warp acl!");
        throw new IllegalArgumentException("Failed to get warp acl for file " + path.getFileName());
    }

    private static <E extends Exception> IllegalArgumentException wrapper(String message, E exception) throws IllegalArgumentException {
        if (exception != null) {
            String exceptionMsg = ExceptionUtils.getRootCauseMessage(exception);
            String msg = MessageFormat.format("Wrapper Exception: {0} -> Cause By {1}", message, exceptionMsg);
            LOGGER.error(msg);
        }
        LOGGER.error(message);
        return new IllegalArgumentException(message);
    }

    private static void initPosixCommonAttributes() {
        try {
            posixRoot = LOOKUP_SERVICE.lookupPrincipalByName("root");
            posixUser = LOOKUP_SERVICE.lookupPrincipalByName(CURRENT_USER_NAME);
            posixFilePermissionsAttribute = PosixFilePermissions.asFileAttribute(new HashSet<PosixFilePermission>(Arrays.asList(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE)));
            posixDirectoryPermissionsAttribute = PosixFilePermissions.asFileAttribute(new HashSet<PosixFilePermission>(Arrays.asList(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE)));
        }
        catch (IOException ex) {
            LOGGER.error("Failed to lookup principal for root or user {} cause by {}!", (Object)CURRENT_USER_NAME, (Object)ex.getMessage());
        }
    }

    private static void initAclCommonAttributes() {
        try {
            aclUser = LOOKUP_SERVICE.lookupPrincipalByName(CURRENT_USER_NAME);
            aclAdministrators = LOOKUP_SERVICE.lookupPrincipalByGroupName("administrators");
            aclSystem = LOOKUP_SERVICE.lookupPrincipalByGroupName("system");
            final AclEntry entry = AclEntry.newBuilder().setType(AclEntryType.ALLOW).setPrincipal(aclUser).setPermissions(EnumSet.allOf(AclEntryPermission.class)).build();
            final AclEntry entryAdmin = AclEntry.newBuilder().setType(AclEntryType.ALLOW).setPrincipal(aclAdministrators).setPermissions(EnumSet.allOf(AclEntryPermission.class)).build();
            final AclEntry entrySystem = AclEntry.newBuilder().setType(AclEntryType.ALLOW).setPrincipal(aclSystem).setPermissions(EnumSet.allOf(AclEntryPermission.class)).build();
            aclFilePermissionsAttribute = new FileAttribute<List<AclEntry>>(){

                @Override
                public String name() {
                    return "acl:acl";
                }

                @Override
                public List<AclEntry> value() {
                    ArrayList<AclEntry> list = new ArrayList<AclEntry>();
                    list.add(entry);
                    list.add(entryAdmin);
                    list.add(entrySystem);
                    return list;
                }
            };
        }
        catch (IOException ex) {
            LOGGER.error("Failed to lookup principal for administrators or system or user {} cause by {}!", (Object)CURRENT_USER_NAME, (Object)ex.getMessage());
        }
    }

    static {
        if (UsFileSafeUtils.isPosix()) {
            UsFileSafeUtils.initPosixCommonAttributes();
        }
        if (UsFileSafeUtils.isAcl()) {
            UsFileSafeUtils.initAclCommonAttributes();
        }
    }

    private static class WrapAcl {
        private Path path;
        private Set<OpenOption> options;
        private FileAttribute<?> attr;

        private WrapAcl(Path path, Set<OpenOption> options, FileAttribute<?> attr) {
            this.path = path;
            this.options = options;
            this.attr = attr;
        }

        private Path getPath() {
            return this.path;
        }

        private Set<OpenOption> getOptions() {
            return this.options;
        }

        private FileAttribute<?> getAttr() {
            return this.attr;
        }
    }
}

