#!/bin/bash
###############################################
# check before dpc online upgrade, interactive mode
###############################################

CUR_CHEFCK_DIR="$( cd "$( dirname "$0"  )" && pwd )"
DPC_LOG_DIR="/var/log/dpc_script"
DPC_LOG_FILE="dpc.action.log"
CPU_FREE_MIN_LIMIT="10"     # 10%
MEMORY_FREE_MIN_LIMIT="8388608"   # 8G
ODC_CONTINUOUS_MEM_MIN=400  # 400M

function init_log()
{
    if [ ! -d "$DPC_LOG_DIR" ]; then
        mkdir -p $DPC_LOG_DIR
    fi

    if [ ! -f "$DPC_LOG_DIR/$DPC_LOG_FILE" ]; then
        touch "$DPC_LOG_DIR/$DPC_LOG_FILE"
    fi

    return
}

function LOG()
{
    if [ ! -L "$DPC_LOG_DIR/$DPC_LOG_FILE" ]; then
        echo "[`date "+%Y-%m-%d %T"`:$$][${FUNCNAME[1]}]$*" >> "$DPC_LOG_DIR/$DPC_LOG_FILE"
    fi
}

function LOG_ERR()
{
    if [ ! -L "$DPC_LOG_DIR/$DPC_LOG_FILE" ]; then
        echo "[`date "+%Y-%m-%d %T"`:$$][${FUNCNAME[1]}][ERR]$*" >> "$DPC_LOG_DIR/$DPC_LOG_FILE"
    fi
}


function calc_cpu_free_usage()
{
    proc_stat1=`head -n 1 /proc/stat`
    cpu_info1=($( echo $proc_stat1 | sed 's/cpu//'| awk '$1=$1'))
    cpu_idle1=$((${cpu_info1[3]}+${cpu_info1[4]}))
    for value in ${cpu_info1[*]}
    do
        cpu_total1=$(($cpu_total1+$value))
    done
    LOG "proc_stat1=$proc_stat1, cpu_total1=$cpu_total1, cpu_idle1=$cpu_idle1"


    sleep 2
    proc_stat2=`head -n 1 /proc/stat`
    cpu_info2=($(echo $proc_stat2 | sed 's/cpu//'| awk '$1=$1'))
    cpu_idle2=$((${cpu_info2[3]}+${cpu_info2[4]}))
    for value in ${cpu_info2[*]}
    do
        cpu_total2=$(($cpu_total2+$value))
    done
    LOG "proc_stat2=$proc_stat2, cpu_total2=$cpu_total2, cpu_idle2=$cpu_idle2"
    diff_idle=$((cpu_idle2 - cpu_idle1))
    diff_total=$((cpu_total2 - cpu_total1))
    LOG "diff_total=$diff_total, diff_idle=$diff_idle"
    diff_idle_usage=$((($diff_idle*100)/($diff_total)))
    echo $diff_idle_usage
}

function check_cpu()
{
    LOG "begin check cpu usage"

    local cpu_free=$(calc_cpu_free_usage)
    if [ "$cpu_free" -le "$CPU_FREE_MIN_LIMIT" ];then
        LOG "cpu free $cpu_free %, less than $CPU_FREE_MIN_LIMIT %."
        #由于获取的瞬时CPU占用率高于阈值, 导致检查不过, 增加重试机制
        sleep 2
        cpu_free=$(calc_cpu_free_usage)
        if [ "$cpu_free" -le "$CPU_FREE_MIN_LIMIT" ];then
            LOG "second check:cpu free $cpu_free %, less than $CPU_FREE_MIN_LIMIT %"
            return 1
        fi
    fi

    LOG "cpu free $cpu_free %, ok."
    return 0
}

function check_memory()
{
    LOG "begin check memory usage"
    local mem_free=$(cat /proc/meminfo | grep -w MemFree | awk '{print $2}')
    if [ "$mem_free" -le "$MEMORY_FREE_MIN_LIMIT" ];then
        LOG "memory free $mem_free KB, less than or equal to $MEMORY_FREE_MIN_LIMIT KB(8GB)."
        #由于获取的瞬时MEMORY占用率高于阈值, 导致检查不过, 增加重试机制
        sleep 2
        mem_free=$(cat /proc/meminfo | grep -w MemFree | awk '{print $2}')
        if [ "$mem_free" -le "$MEMORY_FREE_MIN_LIMIT" ];then
            LOG "second check:memory free $mem_free KB, less than or equal to $MEMORY_FREE_MIN_LIMIT KB(8GB)."
            return 1
        fi
    fi

    LOG "memory free $mem_free KB, ok."
    return 0
}

function check_free_mem_percent()
{
    local memfree=$(cat /proc/meminfo | grep -w MemFree | awk '{print $2}')
    local memtotal=$(cat /proc/meminfo | grep MemTotal | awk '{print $2}' | awk '{printf ("%.1f\n",$1*0.2)}')
    result=$(awk -v num1=$memfree -v num2=$memtotal 'BEGIN{print(num1>num2)?"0":"1"}')
    
    return $result
}

function check_memory_percent()
{
    check_free_mem_percent
    local rc=$?
    if [[ $rc -eq 0 ]]; then
        LOG "check_memory_percent success."
        return 0
    fi
    LOG_ERR "The current memory usage is higher than 80%."
    echo "The current memory usage is higher than 80%. If you perform an online upgrade, an online upgrade may fail. You are advised to perform an online upgrade after the memory usage decreases to 80%."
    return 1
}

function handle_input()
{
    local num=3
    local try=0

    while [ $try -le $num ]
    do
        read -p "To continue the online upgrade, enter (yes/y). To stop the upgrade, enter (no/n):" input
        case "$input" in
            "yes" | 'y' | "YES" | 'Y')
                return 0
                ;;  
            "no" | 'n' | "NO" | 'N')
                return 1
                ;;  
            *)  
                try=$((try + 1))
                [[ $try -eq 3 ]] && echo "Upgrade: Have exhausted maximum number of retries for serverce" && return 1
                echo "Wrong input,please try again."
                ;;  
        esac
    done
}

function check_cpu_usage()
{
    check_cpu
    local rc=$?
    [[ $rc -eq 0 ]] && return 0

    echo "Note: The current CPU usage is higher than 90%. If you perform an online upgrade, I/Os may not respond for a long time. You are advised to perform an online upgrade after the CPU usage decreases to 90%."

    handle_input
    return $?
}

function check_memory_usage()
{
    check_memory
    check_memory_res=$?
    check_memory_percent
    check_memory_percent_res=$?
    if [ $check_memory_res -eq 0 ] && [ $check_memory_percent_res -eq 0 ]; then
        return 0
    fi

    echo "Note: The remaining system memory percentage is less than or equal to 20% or the remaining memory is less than or equal to 8 GB. Therefore, an online upgrade may fail. You are advised to perform an online upgrade after the remaining system memory percentage is more than 20% and the remaining memory is more than 8GB."

    handle_input
    return $?
}

function main()
{
    LOG "start dpc interactive mode check"
    check_cpu_usage
    [[ $? -ne 0 ]] && return 1

    check_memory_usage
    [[ $? -ne 0 ]] && return 1
    
    LOG "end dpc interactive mode check"
    return 0
}

main
exit $?
