# -*- coding: UTF-8 -*-
import re

from cbb.frame.cli import cliUtil
from cbb.frame.base.exception import UnCheckException
from cbb.frame.base.config import (
    ARM_HIGH_END_NEW,
    DORADO_DEVS_V6_HIGH
)

# 默认漂移组
DEFAULT_FG = '0'
NO_SPH = "--"
RISK_SOFTWARE_VERSION = {
    "V500R007C60SPC200": NO_SPH,
    "V500R007C60SPC300": "V500R007C60SPH308",
    "V500R007C70": NO_SPH,
    "6.0.0": NO_SPH,
    "6.0.RC1": NO_SPH,
    "6.0.1": "SPH8",
}

KUNPENG_VERSION_LIST = [
    "V500R007C60SPC200 Kunpeng",
    "V500R007C60SPC300 Kunpeng",
    "V500R007C70 Kunpeng",
]


class CrossIntfBondCheck(object):
    """
    检查是否在跨卡绑定端口上配置了IPv6地址
    """

    def __init__(self, cli, lang, logger, p_version, p_model, bond_port, is_fru=False):
        self.cli = cli
        self.lang = lang
        self.logger = logger
        self.p_version_mem = p_version
        self.p_model = p_model
        self.filter_bond_port_list = bond_port
        self.is_fru = is_fru
        self.all_cli_ret = list()
        self.physic_port_err_bond_name = list()
        self.logical_port_err_logic_name = list()
        self.default_fg_err_port_list = list()
        self.fg_err_port_dict = dict()
        self.logic_port_dict = dict()
        self.vlan_info_dict = dict()
        self.fg_list = list()
        self.fg_info_dict = dict()
        self.fg_port_info = dict()

    def execute_check(self):
        if self.is_no_risk_version():
            return True

        bond_info_dict = self.get_bond_port_info()
        self.logger.logInfo(
            "bond_info_dict:{}".format(bond_info_dict))
        if not bond_info_dict:
            return True

        ipv6_host_port_list = self.get_port_ip_info()
        self.logger.logInfo("ipv6_host_port_list:{}".format(ipv6_host_port_list))
        self.check_physic_port(bond_info_dict, ipv6_host_port_list)
        if self.physic_port_err_bond_name:
            return False

        self.get_logical_port_info()
        self.logger.logInfo(
            "logic_port_dict:{}".format(self.logic_port_dict))
        if not self.logic_port_dict:
            return True

        self.get_vlan_info()
        self.logger.logInfo("vlan_info_dict:{}".format(self.vlan_info_dict))

        self.check_logic_port()
        if self.logical_port_err_logic_name:
            return False

        if self.is_fru:
            return True

        self.get_failover_group_info()
        self.logger.logInfo("fg_info_dict:{}".format(self.fg_info_dict))
        self.check_fg()

        return not any([self.default_fg_err_port_list, self.fg_err_port_dict])

    def check_fg(self):
        """
        6 如果步骤5查询到的"Failover Group ID"字段值为0，则检查不通过。
        否则使用获取到的"Failover Group ID"值执行步骤7，如果步骤7查询到的"Name"列表中，
        存在与跨卡绑定的"Name"相同的记录，则检查不通过，否则检查通过。
        :return:
        """
        for fg_id, fg_info in self.fg_info_dict.items():
            fg_err_port_list = []
            if fg_id == DEFAULT_FG:
                self.default_fg_err_port_list.extend(
                    self.fg_port_info.get(fg_id, [])
                )
                continue

            bond_dict = fg_info.get("bond_dict")
            err_bond_list = [
                bond_name
                for bond_name in bond_dict.keys()
                if bond_name in self.bond_name_list
            ]
            fg_err_port_list.extend(err_bond_list)
            vlan_dict = fg_info.get("vlan_dict")
            err_vlan_list = [
                bond_name
                for bond_name in vlan_dict.values()
                if bond_name in self.bond_name_list
            ]
            fg_err_port_list.extend(err_vlan_list)
            if fg_err_port_list:
                self.fg_err_port_dict[fg_id] = {
                    "bond_names": ",".join(list(set(fg_err_port_list))),
                    "logic_ports": ",".join(list(set(self.fg_port_info.get(fg_id, []))))
                }

    def check_logic_port(self):
        """
        5 若步骤5查询到配置了IPv6地址的逻辑端口，检查配置了IPv6地址的逻辑端口，
        如果Home Port ID与步骤3中，跨卡绑定的"Name"相同，则检查不通过。
        否则，如果Home Port ID存在于步骤6中查询到的"Name"中，并且该"Name"对应的"Port ID"与步骤3中，
        跨卡绑定的"Name"相同，则检查不通过。否则继续检查。
        :return:
        """
        for port_name, port_info in self.logic_port_dict.items():
            home_port_id = port_info.get("home_port_id")
            current_port_id = port_info.get("current_port_id")
            if any(
                [
                    home_port_id in self.bond_name_list,
                    home_port_id in self.vlan_info_dict
                    and self.vlan_info_dict.get(home_port_id)
                    in self.bond_name_list,
                    self.is_fru and current_port_id in self.bond_name_list,
                    self.is_fru and current_port_id in self.vlan_info_dict
                    and self.vlan_info_dict.get(current_port_id)
                    in self.bond_name_list,
                ]
            ):
                self.logical_port_err_logic_name.append(port_name)

    def check_physic_port(self, bond_info_dict, ipv6_host_port_list):
        for bond_id, info_dict in bond_info_dict.items():
            tmp_res_list = []
            bond_port_list = info_dict.get("bond_port_list", [])
            bond_name = info_dict.get("bond_name")
            tmp_res_list.extend(
                [hp for hp in ipv6_host_port_list if hp in bond_port_list]
            )
            if tmp_res_list:
                self.physic_port_err_bond_name.append(bond_name)

    def is_no_risk_version(self):
        """
        系统软件版本不为 V500R007C60SPC200 Kunpeng、V500R007C60SPC300 Kunpeng、
        V500R007C70 Kunpeng、6.0.0、6.0.RC1、6.0.1，则检查结果为通过，否则继续检查；
        系统软件版本和热补丁关系如下，则检查结果为通过，否则继续检查；
        V500R007C60SPC300 Kunpeng：安装了SPH308或更高版本的补丁；
        Dorado V6 6.0.1：安装了SPH8或更高版本的补丁。
        :return: False 风险， True 非风险版本
        """
        (
            __,
            self.p_version,
            self.p_patch,
            cli_ret,
        ) = cliUtil.get_system_version_with_ret(self.cli, self.lang)
        self.all_cli_ret.append(cli_ret)
        self.is_sugges_patch = True
        target_sph_ver = RISK_SOFTWARE_VERSION.get(self.p_version)
        self.suggess_patch_version = target_sph_ver
        if any([
            self.p_model in ARM_HIGH_END_NEW,
            self.p_model in DORADO_DEVS_V6_HIGH,
        ]):
            return True
        if self.p_version.startswith("V5") and \
                self.p_version_mem not in KUNPENG_VERSION_LIST:
            return True
        if self.p_version not in RISK_SOFTWARE_VERSION:
            return True
        if target_sph_ver == NO_SPH:
            self.is_sugges_patch = False
            return False

        if not self.p_patch or self.p_patch == '--':
            return False
        sph_ver = self.p_patch
        sph_ver_res = re.compile("SPH(\d+)").search(self.p_patch)
        if sph_ver_res:
            sph_ver = int(sph_ver_res.group(1))
        target_res = re.compile("SPH(\d+)").search(target_sph_ver)
        if target_res:
            target_sph_ver = int(target_res.group(1))
        return sph_ver >= target_sph_ver

    def get_bond_port_info(self):
        """
        执行命令：show bond_port，获取每个绑定口的"name"和"Port ID List"；
        :return:
        """
        self.bond_name_list = []
        cmd = "show bond_port"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )

        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        ret_lines_list = cliUtil.getHorizontalCliRet(ret)
        bond_info_dict = dict()
        for line in ret_lines_list:
            bond_id = line.get("ID")
            bond_name = line.get("Name")
            bond_port_list_str = line.get("Port ID List", "")
            if bond_port_list_str == "--":
                continue
            bond_port_list = bond_port_list_str.split(",")
            if self.is_same_card(bond_port_list):
                continue

            # 备件更换只检查对应端口
            if self.is_fru:
                bond_port_list = self.filter_fru_port(bond_port_list)
            if not bond_port_list:
                continue

            bond_info_dict[bond_id] = {
                "bond_name": bond_name,
                "bond_port_list": bond_port_list,
            }
            self.bond_name_list.append(bond_name)
        return bond_info_dict

    def filter_fru_port(self, bond_port_list):
        """
        过滤FRU端口
        :param bond_port_list:
        :return:
        """
        tmp_bond_list = []
        for filter_bond_port in self.filter_bond_port_list:
            for bond_port in bond_port_list:
                if bond_port.startswith(filter_bond_port):
                    tmp_bond_list.append(bond_port)
        return list(set(tmp_bond_list))

    def get_vlan_info(self):
        """
        执行命令: show vlan general 查询VLAN信息。
        :return:
        """
        cmd = "show vlan general"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )

        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        ret_lines_list = cliUtil.getHorizontalCliRet(ret)
        for line in ret_lines_list:
            vlan_name = line.get("Name")
            vlan_port_id = line.get("Port ID", "")
            self.vlan_info_dict[vlan_name] = vlan_port_id

    def is_same_card(self, port_list):
        """
        是否是同一张卡
        :param port_list:
        :return:
        """
        res = [".".join(port_id.split(".")[:-1]) for port_id in port_list]
        return len(set(res)) == 1

    def get_port_ip_info(self):
        """
        执行命令：show bond_port，获取每个绑定口的"name"和"Port ID List"；
        :return:
        """
        cmd = "show port ip"
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )

        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)

        tmp_ret = ret[ret.find("Host Port:"): ret.find("Management Port:")]
        ret_lines_list = cliUtil.getHorizontalCliRet(tmp_ret)

        ipv6_port_list = list()
        for line in ret_lines_list:
            port_id = line.get("Port ID")
            ip_addr = line.get("IPv6 Address")
            port_type = line.get("Type")
            if self.is_not_ip_v6(ip_addr) or port_type != "Host Port":
                continue
            ipv6_port_list.append(port_id)

        return ipv6_port_list

    def is_not_ip_v6(self, ip_addr):
        """
        判断是否IPV6
        :param ip_addr:
        :return:
        """
        return any(
            [
                ip_addr == "--",
                ":" not in ip_addr
            ]
        )

    def get_logical_port_info(self):
        """
        获取逻辑端口的信息
        :return:
        """
        cmd = (
            "show logical_port general |filterColumn include "
            "columnList=Logical\sPort\sName,Failover\sGroup\sID,"
            "Home\sPort\sID,IPv6\sAddress,Current\sPort\sID"
        )
        flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
            self.cli, cmd, True, self.lang
        )
        self.all_cli_ret.append(ret)
        if flag is not True:
            raise UnCheckException(err_msg)
        ret_lines_list = cliUtil.getHorizontalCliRet(ret)
        for line in ret_lines_list:
            port_name = line.get("Logical Port Name")
            home_port_id = line.get("Home Port ID")
            current_port_id = line.get("Current Port ID")
            fg_id = line.get("Failover Group ID")
            ip_info = line.get("IPv6 Address")
            if self.is_not_ip_v6(ip_info):
                continue
            self.logic_port_dict[port_name] = {
                "home_port_id": home_port_id,
                "current_port_id": current_port_id,
                "fg_id": fg_id,
            }
            if fg_id not in self.fg_list:
                self.fg_list.append(fg_id)

            tmp_list = self.fg_port_info.get(fg_id, [])
            tmp_list.append(port_name)
            self.fg_port_info[fg_id] = tmp_list

    def get_failover_group_info(self):
        """
        查询指定漂移组
        :return:
        """
        for fg_id in self.fg_list:
            if fg_id == DEFAULT_FG:
                self.fg_info_dict[fg_id] = dict()
                continue
            bond_dict = dict()
            vlan_dict = dict()
            cmd = " show failover_group member failover_group_id={}".format(
                fg_id
            )
            flag, ret, err_msg = cliUtil.excuteCmdInCliMode(
                self.cli, cmd, True, self.lang
            )
            self.all_cli_ret.append(ret)
            if flag is not True:
                raise UnCheckException(err_msg)

            bond_ret = ret[ret.find("Bond Port:"): ret.find("VLAN:")]
            vlan_ret = ret[ret.find("VLAN:"):]
            bond_ret_list = cliUtil.getHorizontalCliRet(bond_ret)
            vlan_ret_list = cliUtil.getHorizontalCliRet(vlan_ret)
            for line in bond_ret_list:
                bond_name = line.get("Name")
                port_id_list = line.get("Port ID List")
                bond_dict[bond_name] = port_id_list
            for line in vlan_ret_list:
                bond_name = line.get("Name")
                port_id = line.get("Port ID")
                vlan_dict[bond_name] = port_id
            self.fg_info_dict[fg_id] = {
                "bond_dict": bond_dict,
                "vlan_dict": vlan_dict,
            }
