#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

import re
from collections import namedtuple

MaxDiskNumPerEngine = namedtuple('MaxDiskNumPerEngine',
                                 field_names=[
                                     'product_model_and_ver_reg_pattern',
                                     'max_disk_num'])

NO_LIMIT_MAX_DISK_NUM_PER_ENGINE = -1

#  不同版本和产品型号单引擎支持的最大成员盘个数配置。
DISK_NUM_PER_ENGINE_CONF_LIST = [
    MaxDiskNumPerEngine(re.compile(r'^5100 V3_.*$', re.I), 400),

    MaxDiskNumPerEngine(
        re.compile(r'^(5300 V3|2600F? V3( Enhanced| for Video)?)_.*$', re.I),
        500),

    MaxDiskNumPerEngine(re.compile(r'^(5500F? V3|2800 V3|2800 V5)_.*$', re.I),
                        750),

    MaxDiskNumPerEngine(
        re.compile(r'^5600F? V3_(V300R00[1|2|3].*|V300R006C00.*|V300R006C10)$',
                   re.I), 1000),
    MaxDiskNumPerEngine(
        re.compile(r'^5600F? V3_V300R006C(10SPC.*|[2|3|5|6]0(SPC.*)?)$', re.I),
        1200),

    MaxDiskNumPerEngine(
        re.compile(r'^5800 V3_(V300R00[1|2].*|V300R003C00)$', re.I), 1250),

    MaxDiskNumPerEngine(re.compile(
        r'^5800 V3_V300(R003C00SPC.*|R003C[1|2]0(SPC.*)?|R006C[0|1|2|5]0.*)$',
        re.I), 1500),

    MaxDiskNumPerEngine(
        re.compile(r'^5800F V3_V300R006C[0|1|2|5|6]0(SPC.*)?$', re.I),
        1500),

    MaxDiskNumPerEngine(
        re.compile(r'^6800 V3_(V300R001.*|V300R002C10)$', re.I),
        1500),

    MaxDiskNumPerEngine(
        re.compile(r'^6800 V3_(V300R002C10SPC.*|V300R00[3|6].*)$', re.I),
        3200),
    MaxDiskNumPerEngine(
        re.compile(r'^6800F V3_V300R006C[0|1|2|5|6]0(SPC.*)?$', re.I),
        3200),

    MaxDiskNumPerEngine(re.compile(r'^6900 V3_V300R001C[1|2]0.*$', re.I),
                        2000),

    MaxDiskNumPerEngine(
        re.compile(r'^(18500 V3|18800F? V3|18[5|8]00 V5)_V[3|5].*$', re.I),
        2400),
    MaxDiskNumPerEngine(re.compile(r'^2100 V3_.*$', re.I), 200),
    MaxDiskNumPerEngine(re.compile(r'^2200 V3( Enhanced)?_.*$', re.I), 300),

    MaxDiskNumPerEngine(
        re.compile(r'^(5300F? V5|5500F V5)_V500R007C[0|1]0.*$', re.I), 500),
    MaxDiskNumPerEngine(
        re.compile(r'^(5300 V5|5500F V5)_V500R007C[2|3|5]0.*$', re.I), 1000),

    MaxDiskNumPerEngine(
        re.compile(r'^5300 V5 Enhanced_V500R007C[0|1|3|5]0.*$', re.I), 1000),

    MaxDiskNumPerEngine(re.compile(r'^5300F V5_V500R007C[2|3|5]0.*$', re.I),
                        750),

    MaxDiskNumPerEngine(
        re.compile(r'^5500 V5( Elite)?_V500R007C[0|1]0.*$', re.I), 750),
    MaxDiskNumPerEngine(
        re.compile(r'^5500 V5( Elite)?_V500R007C[2|3|5]0.*$', re.I), 1200),

    MaxDiskNumPerEngine(re.compile(r'^5600 V5_V500R007C[0|1]0.*$', re.I),
                        1200),
    MaxDiskNumPerEngine(re.compile(r'^5600 V5_V500R007C[2|3|5]0.*$', re.I),
                        1600),

    MaxDiskNumPerEngine(re.compile(r'^5600F V5_V500R007C[0|1]0.*$', re.I),
                        1000),
    MaxDiskNumPerEngine(re.compile(r'^5600F V5_V500R007C[2|3|5]0.*$', re.I),
                        1200),

    MaxDiskNumPerEngine(
        re.compile(r'^(5800|6800F) V5_V500R007C[0|1]0.*$', re.I),
        1500),
    MaxDiskNumPerEngine(
        re.compile(r'^(5800|6800F) V5_V500R007C[2|3|5]0.*$', re.I), 2000),

    MaxDiskNumPerEngine(re.compile(r'^5800F V5_V500R007C[0|1]0.*$', re.I),
                        1200),
    MaxDiskNumPerEngine(re.compile(r'^5800F V5_V500R007C[2|3|5]0.*$', re.I),
                        1500),

    MaxDiskNumPerEngine(re.compile(r'^(6800|18800F) V5_.*$', re.I), 3200),

    # Dorado 配置："产品内部型号_产品版本"对应的正则表达式。
    MaxDiskNumPerEngine(re.compile(r'^Dorado3000_V3_.*$', re.I), 100),

    # Dorado 3000 >=V3R2C10 support internal version
    # Dorado5000_V3 SAS V300R001C21 及之后版本支持内部型号查询,不支持的版本需转换.
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3(_SAS)?_V300R001C(01|21|30).*$', re.I),
        100),
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3(_SAS)?_V300R002C[0|1|2]0.*$', re.I), 100),

    # Dorado5000 NVMe
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3_NVMe_V300R001C00.*$', re.I),
        100),
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3_NVMe_V300R001C(01|20|21|30).*$', re.I),
        25),
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3_NVMe_V300R002C00.*$', re.I),
        25),
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado5000_V3_NVMe_V300R002C[1|2]0.*$', re.I), 50),

    # Dorado6000 V3 SAS
    MaxDiskNumPerEngine(re.compile(r'^Dorado6000_V3_V300R001C00.*$', re.I),
                        200),
    MaxDiskNumPerEngine(re.compile(r'^Dorado6000_V3_V300R001C01.*$', re.I),
                        100),

    MaxDiskNumPerEngine(
        re.compile(r'^Dorado6000_V3_V300R001C(20|21|30).*$', re.I), 300),
    MaxDiskNumPerEngine(
        re.compile(r'^Dorado6000_V3_V300R002C[0|1|2]0.*$', re.I),
        300),

    MaxDiskNumPerEngine(re.compile(r'^Dorado18000N?_V3_.*$', re.I), 200)
]


def get_max_disk_num_per_engine(product_model, version):
    """获取单引擎（逻辑引擎）的支持的最大硬盘个数。

    :param product_model: 产品型号，Dorado产品为内部型号
    :param version: 产品完整的版本。
    :return: 单个逻辑引擎支持的最大硬盘个数。
    """
    model_and_version = product_model + '_' + version
    for disk_num_per_engine in DISK_NUM_PER_ENGINE_CONF_LIST:
        model_and_ver_pattern = \
            disk_num_per_engine.product_model_and_ver_reg_pattern
        if model_and_ver_pattern.match(model_and_version):
            return disk_num_per_engine.max_disk_num
    else:
        return NO_LIMIT_MAX_DISK_NUM_PER_ENGINE
