#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

import itertools
import traceback
from collections import OrderedDict
from collections import namedtuple

from cbb.frame.cli.cliUtil import excuteCmdInCliMode
from cbb.frame.cli.cliUtil import getMsg
from cbb.frame.cli.cliUtil import getHorizontalNostandardCliRet

CommandResult = namedtuple('CommandResult',
                           field_names=['cmd_ret_flag', 'cmd_raw_output',
                                        'cmd_handled_output', 'err_msg'])

Disk = namedtuple('Disk',
                  field_names=['id', 'physic_engine_id', 'logic_engine_id'])


def query_member_disk_id_list(cli, lang, logger):
    """查询系统成员盘的列表

    :param cli: cli对象
    :param lang: 语言环境
    :param logger:日志对象
    :return: 成员盘的命名元组对象列表
    """
    try:
        cmd = "show disk general|filterRow column=Role predict=match value=" \
              "Member\sDisk |filterColumn include columnList=ID"
        cmd_flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd,
                                                        isHasLog=True,
                                                        lang=lang)

        if cmd_flag is not True:
            return CommandResult(cmd_ret_flag=cmd_flag, cmd_raw_output=cli_ret,
                                 cmd_handled_output=[], err_msg=err_msg)

        member_disk_id_list = []
        for cli_line in cli_ret.splitlines():
            strip_cli = cli_line.strip()
            if strip_cli.startswith('DAE') or strip_cli.startswith('CTE'):
                member_disk_id_list.append(strip_cli)

        member_disk_list = [Disk(id=disk_id, physic_engine_id=int(disk_id[3]),
                                 logic_engine_id=int(
                                     disk_id[3]) if 'CTE' in disk_id else int(
                                     disk_id[4], 16) // 8)
                            for disk_id in member_disk_id_list]
        return CommandResult(cmd_ret_flag=True, cmd_raw_output=cli_ret,
                             cmd_handled_output=member_disk_list,
                             err_msg=err_msg)
    except Exception as e:
        logger.logError(e)
        logger.logError(traceback.format_exc())
        err_msg = getMsg(lang, "cannot.get.sys.info")
        return CommandResult(cmd_ret_flag=False, cmd_raw_output='',
                             cmd_handled_output=[],
                             err_msg=err_msg)


def get_existing_member_disk_num_per_engine(member_disks,
                                            according_logic_engine):
    """获取系统各个引擎上的成员盘数量。

    :param member_disks: 成员盘命名元组对象列表
    :param according_logic_engine:  是否按照逻辑引擎计算
    :return: 各个（逻辑/物理）引擎对应的成员盘个数的字典。
    """
    member_disks.sort(key=lambda disk:
                      disk.physic_engine_id if not according_logic_engine
                      else (disk.physic_engine_id, disk.logic_engine_id))

    member_disks.sort()
    grouped_disks = itertools.groupby(member_disks,
                                      key=lambda
                                      disk: disk.physic_engine_id
                                      if not according_logic_engine else (
                                          disk.physic_engine_id,
                                          disk.logic_engine_id))

    engine_and_disk_count_dict = OrderedDict()
    for engine, disks in grouped_disks:
        if not according_logic_engine:
            engine_k = 'CTE{physic_engine_id}'.format(physic_engine_id=engine)
        else:
            logic_engine_id = engine[1]
            engine_k = '{physic_engine_id}A/{physic_engine_id}B'.format(
                physic_engine_id=engine[0]) if logic_engine_id == 0 else \
                '{physic_engine_id}C/{physic_engine_id}D'.format(
                physic_engine_id=engine[0])
        engine_and_disk_count_dict[engine_k] = len(list(disks))

    return engine_and_disk_count_dict


def get_disk_general(cli, lang, logger):
    """获取磁盘通用消息

    :param cli:
    :param lang:
    :param logger:
    :return:
    """
    cmd = "show disk general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(
        cli, cmd, isHasLog=True, lang=lang)
    if flag is not True:
        return False, [], cli_ret, err_msg

    disk_general_list = getHorizontalNostandardCliRet(cli_ret)

    return True, disk_general_list, cli_ret, err_msg
