#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.
import itertools

from cbb.frame.cli import cliUtil
from cbb.common.conf import productConfig


def get_all_engines(cli, lang, according_logic_engine=False):
    """获取所有物理引擎或逻辑引擎。
    物理引擎：[CTE0, CTE1, ...CTEn]
    逻辑引擎：['0A/0B', '0C/OD', '1A/1B', '1C/1D' ...]
    :param cli: cli连接对象
    :param lang:　语言环境
    :param according_logic_engine: 是否区分逻辑引擎（6800 V3 、Dorado 18000 V3
            区分逻辑引擎.
    :return: （物理引擎或者逻辑）引擎列表
    """
    cli_ret_list = []
    engines = []
    cmd_exe_succ, cli_raw_output, engine_err_msg, all_eng_id_list \
        = cliUtil.getEngineIdList(cli, lang)
    cli_ret_list.append(cli_raw_output)

    if cmd_exe_succ is not True:
        return False, cli_ret_list, engine_err_msg, engines

    if not according_logic_engine:
        engines = all_eng_id_list
        return True, cli_ret_list, '', engines

    cmd_exe_succ, cli_ret_or_all_ctrl_id_list, engine_err_msg \
        = cliUtil.getControllerIdList(cli, lang)
    if cmd_exe_succ is not True:
        cli_ret_list.append(cli_ret_or_all_ctrl_id_list)
        return False, cli_ret_list, engine_err_msg, engines

    for k, grouped_engines in itertools.groupby(cli_ret_or_all_ctrl_id_list,
                                                lambda ctrl_id: (
                                                        ctrl_id[0],
                                                        '0' if ctrl_id[1] in [
                                                            'A',
                                                            'B'] else '1')):
        engines.append('/'.join(grouped_engines))

    return True, cli_ret_list, '', engines


def get_all_eng_id_list(cli, lang, according_to_logic=False):
    """获取系统中所有引擎ID列表（数字ID，引擎0,1,2,3）

    :param cli:
    :param lang:
    :param according_to_logic: 是否区分逻辑引擎
    :return:
    """
    if not according_to_logic:
        # 查询引擎信息
        flag, _, _, eng_id_list = cliUtil.getEngineIdList(cli, lang)
        return flag, [eng[3] for eng in eng_id_list]
    # 查询控制器ID
    flag, ctrl_id_list, _ = cliUtil.getControllerIdList(cli, lang)
    # 逻辑引擎集合
    logic_eng_set = set()
    for ctrl_id in ctrl_id_list:
        logic_eng_set.add(get_eng_id_by_ctrl_id(ctrl_id, True))
    return flag, list(logic_eng_set)


def get_eng_id_by_disk_id(disk_id, according_to_logic=False):
    """根据硬盘ID获取归属引擎ID

    :param disk_id: 硬盘ID，CTE0.1 / DAE000.1
    :param according_to_logic: 是否区分逻辑引擎
    :return:
    """
    if len(disk_id) < 5:
        return ""
    if according_to_logic:
        phy_eng = int(disk_id[3])
        return str(phy_eng * 2 + 1) if disk_id[4] >= "8" else str(phy_eng * 2)
    else:
        return disk_id[3]


def get_eng_id_by_ctrl_id(ctrl_id, according_to_logic=False):
    """根据控制器ID获取归属引擎ID

    :param ctrl_id: 控制器ID ，例：0A/0B/1A/AB
    :param according_to_logic: 是否区分逻辑引擎
    :return:
    """
    phy_eng = int(ctrl_id[0])
    node = ctrl_id[-1]
    if according_to_logic:
        return str(phy_eng * 2 + productConfig.NODE_MAP.get(node, 0))
    else:
        return str(phy_eng)
