# -*- coding: UTF-8 -*-
import json
import os
import traceback

from com.huawei.ism.tool.framework.platform.exception import ToolException
from com.huawei.ism.tool.obase.entity import DevNode
from com.huawei.ism.tool.framework.pubservice.entity import DevNodeCommon
from com.huawei.ism.tool.obase.exception import PwdException
from java.util import Map
from java.lang import Exception as JException
from com.huawei.tool.framwork.util.func import LocalHostUtil
from com.huawei.ism.tool.obase.entity import EntityUtils

from cbb.business.operate.fru.common import BaseFactory
from cbb.frame.base import config
from cbb.frame.base import jsonUtil
from cbb.frame.base.config import UserOpDataFields

from cbb.frame.base.exception import CliCmdException
from cbb.frame.cli.cliFactory import CliConnection
from cbb.frame.rest import restData
from cbb.frame.rest.restFactory import RestConnection
from cbb.frame.sftp.sftpFactory import SftpConnection
from cbb.frame.tlv.tlvFactory import TlvConnection
from cbb.frame.tlv.tlvFactory import TlvConnectionSVP
from cbb.frame.base.funcUtils import wrap_all_exception_logged


SCRIPT_DEF_CONTEXT = "SCRIPT_DEF_CONTEXT"
RESOURCE_DEF_CONTEXT = "RESOURCE_DEF_CONTEXT"

CLI_CONNECTION = "CLI_CONNECTION"
TLV_CONNECTION = "TLV_CONNECTION"
SVP_CONNECTION = "SVP_CONNECTION"
REST_CONNECTION = "REST_CONNECTION"
SFTP_CONNECTION = "SFTP_CONNECTION"
SVP_REST_CONNECTION = "SVP_REST_CONNECTION"
SSH_CONNECTOR = "SSH_CONNECTOR"

UPGRADE_ERROR_CODE = "1077949021"
# 网管消息中命令字未注册，非法命令字
COMMAMD_NOT_REGIST = ["1077949002", "50331651"]

COMMUNICATION_FAIL_ERROR_CODE = [
    "16797697",  # 与设备断开连接
    "1073949185",  # 与设备通信异常，请检查网络连接或设备状态是否正常
    "16797698",  # 与设备通信异常
]
CLI_CAN_NOT_EXECUTE = ["1077936891"]  # 命令执行失败
ENGINE_CAN_NOT_MONITOR = ["1077949083"]  # 引擎无法监控

EXP_FLOW_STATUS_KEY = "EXP_FLOW_STATUS_KEY"  # 扩容工具状态
EXP_FLOW_STATUS_CHANGE_DATA = "EXP_FLOW_STATUS_CHANGE_DATA"  # 扩容工具状态：已修改数据
EXP_FLOW_STATUS_REBOOT_NEW_CONTROLLER = "EXP_FLOW_STATUS_REBOOT_NEW_CONTROLLER"  # 扩容工具状态：重启待扩控
EXP_FLOW_STATUS_JOIN_SYS_SUCCESS = "EXP_FLOW_STATUS_JOIN_SYS_SUCCESS"  # 扩容工具状态: 新控制已加入集群

SINGLE_CTRL_KEY = "SINGLE_CTRL_KEY"  # 单控标志

ERRCODE_MESSAGES_DICT = {
    "16797697":
        {
            "errMsg_zh": u"与设备通信异常。",
            "errMsg_en": "Communicating with the device failed.",
            "suggestion_zh": u"请检查网络连接或设备状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en": "Please check that the network connection or the system is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "1073949185":
        {
            "errMsg_zh": u"与设备通信异常。",
            "errMsg_en": "Communicating with the device failed.",
            "suggestion_zh": u"请检查网络连接或设备状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en": "Please check that the network connection or the system is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "1077936891":
        {
            "errMsg_zh": u"命令执行失败。",
            "errMsg_en": "Command execute failed.",
            "suggestion_zh": u"",
            "suggestion_en": "",
        },
    "1073949187":
        {
            "errMsg_zh": u"sftp传输失败。",
            "errMsg_en": "Failed to download file by sftp.",
            "suggestion_zh": u"请在网络和设备状态正常时重试。",
            "suggestion_en": "Retry when the network and device is normal.",
        },
    "DO_NOT_HAS_SUPER_ADMIN_RIGHTS":
        {
            "errMsg_zh": u"连接到设备失败，请检查网络及设备状态是否正常，登录用户是否拥有超级管理员权限，登录信息是否正确，以及帐号是否被锁定。",
            "errMsg_en": "Failed to connect the device. Please check whether the status of the network and device is normal, whether the login user has the super administer permission, whether login information is correct, and whether the login account is locked.",
            "suggestion_zh": u"",
            "suggestion_en": "",
        },
    "9999900001":
        {
            "errMsg_zh": u"命令执行失败。",
            "errMsg_en": "Command execute failed.",
            "suggestion_zh": u"",
            "suggestion_en": "",
        },
}
def getLogger(context):
    '''
    @summary: 获取日志打印对象
    @param context: 上下文对象
    '''
    return context.get("logger")

def getLang(context):
    '''
    @summary: 获取语言
    @param context: 上下文对象
    '''
    for key in ["lang", "lan", "language"]:
        lang = context.get(key)
        if lang is not None:
            return lang
    # 默认为英文
    return "en"


def getMsg(lang, msg, msgDict=ERRCODE_MESSAGES_DICT):
    '''
    @summary: 获取错误信息及建议
    @param context: 上下文对象
    '''
    errMsgDict = msgDict.get(msg, {})
    errMsg = errMsgDict.get("errMsg_%s" % lang, "--")
    suggestion = errMsgDict.get("suggestion_%s" % lang, "")

    return (errMsg, suggestion)


def getScriptDefDict(context):
    '''
    @summary: 获取上下文自定义字典
    @param context: 上下文对象
    '''
    if isinstance(context, dict):
        if not context.has_key(SCRIPT_DEF_CONTEXT):
            context.setdefault(SCRIPT_DEF_CONTEXT, {})
    else:
        if not context.containsKey(SCRIPT_DEF_CONTEXT):
            context.put(SCRIPT_DEF_CONTEXT, {})
    return context.get(SCRIPT_DEF_CONTEXT)


def getResourceDict(context):
    scriptDefDict = getScriptDefDict(context)
    if not scriptDefDict.has_key(RESOURCE_DEF_CONTEXT):
        resourceFilePath = getResourecePath(context)
        with open(resourceFilePath, 'r') as resource_file:
            contents = resource_file.read().splitlines()
        resourceDict = {}
        for line in contents:
            fields = line.split("=")
            if len(fields) < 2:
                continue
            key = fields[0].strip()
            value = "=".join(fields[1:]).strip()
            resourceDict.setdefault(key, value)
        scriptDefDict.setdefault(RESOURCE_DEF_CONTEXT, resourceDict)

    return scriptDefDict.get(RESOURCE_DEF_CONTEXT)


def setItem(context, key, value):
    '''
    @summary: 往自定义字典中添加项
    @param context: 上下文对象
    @param key: 添加项的键
    @param value: 添加项的值
    '''
    scriptDefDict = getScriptDefDict(context)
    scriptDefDict[key] = value

    if isinstance(context, dict):
        context[SCRIPT_DEF_CONTEXT] = scriptDefDict.copy()
    else:
        context.put(SCRIPT_DEF_CONTEXT, scriptDefDict.copy())
    return


def removeItem(context, key):
    '''
    @summary: 删除自定义字典中的项
    @param context: 上下文对象
    @param key: 删除项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    if scriptDefDict.has_key(key):
        scriptDefDict.pop(key)
    return


def getItem(context, key, defaultVal=None):
    '''
    @summary: 获取自定义字典中的项
    @param context: 上下文对象
    @param defaultVal: 默认值
    @param key: 项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    return scriptDefDict.get(key, defaultVal)


def getConnectorFactoryObj(context):
    '''
    @summary: 获取上下文对象中创建连接的对象
    @param context: 上下文对象
    '''
    protocalContext = context.get("protocalContext")
    if protocalContext == None:
        protocalContext = context
    factory = protocalContext.get("connectorFactory")
    if factory:
        return factory
    return context.get("sshFactory")


def getDevObj(context):
    """获取上下文对象中dev对象

    :param context: 上下文对象
    :return:
    """
    devObj = context.get("dev")
    if isinstance(devObj, Map):
        return devObj
    elif isinstance(devObj, DevNode):
        return context.get('newDev')
    else:
        return context.get('newDev')


def isOem(context):
    '''
    @summary: 获取上下文对象中isOem对象
    @param context: 上下文对象
    '''
    return context.get("isOem")


def isSingleCtrl(context):
    '''
    @summary: 获取上下文对象中单控状态对象
    @param context: 上下文对象
    '''
    return getItem(context, SINGLE_CTRL_KEY, False)


def setSingleCtrl(context, status=True):
    '''
    @summary: 将单控状态设置到上下文中
    @param context: 上下文对象
    '''
    return setItem(context, SINGLE_CTRL_KEY, status)


def getOemProductModel(context):
    '''
    @summary: 获取上下文对象中oemModel对象
    @param context: 上下文对象
    '''
    return context.get("oemModel")


def initCliConnection(context):
    '''
    @summary: 初始化cli连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)

    ip = devObj.get("ip")
    sshPort = int(devObj.get("sshPort"))

    conn = CliConnection(connectorFactoryObj, ip, sshPort, getItem(context, SSH_CONNECTOR), logger=context.get("logger"))
    conn.create(devObj.get("user"), devObj.get("pawd"))
    setItem(context, SSH_CONNECTOR, conn.sshConnector)

    setItem(context, CLI_CONNECTION, conn)
    return


def getCli(context):
    """获取cli连接，支持跳转
    conn缓存的conn是一个单例，多个设备使用的是同一个conn
    2个设备交叉执行命令，可能存在问题
    :param context: 上下文对象
    :return:
    """
    conn = getItem(context, CLI_CONNECTION)
    if conn is None:
        devObj = getDevObj(context)
        ip = devObj.get("ip")
        createCliConnection(context, ip)
        conn = getItem(context, CLI_CONNECTION)

    base_dev = get_base_dev(context)
    cli = conn.getCli(base_dev)
    return cli


def check_is_svp_and_return_cli(func):
    """
    判断是否是SVP模式，是则返回SVP cli，否则返回正常cli
    :param func:
    :return:
    """
    from cbb.frame.base import baseUtil
    from cbb.frame.cli.cli_con_mgr import get_conn_for_svp

    def inner(*params):
        context = params[0]
        logger = context.get("logger")
        try:
            if baseUtil.has_svp_module(context):
                logger.info("check_is_svp_and_return_cli get_ctrl_cli has svp.")
                return get_conn_for_svp(context, cli=None)
            else:
                return func(*params)
        except Exception as e:
            logger.error('check_is_svp_and_return_cli failed. {}'.format(e))
            return func(*params)

    return inner


@check_is_svp_and_return_cli
def get_new_cli(context):
    """获取cli连接，支持以设备IP+sn作为缓存

    :param context: 上下文对象
    :return:
    """
    devObj = getDevObj(context)
    logger = context.get("logger")
    ip = devObj.get("ip")
    sn = devObj.get("sn")
    conn_key = CLI_CONNECTION + ip + sn
    conn = getItem(context, conn_key)
    logger.info("get_new_cli get cli ip={}".format(ip))
    if conn is None:
        logger.info("conn is None")
        conn = createCliConnection(context, ip)
        setItem(context, conn_key, conn)
    else:
        conn = CliConnection.get_available_con(conn, logger)
    return conn


def getCliCommon(context):
    """
    获取上下文中的cli对象
    :param context: 上下文
    :return: cli对象
    """
    protocalContext = context.get("protocalContext")
    if protocalContext and "SSH" in protocalContext:
        # 升级工具
        return protocalContext.get("SSH")
    # 评估、补丁工具
    return context.get("ssh")


@wrap_all_exception_logged(logger=None)
def destroyCliConnection(context):
    '''
    @summary: 销毁cli连接
    @param context: 上下文对象
    '''
    conn = getItem(context, CLI_CONNECTION)
    if conn is not None:
        conn.close()
        removeItem(context, CLI_CONNECTION)


def initTlvConnection(context):
    '''
    @summary: 初始化tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    base_dev = get_base_dev(context)
    ip = devObj.get("ip")
    tlvPort = devObj.get("tlvPort")

    conn = TlvConnection(connectorFactoryObj, ip, tlvPort)
    conn.createForward(base_dev)

    setItem(context, TLV_CONNECTION, conn)
    return


def createTlvConnection(context, ip, port, user, pawd):
    connectorFactoryObj = getConnectorFactoryObj(context)
    conn = TlvConnection(connectorFactoryObj, ip, port)
    tlv = conn.create(user, pawd)
    return tlv


def getTlv(context, reBuild=False):
    '''
    @summary: 获取rest连接，使用原有tlv接口
    @param context: 上下文对象
    '''
    return getRest(context, reBuild)


def getOldTlv(context):
    """获取tlv连接，支持跳转

    :param context: 上下文对象
    :return:
    """
    conn = getItem(context, TLV_CONNECTION)
    if conn is None:
        initTlvConnection(context)
        conn = getItem(context, TLV_CONNECTION)

    base_dev = get_base_dev(context)
    tlv = conn.getTlvForward(base_dev)
    return tlv


def destroyTlvConnection(context):
    '''
    @summary: 销毁tlv连接
    @param context: 上下文对象
    '''
    logger = context.get("logger")
    try:
        conn = getItem(context, REST_CONNECTION)
        if conn is not None:
            conn.close()
            removeItem(context, REST_CONNECTION)
        return
    except:
        logger.error("close tlv connection failed.")


def destroyOldTlvConnection(context):
    """销毁tlv连接
    :param context:上下文对象
    :return:
    """
    logger = context.get("logger")
    try:
        conn = getItem(context, TLV_CONNECTION)
        if conn is not None:
            conn.close()
            removeItem(context, TLV_CONNECTION)
        return
    except Exception:
        logger.error("close old tlv connection failed.")


def initTlvConnectionSVP(context):
    '''
    @summary: 初始化SVP的tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)

    conn = TlvConnectionSVP(connectorFactoryObj, deviceType=devObj.get("type"), sn=devObj.get("sn"))
    conn.create()

    setItem(context, SVP_CONNECTION, conn)
    return


def getTlvSVP(context):
    '''
    @summary: 获取SVP的tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, SVP_CONNECTION)
    if conn is None:
        initTlvConnectionSVP(context)
        conn = getItem(context, SVP_CONNECTION)

    tlv = conn.getTlv()
    return tlv


def destroyTlvConnectionSVP(context):
    '''
    @summary: 销毁SVP的tlv连接
    @param context: 上下文对象
    '''
    logger = context.get("logger")
    try:
        conn = getItem(context, SVP_CONNECTION)
        if conn is not None:
            conn.close()
            removeItem(context, SVP_CONNECTION)
        return
    except:
        logger.error("close tlv connection failed.")


def setResult(context, flag, errMsg="", suggestion="", ignore_flag=False):
    '''
    @summary: 往上下文对象中传递处理结果
    @param context: 上下文对象
    @param flag: 标识
    @param errMsg: 错误消息
    @param suggestion: 修复建议
    @param ignore_flag: 是否可忽略该检查项的标志
    '''
    context["succ"] = flag
    context["errMsg"] = errMsg
    context["suggestion"] = suggestion
    context["ignoreFlag"] = ignore_flag
    return


def handleSuccess(context):
    '''
    @summary: 处理成功时，返回成功的结果给上下文对象
    @param context: 上下文对象
    '''
    setResult(context, True)
    return


def handleFailure(context, resultDict={"flag": False, "errMsg": "", "suggestion": ""}):
    '''
    @summary: 处理失败时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param resultDict: 结果字典
    '''
    errMsg = resultDict.get("errMsg", "")
    suggestion = resultDict.get("suggestion", "")
    ignore_flag = resultDict.get("ignoreFlag", False)

    lang = getLang(context)
    if len(errMsg) == 0:
        if lang == "zh":
            errMsg = u"系统处理消息失败。"
        else:
            errMsg = "Failed to process the message."
    if len(suggestion) == 0:
        if lang == "zh":
            suggestion = u"请检查系统状态或稍后重试。如果有任何疑问，请联系技术支持工程师协助处理。"
        else:
            suggestion = "Please check the system status or try again later. If you have any problems, please contact technical support engineers for help."

    setResult(context, False, errMsg, suggestion, ignore_flag)
    return


def handleException(context, exception, resultDict={}):
    '''
    @summary: 处理异常时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param exception: 异常信息
    '''
    lang = getLang(context)
    removeItem(context, TLV_CONNECTION)
    removeItem(context, CLI_CONNECTION)

    if isinstance(exception, CliCmdException):
        handleNetworkFailure(context, lang)
        return

    expArgs = exception.args
    if expArgs is not None and len(expArgs) == 2:
        errCode = str(expArgs[0]).strip()
        # 设备未连接
        if errCode in COMMUNICATION_FAIL_ERROR_CODE:
            handleNetworkFailure(context, lang)
            return

        # 引擎无法监控
        if errCode in ENGINE_CAN_NOT_MONITOR:
            handleEngineUnMonitor(context, lang)
            return

        # 命令执行失败
        if errCode in CLI_CAN_NOT_EXECUTE:
            handleCmdExecuteFailure(context, lang)
            return
        elif errCode == UPGRADE_ERROR_CODE:
            hanldeArrayUpgrading(context, lang)
            return
    handleFailure(context, resultDict)
    return


def hanldeArrayUpgrading(context, lang):
    if lang == "zh":
        errMsg = u"系统正在升级，不能进行扩容。"
        suggestion = u""
    else:
        errMsg = "The system cannot be expanded when it is being upgraded."
        suggestion = ""
    setResult(context, False, errMsg, suggestion)


def handleCmdExecuteFailure(context, lang):
    if lang == "zh":
        errMsg = u"命令执行失败。"
        suggestion = u""
    else:
        errMsg = "Command execute failed."
        suggestion = ""
    setResult(context, False, errMsg, suggestion)


def handle_connect_failure(context, lang):
    """处理连接失败异常

    :param context:
    :param lang:
    :return:
    """
    if lang == "zh":
        err_msg = u"与设备通信异常，请检查网络连接或设备状态是否正常。"
        suggestion = u""
    else:
        err_msg = "The communication with the device is abnormal. " \
                  "Check whether the network connection and " \
                  "device status are normal."
        suggestion = ""
    setResult(context, False, err_msg, suggestion)


def handleEngineUnMonitor(context, lang):
    if lang == "zh":
        errMsg = u"引擎无法监控。"
        suggestion = u"请查看系统中告警并按照告警修复建议进行处理。"
    else:
        errMsg = "The engine can not be monitored."
        suggestion = "Please rectify the faults by referring to recommended actions."
    setResult(context, False, errMsg, suggestion)


def handleNetworkFailure(context, lang):
    destroyTlvConnection(context)
    destroyCliConnection(context)
    setItem(context, "isADStarted", False)
    expFlowStatus = getItem(context, EXP_FLOW_STATUS_KEY)
    if lang == "zh":
        # 默认情况下关闭工具重新执行扩容流程
        errMsg = u"与原阵列通信异常。"
        suggestion = u"请检查网络连接或原阵列状态是否正常（检查方法：登录DeviceManager管理界面，查看控制器 “健康状态”为“正常”，“运行状态”为“在线”），" \
                     u"在状态正常后点击“重新检测”或”重新执行”按钮。" \
                     u"如果扩容过程中原阵列发生过复位或掉电，请关闭工具重新启动扩容流程。如果有任何疑问，请联系技术支持工程师协助处理。"
        if expFlowStatus == EXP_FLOW_STATUS_CHANGE_DATA:  # 提示扩容回退
            suggestion = u"请检查网络连接或原阵列状态是否正常（检查方法：登录DeviceManager管理界面，查看控制器 “健康状态”为“正常”，“运行状态”为“在线”），" \
                         u"在状态正常后点击“重新检测”或”重新执行”按钮。" \
                         u"如果扩容过程中原阵列发生过复位或掉电，请参考《扩容指导书》进行扩控回退操作。如果有任何疑问，请联系技术支持工程师协助处理。"
        elif isSingleCtrl(context) or expFlowStatus == EXP_FLOW_STATUS_JOIN_SYS_SUCCESS:  # 扩容成功后步骤、单控都 按照原场景提示
            suggestion = u"请检查网络连接或原阵列状态是否正常（检查方法：登录DeviceManager管理界面，查看控制器 “健康状态”为“正常”，“运行状态”为“在线”），" \
                         u"在状态正常后点击“重新检测”或”重新执行”按钮。如果有任何疑问，请联系技术支持工程师协助处理。"
        elif expFlowStatus == EXP_FLOW_STATUS_REBOOT_NEW_CONTROLLER:  # 提示重新执行当前步骤
            suggestion = u"请检查工具与原阵列的网络连接状态和原阵列状态是否正常（检查方法：登录DeviceManager管理界面，查看控制器“健康状态”为“正常”，“运行状态”为“在线”），" \
                         u"如果正常，请点击”重新执行”按钮。否则，请联系技术支持工程师协助处理。"
    else:
        errMsg = "Communicating with the original storage array failed."
        suggestion = "Check whether the network connection or status of the original storage array is normal. (Method: Log in to DeviceManager and check " \
                     "that Health Status and Running Status of the controller are Normal and Online respectively.) After the status is normal, click Recheck or Re-Execute." \
                     "If the original storage array has been reset " \
                     "or a power failure occurs during the expansion, close the tool and perform the expansion procedure again. If you have any questions, " \
                     "contact technical support engineers."
        if expFlowStatus == EXP_FLOW_STATUS_CHANGE_DATA:
            suggestion = "Check whether the network connection or status of the original storage array is normal. (Method: Log in to DeviceManager and check " \
                         "that Health Status and Running Status of the controller are Normal and Online respectively.) After the status is normal, click Recheck or Re-Execute." \
                         "If the original storage array has been reset " \
                         "or a power failure occurs during the expansion, perform rollback operations by referring to the Capacity Expansion Guide. " \
                         "If you have any questions, contact technical support engineers."
        elif isSingleCtrl(context) or expFlowStatus == EXP_FLOW_STATUS_JOIN_SYS_SUCCESS:
            suggestion = "Check whether the network connection or status of the original storage array is normal. (Method: Log in to DeviceManager and check " \
                         "that Health Status and Running Status of the controller are Normal and Online respectively.) After the status is normal, click Recheck or Re-Execute." \
                         "If you have any questions, contact technical support engineers."
        elif expFlowStatus == EXP_FLOW_STATUS_REBOOT_NEW_CONTROLLER:  # 提示重新执行当前步骤
            suggestion = "Check whether the network connection between the tool and the original storage array is normal, and the status of the original storage array " \
                         "is normal. (Method: Log in to DeviceManager and check that Health Status and Running Status of the controller are Normal " \
                         "and Online respectively.) If both the network connection and status are normal, click Re-Execute. " \
                         "Otherwise, contact technical support engineers."
    setResult(context, False, errMsg, suggestion)
    scriptDefDict = getScriptDefDict(context)
    return scriptDefDict.get(RESOURCE_DEF_CONTEXT)


def getResource(context, key):
    '''
    @summary: 根据property资源文件中的键，获取对应的值
    @param context: 上下文对象
    @param key: property资源文件中的键
    '''
    resourceDict = getResourceDict(context)
    return resourceDict.get(key, "--")


def getResourecePath(context):
    '''
    @summary: 返回property文件所在的路径
    @param context: 上下文对象
    '''
    lang = getLang(context)
    return os.path.join(getImportRootDir(context), "res", "language", "common_%s.properties" % lang)


def getImportRootDir(context):
    return os.path.dirname(context.get("importRootDir"))


def setCableDetectItem(context, checkItems):
    '''
    @summary: 设置上下文中检测项(限线缆检测)
    @param context: 上下文对象
    '''
    context["retData"] = jsonUtil.dictList2JsonArray(checkItems)
    return


def setCableDetectFailure(context, infoList):
    '''
    @summary: 返回上下文中检测项的检测结果(限线缆检测)
    @param context: 上下文对象
    '''
    dictList = []
    cnt = 0
    for info in infoList:
        dictList.append({"id": str(cnt), "title": info})
        cnt += 1
    context["retData"] = jsonUtil.dictList2JsonArray(dictList)
    handleFailure(context)
    return


def handleDetectException(context, exception):
    '''
    @summary: 处理异常时，返回失败的结果给上下文对象
    @param context: 上下文对象
    @param exception: 异常信息
    '''
    handleException(context, exception)
    infoList = [context["errMsg"], context["suggestion"]]
    setCableDetectFailure(context, infoList)
    return


def checkConnection(context):
    """
    @summary:检查工具与整列的连接，阵列密码状态，密码过期或即将过期时提示用户修改
    @return：
    """
    dev = context.get("dev")
    ip = dev.get("ip")
    user = dev.get("user")
    sshPort = int(dev.get("sshPort"))
    pawd = dev.get("pawd")
    cliConnection = None

    try:
        sshConnector = getItem(context, SSH_CONNECTOR)
        if sshConnector is None:
            connFactory = context.get("connectorFactory")
            sshConnector = connFactory.createSshConnector(ip, sshPort, user, pawd)
            setItem(context, SSH_CONNECTOR, sshConnector)
        cliConnection = sshConnector.getConnectionForPwdWillExpireBreak()
        return True, ""

    # 密码已经过期或即将过期
    except PwdException as pwdExp:
        errMsg = pwdExp.getErrorMsg()
        return False, errMsg

    # 框架抛出的其它连接异常
    except ToolException:
        errMsg = getConnFailMsg(context)
        return False, errMsg

    finally:
        del pawd


def getConnFailMsg(context):
    """
    @summary：设置与设备连接失败的通用错误消息
    """
    lan = context.get("lan")

    if lan == "zh":
        return u"与设备通信异常，请检查网络连接或设备状态是否正常。"
    else:
        return "The communication with the device is abnormal. Check whether the network connection and device status are normal."


def setTableItemInit(context, tableItems):
    '''
    @summary: 扩框灵活配置的列表对象
    @param context: 上下文对象
    @param tableItems: 列表的字典列表
    '''
    tableItemsStrList = []
    for d1 in tableItems:
        tableItemsStrList.append(json.dumps(d1))
    tableItemsJsonArray = "[%s]" % (','.join(tableItemsStrList))

    context["retData_init"] = tableItemsJsonArray
    return tableItemsJsonArray


def setNewRowItem(context, rowItems):
    '''
    @summary: 扩框灵活配置的列表对象
    @param context: 上下文对象
    @param tableItems: 列表的字典列表
    '''
    context["retData"] = json.dumps(rowItems)
    return


def updateSelectedConfig(context, rowId, item, value):
    '''
    @summary: 更新扩容配置信息
    @param context: 上下文对象
    @param rowId: 所在行号
    @param item: 修改项目
    @param value: 修改值
    '''
    logger = context.get("logger")
    rowId = str(rowId)
    selectedConfig = getItem(context, "selectedConfig")

    rowConfigInfo = selectedConfig.get(rowId)
    rowConfigInfo[item] = value
    selectedConfig[rowId] = rowConfigInfo.copy()
    setItem(context, "selectedConfig", selectedConfig)
    logger.info("rowConfigInfo:%s" % rowConfigInfo)
    logger.info("item:%s, value:%s " % (item, str(value)))
    return


def deleteSelectedConfigByRow(context, rowId):
    '''
    @summary: 更新扩容配置信息
    @param context: 上下文对象
    @param rowId: 所在行号
    @param item: 修改项目
    @param value: 修改值
    '''
    logger = context.get("logger")
    rowId = str(rowId)
    selectedConfig = getItem(context, "selectedConfig")

    # 删除对应行数据
    selectedConfig.pop(rowId)
    setItem(context, "selectedConfig", selectedConfig)
    logger.info("the selectedConfig after delete a row data:%s" % selectedConfig)
    return


def getCabinetNameInRow(context, rowId):
    '''
    @summary: 更新扩容配置信息
    @param context: 上下文对象
    @param rowId: 所在行号
    '''
    rowId = str(rowId)
    selectedConfig = getItem(context, "selectedConfig")
    rowConfigInfo = selectedConfig.get(rowId)
    cabinetName = rowConfigInfo.get("cabinetName")
    return cabinetName


def getOwningCtrlInRow(context, rowId):
    """获取扩框配置信息中的指定行的归属控制器。
    :param context:上下文对象
    :param rowId:所在行号
    :return:
    """
    rowId = str(rowId)
    selectedConfig = getItem(context, "selectedConfig")
    rowConfigInfo = selectedConfig.get(rowId)
    owningCtrl = rowConfigInfo.get("owningCtrl")
    return owningCtrl


def updateEclosures(context):
    '''
    @summary: 更新新增硬盘框信息
    @param context: 上下文对象
    '''

    newEnc2UList = []
    newEnc2UNVMeList = []
    newEnc4UList = []
    selectedConfig = getItem(context, "selectedConfig")
    for item in selectedConfig.values():
        for key in item.keys():
            if key == "newEnc2UList":
                newEnc2UList.extend(item[key])
            elif key == "newEnc2UNVMeList":
                newEnc2UNVMeList.extend(item[key])
            elif key == "newEnc4UList":
                newEnc4UList.extend(item[key])
            else:
                continue
    setItem(context, "newEnc2UList", newEnc2UList)
    setItem(context, "newEnc2UNVMeList", newEnc2UNVMeList)
    setItem(context, "newEnc4UList", newEnc4UList)
    return


def updateEnclosuresNum(context):
    """更新新增硬盘框信息(Dorado5000/6000 需要调用，检查硬盘上电时使用）

    :param context:上下文对象
    :return:
    """
    selectedConfig = getItem(context, "selectedConfig", {})
    newEnc2UNum = sum([selectedConfig.get(rowKey, {}).get('newEnc2UNum', 0) for rowKey in selectedConfig])
    setItem(context, "newEnc2UNum", newEnc2UNum)

    return


def getSelectedEncNumByCabinet(context, cabinetName):
    '''
    @summary: 获取当前刷新的行号
    @param context: 上下文对象
    '''
    enc2UNum = 0
    enc4UNum = 0
    selectedConfig = getItem(context, "selectedConfig")
    for d in selectedConfig.values():
        if cabinetName in d.values():
            enc2UNum = len(d.get("newEnc2UList"))
            enc4UNum = len(d.get("newEnc4UList"))
            return (enc2UNum, enc4UNum)

    return (enc2UNum, enc4UNum)


def updateSelectedCabinets(context):
    """更新已选择柜列表

    :param context: 上下文对象
    :return:
    """
    # 已选柜列表
    selected_cabinets = list()
    # 已选柜排序的列表
    selected_cabinets_sort = list()
    # 剩余可选柜列表
    available_cabinets = list()
    available_cabinets_group = getItem(context, "availableCabinetsGroup", [])
    selected_config = getItem(context, "selectedConfig")
    for d in selected_config.values():
        selected_cabinets.append(d.get("cabinetName"))
    # 按原有顺序分别保存已选柜和剩余可选柜
    for e in available_cabinets_group:
        if e not in selected_cabinets:
            available_cabinets.append(e)
        else:
            selected_cabinets_sort.append(e)
    setItem(context, "selectedCabinets", selected_cabinets_sort)
    setItem(context, "availableCabinets", available_cabinets)
    return


def updateOwningCtrlInfoInContext(context):
    """更新已选择归属控制器的列表
    :param context:上下文对象
    :return:
    """
    selectedOwningCtrls = set()
    availableOwningCtrls = set()
    sysAvailableOwingCtrl = getItem(context, "sysAvailableOwingCtrls")
    selectedConfig = getItem(context, "selectedConfig")

    for d in selectedConfig.values():
        owingCtrl = d.get('owningCtrl')
        selectedOwningCtrls.add(owingCtrl)

    selectedOwningCtrls = sorted(selectedOwningCtrls)
    setItem(context, "selectedOwningCtrls", selectedOwningCtrls)

    for e in sysAvailableOwingCtrl:
        if e not in selectedOwningCtrls:
            availableOwningCtrls.add(e)

    availableOwningCtrls = sorted(availableOwningCtrls)
    setItem(context, 'availableOwningCtrls', availableOwningCtrls)

    logger = context.get("logger")
    logger.info('availableOwningCtrls in context:%s' % availableOwningCtrls)
    return


def updateNewBayList(context):
    '''
    @summary:扩容硬盘柜时，更新新增的柜列表
    @param context: 上下文对象
    '''
    selectedCabinets = getItem(context, "selectedCabinets")
    setItem(context, "newBayList", selectedCabinets)
    return


def updateNewBayListForDorado18000(context):
    """扩容硬盘框时，更新新增的柜列表
    :param context:
    :return:
    """
    selectedOwningCtrls = getItem(context, "selectedOwingCtrls")
    baySet = set()
    for owingCtrl in selectedOwningCtrls:
        smbNum = owingCtrl.split('CTE')[1].split('(')[0]
        baySet.add('SMB' + smbNum)
    setItem(context, "newBayList", list(baySet))
    return


def getRest(context, reBuild=False):
    '''
    @summary: 获取rest连接
    @param context: 上下文对象
    '''
    restConn = getItem(context, REST_CONNECTION)
    if reBuild or restConn is None:
        initRestConnection(context)
        restConn = getItem(context, REST_CONNECTION)

    devObj = getDevObj(context)
    restConn.getRestConnection(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
    return restConn


def releaseRest(context):
    """释放 Rest 连接。

    :param context:
    :return:
    """
    restConn = getItem(context, REST_CONNECTION)
    if restConn is None:
        return

    # noinspection PyBroadException
    try:
        restConn.close()
    except (Exception, JException):
        return


def initRestConnection(context):
    '''
    @summary: 初始化rest连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)

    ip = devObj.get("ip")
    devSN = devObj.get("sn")
    conn = RestConnection(connectorFactoryObj, ip, devSN,
                          timeOut=context.get("REST_RESPONSE_TMOUT"))
    conn.create(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
    setItem(context, REST_CONNECTION, conn)
    return


def createRest(context, newIp):
    '''
    @summary: 使用ip创建rest连接
    @param context: 上下文对象
    @param newIp: 创建连接的ip
    '''
    try:
        connectorFactoryObj = getConnectorFactoryObj(context)
        devObj = getDevObj(context)

        devSN = devObj.get("sn")
        restConn = RestConnection(connectorFactoryObj, newIp, devSN)
        result = restConn.create(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
        return (result, restConn)
    except:
        return (False, None)


def getConnThroughSpecificIp(context, newIp):
    '''
    @summary: 使用指定ip创建新的rest连接，用于工具与阵列交互
    @param context: 上下文对象
    @param newIp: 创建新连接的ip
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    devSN = devObj.get("sn")
    restConn = RestConnection(connectorFactoryObj, newIp, devSN)
    result = restConn.create(devObj.get("user"), devObj.get("pawd"), config.ERR_CODE_NORMAL)
    setItem(context, REST_CONNECTION, restConn)
    devObj["ip"] = newIp


def createSvpRest(context):
    '''
    @summary: 使用ip创建rest连接
    @param context: 上下文对象
    @param newIp: 创建连接的ip
    '''
    try:
        connectorFactoryObj = getConnectorFactoryObj(context)
        devObj = getDevObj(context)
        ip = devObj.get("ip")
        devSN = "SVP"
        restConn = RestConnection(connectorFactoryObj, ip, devSN)
        result = restConn.create(devObj.get("user"), devObj.get("pawd"), config.ERR_CODE_NORMAL)
        return (result, restConn)
    except:
        return (False, None)


def initSftpConnection(context):
    """初始化sftp连接

    :param context: 上下文对象
    :return:
    """
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    base_dev = get_base_dev(context)
    ip = devObj.get("ip")
    conn = SftpConnection(connectorFactoryObj, ip)
    conn.create(base_dev)
    setItem(context, SFTP_CONNECTION, conn)
    return


def getSftp(context):
    """获取sftp连接

    :param context: 上下文对象
    :return:
    """
    restConn = getItem(context, SFTP_CONNECTION)
    if restConn is None:
        initSftpConnection(context)
        restConn = getItem(context, SFTP_CONNECTION)

    base_dev = get_base_dev(context)
    restConn.getSftp(base_dev)
    return restConn


def getBaseUri(context):
    '''
    @summary: 获取rest的基础链接
    @param context: 上下文对象
    '''
    return getItem(context, "REST_BASE_URI")


def getProductModel(context):
    '''
    @summary: 获取自定义字典中的产品型号
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    productModel = dev.get("type")
    return productModel


def getInnerIpList(context):
    '''
    @summary: 设置高端的内部IP
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.get("devInnerIpList")


def setInnerIpList(context):
    '''
    @summary: 设置高端的内部IP
    @param context: 上下文对象
    '''
    setItem(context, "DEV_INNER_IP_LIST", getInnerIpList(context))
    return


def getContextItem(context, key):
    '''
    @summary: 获取框架字典中的项
    @param context: 上下文对象
    @param key: 项的键
    '''
    return context.get(key)


def getSelectedItem(context, key):
    '''
    @summary: 获取框架字典中的被选中的数据，用于FRU工具
    @param context: 上下文对象
    @param key: 项的键
    '''
    selectedInfo = getContextItem(context, key)
    return json.loads(selectedInfo)


def getIp(context):
    '''
    @summary: 获取当前工具连接的ip
    @param context: 上下文对象
    '''
    devObj = getDevObj(context)
    if isinstance(devObj, DevNode):
        return devObj.getIp()
    return devObj.get("ip")


def get_dev_type(context):
    '''
    @summary: 获取当前工具的设备型号
    @param context: 上下文对象
    '''
    dev_obj = getDevObj(context)
    if isinstance(dev_obj, DevNode):
        return str(dev_obj.getDeviceType())
    return dev_obj.get("type", "")


def initConnection(context):
    '''
    @summary: 初始化连接（cli, tlv, rest, sftp）
    @param context: 上下文对象
    '''
    rest_conn = getItem(context, REST_CONNECTION)
    if rest_conn:
        rest_conn.close()
    removeItem(context, TLV_CONNECTION)
    removeItem(context, CLI_CONNECTION)
    setItem(context, REST_CONNECTION, None)
    setItem(context, SFTP_CONNECTION, None)
    return


def initSvpRestConnection(context):
    '''
    @summary: 创建SVP的rest连接
    @param context: 上下文对象
    @param newIp: 创建连接的ip
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    ip = devObj.get("ip")
    devSN = "SVP"
    conn = RestConnection(connectorFactoryObj, ip, devSN)
    conn.create(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
    setItem(context, SVP_REST_CONNECTION, conn)
    return


def getSvpRest(context):
    '''
    @summary: 获取SVP rest连接
    @param context: 上下文对象
    '''
    svpRestConn = getItem(context, SVP_REST_CONNECTION)
    if svpRestConn is None:
        initSvpRestConnection(context)
        svpRestConn = getItem(context, SVP_REST_CONNECTION)

    devObj = getDevObj(context)
    svpRestConn.getRestConnection(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
    return svpRestConn


def getSceneArgStrValue(context, key):
    '''
    @summary: 获取场景化的路标参数
    @param context: 上下文对象
    '''
    try:
        argsObj = context.get("sceneArgsJsonObj")
        if argsObj:
            return argsObj.getString(key)
    except:
        return None
    return None


def getUserOpDataMap(context):
    '''
    @summary: 获取上下文对象中用户行为数据userOpData空Map,框架输出到日志
    @param context: 上下文对象
    '''
    return context.get("userOpData")


def appendUserOpData(context, expansionType):
    '''
    @summary: 在用户行为数据中设置扩容类型和扩容前后数据，框架输出到用户行为日志
    @param context: 上下文对象
    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict.clear()
        userOpDataDict[UserOpDataFields.TYPE] = getItem(context, expansionType)

        # 扩控制器
        if expansionType == UserOpDataFields.EXPANSION_TYPE_CTRL:
            userOpDataDict[UserOpDataFields.ORIGIN_CTRL_NUM] = getItem(context, "ctrlNum")
            userOpDataDict[UserOpDataFields.ORIGIN_CLUST_TYPE] = getItem(context, "configClustType")
            userOpDataDict[UserOpDataFields.NEW_CTRL_NUM] = getItem(context, "newConfigCtrlNum")
            userOpDataDict[UserOpDataFields.NEW_CLUST_TYPE] = getItem(context, "newConfigClustType")

        # 扩硬盘柜
        elif expansionType in (UserOpDataFields.EXPANSION_TYPE_BAY, UserOpDataFields.EXPANSION_TYPE_ENCLOSURE):
            userOpDataDict[UserOpDataFields.ORIGIN_DISK_ENCLOSURE_NUM] = getItem(context, "diskEnclosureNum")
            newEncList = getItem(context, "newEnc2UList") + getItem(context, "newEnc4UList")
            userOpDataDict[UserOpDataFields.NEW_DISK_ENCLOSURE_NUM] = len(newEncList)
    except:
        logger = context.get("logger")
        logger.error("AppendUserOpData Error!")


# from fru

def setPreCableData(context, fruType="", selectedRow0={}, selectedRow1={}):
    """在用户行为数据中设置备件类型和状态字段，框架输出到用户行为日志

    :param context: 上下文
    :param fruType: 部件类型
    :param selectedRow0: 所选线缆始端
    :param selectedRow1: 所选线缆末端
    :return:
    """
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict.clear()
        # SAS线缆\Ip scaleOut线缆
        if fruType in [UserOpDataFields.SAS_CABLE,
                       UserOpDataFields.IP_SCALEOUT_CABLE]:
            healthStatus0 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow0.get("startPortHealStatus", ""))
            runningStatus0 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow0.get("startPortRunStatus", ""))
            healthStatus1 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow0.get("curPeerPortHealStatus", ""))
            runningStatus1 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow0.get("curPeerPortRunStatus", ""))
        # pcie线缆
        elif fruType == UserOpDataFields.PCIE_CABLE:
            healthStatus0 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow0.get("cpHealthStatus", ""))
            runningStatus0 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow0.get("cpRunningStatus", ""))
            healthStatus1 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow0.get("curPeerPortHealStatus", ""))
            runningStatus1 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow0.get("curPeerPortRunStatus", ""))
        else:
            # 管理线缆
            healthStatus0 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow0.get("healthStatus", ""))
            runningStatus0 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow0.get("runningStatus", ""))
            healthStatus1 = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum,
                                         selectedRow1.get("healthStatus", ""))
            runningStatus1 = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum,
                                          selectedRow1.get("runningStatus", ""))
        userOpDataDict[UserOpDataFields.FRU_TYPE] = fruType
        userOpDataDict[UserOpDataFields.STARTPORT_HEALTHSTATUS] = healthStatus0
        userOpDataDict[UserOpDataFields.STARTPORT_RUNNINGSTATUS] = runningStatus0
        userOpDataDict[UserOpDataFields.ENDPORT_HEALTHSTATUS] = healthStatus1
        userOpDataDict[UserOpDataFields.ENDPORT_RUNNINGSTATUS] = runningStatus1
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set Selected Cable Data into UserOpData: %s" % ex)


def setPostFruData(context, barCode=""):
    '''
    @summary: 在用户行为数据中设置更换后备件信息
    @param barCode: 部件条码
    @param context: 上下文对象
    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict[UserOpDataFields.POST_BARCODE] = barCode
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set replaced FRU Data into UserOpData : %s" % ex)


def setClickReplaceBtnTime(context):
    '''
    @summary: 从上下文中取出存放点击“确认已更换”按钮的时间点clickReplacedButtonTime，统一放入用户行为数据中，用于区分更换前后
    @param context: 上下文对象
    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict[UserOpDataFields.CLICK_REPLACED_BUTTON_TIME] = context["clickReplacedButtonTime"]
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set Click ReplaceButton Time  Data into UserOpData : %s" % ex)


def setPreDiskData(context, fruType="", selectedRow={}):
    '''
    @summary: 在用户行为数据中设置更换前选中的硬盘信息
    @param fruType: 部件类型
    @param context: 上下文对象
    @param healthStatus: 健康状态
    @param runningStatus: 运行状态
    @param diskType: 硬盘类型
    @param model: 型号
    @param runTime: 运行天数
    @param firmwareVersion: 固件版本
    
    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict.clear()
        healthStatus = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum, selectedRow.get("healthStatus", ""))
        runningStatus = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum, selectedRow.get("runningStatus", ""))
        diskType = getEnStrEnum(restData.EnumStr.StrDiskTypeEnum, selectedRow.get("diskType", ""))

        userOpDataDict[UserOpDataFields.FRU_TYPE] = fruType
        userOpDataDict[UserOpDataFields.HEALTH_STATUS] = healthStatus
        userOpDataDict[UserOpDataFields.RUNNING_STATUS] = runningStatus
        userOpDataDict[UserOpDataFields.PRE_DISK_TYPE] = diskType
        userOpDataDict[UserOpDataFields.PRE_MODEL] = selectedRow.get("model", "")
        userOpDataDict[UserOpDataFields.PRE_RUN_TIME] = selectedRow.get("runTime", "")
        userOpDataDict[UserOpDataFields.PRE_FIRMWARE_VERSION] = selectedRow.get("firmwareVersion", "")
        userOpDataDict[UserOpDataFields.PRE_BARCODE] = selectedRow.get("barcode", "")
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set Selected Disk Data into UserOpData: %s" % ex)


def setPostDiskData(context, selectedRow={}):
    '''
    @summary: 在用户行为数据中设置更换后的硬盘信息
    @param context: 上下文对象
    @param healthStatus: 健康状态
    @param runningStatus: 运行状态
    @param diskType: 硬盘类型
    @param model: 型号
    @param runTime: 运行天数
    @param firmwareVersion: 固件版本

    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict[UserOpDataFields.POST_PRE_DISK_TYPE] = getEnStrEnum(restData.EnumStr.StrDiskTypeEnum,
                                                                           selectedRow.get("diskType", ""))
        userOpDataDict[UserOpDataFields.POST_MODEL] = selectedRow.get("model", "")
        userOpDataDict[UserOpDataFields.POST_RUN_TIME] = selectedRow.get("runTime", "")
        userOpDataDict[UserOpDataFields.POST_FIRMWARE_VERSION] = selectedRow.get("firmwareVersion", "")
        userOpDataDict[UserOpDataFields.POST_BARCODE] = selectedRow.get("barcode", "")
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set Relaced Disk Data into UserOpData: %s" % ex)


def setPubPreFruData(context, fruType="", selectedRow={}):
    '''
    @summary: 在用户行为数据中设置备件类型和状态字段，框架输出到用户行为日志
    @param fruType: 部件类型
    @param context: 上下文对象
    @param healthStatus: 健康状态
    @param runningStatus: 运行状态
    '''
    try:
        userOpDataDict = getUserOpDataMap(context)
        userOpDataDict.clear()
        healthStatus = getEnStrEnum(restData.EnumStr.StrHealthStatusEnum, selectedRow.get("healthStatus", ""))
        runningStatus = getEnStrEnum(restData.EnumStr.StrRunningStatusEnum, selectedRow.get("runningStatus", ""))
        barCode = selectedRow.get("BarCode", "")

        userOpDataDict[UserOpDataFields.FRU_TYPE] = fruType
        if healthStatus != "":
            userOpDataDict[UserOpDataFields.HEALTH_STATUS] = healthStatus
        if runningStatus != "":
            userOpDataDict[UserOpDataFields.RUNNING_STATUS] = runningStatus
        if barCode != "":
            userOpDataDict[UserOpDataFields.PRE_BARCODE] = barCode
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to run set Selected FRU Data into UserOpData: %s" % ex)


def setPreFruData(context, fruType, selectedRow):
    try:
        if fruType == UserOpDataFields.DISK:
            setPreDiskData(context, fruType, selectedRow)

        else:
            setPubPreFruData(context, fruType, selectedRow)
    except Exception as ex:
        BaseFactory.log.error(context, "Failed to set UserOpData: %s" % ex)


'''
Function: 据getStr()获取的TlvStr中的枚举字符串，反推获取字符串的keyId，用于获取健康状态和运行状态的枚举值和model号
Params:
    enum: 枚举字典
    enumValue: 枚举字典中的字符串value
Return: 
    enum中的keyId=正常返回值
'''


def getEnStrEnum(enum, enumValue):
    # 参数校验：若enum不为字典，或keyId不为数值，则直接返回
    # 初始化返回值为默认值
    retValue = "--"
    if type(enum) != dict:
        return retValue

    for keyId, value in enum.items():
        # 则按value获取字符串并分割为中英文
        langVals = str(value).split('||')
        if enumValue == langVals[0] or enumValue == langVals[1]:
            return keyId
    return retValue


def getNewDevObj(context):
    """获取上下文对象中dev对象

    :param context: 上下文对象
    :return:
    """
    return context.get("newDev")


def get_dev_node(context):
    """ 获取上下文对象中devNode对象
        用于获取CBB中保存的SVP模块信息
    :param context: 上下文对象
    :return:
    """
    return context.get("devNode0")


def getUiObserver(context):
    """获取上下文对象中uiObserver对象

    :param context: 上下文对象
    :return:
    """
    return context.get("uiObserver")


def getResObj(context):
    """获取上下文对象中resource对象

    :param context:上下文对象
    :return:
    """
    return context.get("resource")


def getCheckType(context):
    """获取上下文对象中checkType对象

    :param context: 上下文对象
    :return:
    """
    return context.get("checkType")


def initCliConnectionForSVP(context, ip):
    """初始化cli连接

    :param context: 上下文对象
    :param ip: ip
    :return:
    """
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getNewDevObj(context)

    sshPort = int(devObj.get("sshPort"))

    conn = CliConnection(connectorFactoryObj, ip, sshPort)
    conn.create(devObj.get("user"), devObj.get("pawd"))

    setItem(context, CLI_CONNECTION, conn)
    return


def getRestIp(context):
    return getNewDevObj(context).get("ip")


def getCurVersion(context):
    """获取系统当前版本

    :param context: 上下文对象
    :return:
    """
    dev_obj = getDevObj(context)
    if isinstance(dev_obj, DevNode):
        return str(dev_obj.getProductVersion())
    return getDevObj(context).get("version")


def getPkgVersion(context):
    """获取升级包版本

    :param context: 上下文对象
    :return:
    """
    return getDevObj(context).getUpgradeSetInfo().getUpgradePkgVersion()


def getDevType(context):
    """获取设备类型

    :param context: 上下文对象
    :return:
    """
    return getNewDevObj(context).get("type")


def getDevSN(context):
    """获取设备序列号Serial Number

    :param context: 上下文对象
    :return:
    """
    return getNewDevObj(context).get("sn")


def releaseTlvConn(context):
    """上下文对象

    :param context: 上下文对象
    :return:
    """
    protocalContext = context.get("protocalContext")
    tlvBuilder = protocalContext.get('TLVBUILDER')
    dev = getDevObj(context)
    tlvBuilder.releaseTlvConnection(dev.getDevNode())


def getTlvConn(context):
    """创建TLV连接

    :param context: 上下文对象
    :return:
    """
    protocalContext = context.get("protocalContext")
    if protocalContext:
        tlvBuilder = protocalContext.get('TLVBUILDER')
    else:
        tlvBuilder = context.get("TLVBUILDER")
    dev = getDevObj(context)
    if isinstance(dev, DevNode):
        return tlvBuilder.createTlvConnByInnerIp(dev)
    elif dev.hasattr('getDevNode'):  # 兼容硬盘升级工具的设备上下文.
        return tlvBuilder.createTlvConnByInnerIp(dev.getDevNode())


def getSelectedDisks(context):
    return getDevObj(context).getUpgradeSetInfo().getSelectedDisks()


def createCliConnection(context, ip, set_item=True):
    """根据ip创建cli连接，支持跳转

    :param context: 上下文对象
    :param ip: 创建cli连接的ip
    :param set_item 是否将连接设置到context中保存
    :return: 创建成功返回cli连接，否则为None
    """
    logger = context.get("logger")

    # 兼容不同工具框架
    base_dev = get_base_dev(context)
    new_dev = EntityUtils.cloneDevNode(base_dev)

    # 先保存原有ip，SN号，使用新ip建立连接后进行恢复
    originalIp = new_dev.getIp()
    originalSN = new_dev.getDeviceSerialNumber()

    try:
        logger.info("conn_sn_suffix={}".
                    format(context.get("conn_sn_suffix", "")))
        # 生成临时SN用于建立cli连接，一个SN号建立一个连接
        sn = originalSN + originalIp + ip + context.get("conn_sn_suffix", "")

        new_dev.setIp(ip)
        new_dev.setDeviceSerialNumber(sn)
        connectorFactoryObj = getConnectorFactoryObj(context)
        sshPort = int(new_dev.getPort())
        conn = CliConnection(connectorFactoryObj, ip, sshPort)
        cliConn = conn.createForwardConnetor(new_dev)
        if set_item:
            setItem(context, CLI_CONNECTION, conn)
            setItem(context, SSH_CONNECTOR, conn.sshConnector)
        return cliConn
    except:
        logger.error(unicode(traceback.format_exc()))
        return None


def getDevPort(context):
    """获取上下文对象中dev对象中的的登录端口
    
    :param context:上下文对象 
    :return: 
    """
    dev = getDevObj(context)
    return dev.getPort()


def isDoradoDev(context):
    dev = getDevObj(context)
    productModel = str(dev.getDevNode().getDeviceType())
    if productModel in config.DORADO_DEVS:
        return True
    return False


def getImportPkgMode(dataDict):
    """获取导包模式。硬盘升级导包模式：0表示不支持导包；1表示支持手动导包；2表示支持独立导包；

    :param dataDict: 上下文对象
    :return:
    """
    dev = dataDict.get("dev")
    importPkgMode = dev.getImportPkgMode()
    return importPkgMode


def getSelectType(context):
    return str(getDevObj(context).getUpgradeSetInfo().getSelectDiskType())


def isSupportParallel(context):
    return getDevObj(context).isSupportParallel()


def isOnlineUpgrade(dataDict):
    dev = dataDict.get("dev")
    upgradeModel = str(dev.getUpgradeSetInfo().getUpgradeModel())
    return "online" in upgradeModel.lower()


def getControllerNum(dataDict):
    dev = dataDict.get("dev")
    controllerNum = dev.getControllerNum()
    return controllerNum


def getContext(py_java_env):
    """从脚本上下文中获取python context

    history:
        1. 新增信息收集工具支持，增加toolName 兼容处理。

    :param py_java_env:
    :return:
    """
    context_map = {"infoCollect": "cbbContext", "DiskScan": "cbbContext"}
    tool_name = py_java_env.get("toolName")
    tool_context_key = context_map.get(tool_name)
    tool_context = py_java_env.get(tool_context_key)

    return (
        py_java_env.get("context")
        if not tool_name
        else tool_context
    )


def is_multi_check(context):
    """判断选择待更换备件是否支持多选

    :param context:
    :return:
    """
    return context.get("needMultiCheck", False)


def get_base_dev(context):
    """
    获取上下文中的CBB里的DevNode
    :param context: 工具上下文
    :return: CBB里的DevNode
    """
    tool_key_list = ["dev", "devInfo", "devNode0"]
    for key in tool_key_list:
        tool_dev_node = context.get(key)
        if tool_dev_node is None:
            continue
        # 继承类关系 升级评估工具
        if isinstance(tool_dev_node, DevNodeCommon):
            return tool_dev_node

        # 组合类关系 升级工具
        if hasattr(tool_dev_node, "getDevNode"):
            try:
                base_dev_node = tool_dev_node.getDevNode()
                if isinstance(base_dev_node, DevNodeCommon):
                    return base_dev_node
            except (Exception, JException) as exception:
                raise Exception("Get base DevNode function occur exception: %s"
                                % str(exception))

    raise Exception("Not found the base DevNode")


def is_run_in_svp():
    """
    是否在SVP内运行工具
    :return:
    """
    return LocalHostUtil.isRunInSvp()
