#!/bin/bash
CURRENT_PATH=$(cd "$(dirname "$0")"; pwd)
readonly components=("FusionStorage_common:sysconfig" "FusionStorage_oam-u:nma" "FusionStorage_cluster-platform:snm" "FusionStorage_oam-u:pma" "FusionStorage_oam-u:fma" "FusionStorage_oam-ftds:oam-ftds-agent" "FusionStorage_vbs:vbs" "FusionStorage_fdsa:fdsa" "FusionStorage_fsm:fsm-agent")
readonly install_actions=("install" "start" "check_status" "online")
readonly upgrade_actions=("ready" "offline" "stop" "upgrade" "start" "check_status" "online")
readonly upgrade_failed_rollback_actions=("offline:online" "stop:start" "upgrade:rollback" "start:stop" "check_status:stop" "online:offline" "all:ready")
readonly rollback_actions=("ready" "offline" "stop" "rollback" "start" "check_status" "online")
readonly LOCKFILE="/opt/.independent_deploy_upgrade.lock"
readonly UPGRADE_ROLLBACK_SCRIPT="/opt/dfv/oam/public/script/upgrade_rollback/"
readonly MICRO_PATH="/opt/fusionstorage/repository/deploymanager_pkg/micro"
readonly REPO_UPGRADE_PATH="/opt/fusionstorage/repository/components/upgrade"
readonly REPO_ROLLBACK_PATH="/opt/fusionstorage/repository/components/rollback"
readonly REPO_TMP_PATH="/opt/fusionstorage/repository/components/tmp"
readonly STEP_PATH="/opt/fusionstorage/repository/components/step"
readonly CLOUDA_CONFIG="/opt/fusionstorage/deploymanager/clouda/conf/cloudagent.ini"
readonly NETWORK_CONFIG="/opt/network/network_config.ini"
readonly UPGRADE_INFO="${STEP_PATH}/upgrade_info"
readonly SCENES_INFO="/opt/fusionstorage/deploy.ini"
readonly HA_PROPERTIES_PATH="/home/HAInfoFromInstallTool.properties"
readonly UPGRADE_STATUS_INFO="/opt/fusionstorage/upgrade_status"
readonly PRECHECK="ready"
readonly POSTCHECK="health_check"
readonly NOTIFY_COMPLETE="notify_finished"
readonly TASK_TYPE_PREINSTALL="preinstall"
readonly PREINSTALL_MAKR="/home/.preinstall"
readonly TASK_TYPE_UPGRADE="upgrade"
readonly TASK_TYPE_ROLLBACK="rollback"
readonly SCENES=("ram_os" "${TASK_TYPE_PREINSTALL}" "${TASK_TYPE_UPGRADE}" "${TASK_TYPE_ROLLBACK}")
readonly ACTION_INIT=0
readonly ACTION_SUCCESS=1
readonly ACTION_FAILED=2
readonly INSTALL_ACTION=("make_iso" "reboot_init")
readonly ROLE=("management" "storage" "compute" "dpc_compute")
readonly MANAGE="management"
readonly STORAGE="storage"
readonly COMPUTE="compute"
readonly DPC_COMPUTE="dpc_compute"
readonly LOG_FILE="/var/log/cloudagent/compute.action.log"
readonly IP_V4_REGULAR="^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$|^(([a-zA-Z]|[a-zA-Z][a-zA-Z0-9\-]*[a-zA-Z0-9])\.)*([A-Za-z]|[A-Za-z][A-Za-z0-9\-]*[A-Za-z0-9])$"
readonly IP_V6_REGULAR="^\s*((([0-9A-Fa-f]{1,4}:){7}(([0-9A-Fa-f]{1,4})|:))|(([0-9A-Fa-f]{1,4}:){6}(:|((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})|(:[0-9A-Fa-f]{1,4})))|(([0-9A-Fa-f]{1,4}:){5}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){0,1}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){0,2}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){0,3}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:)(:[0-9A-Fa-f]{1,4}){0,4}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(:(:[0-9A-Fa-f]{1,4}){0,5}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})))(%.+)?\s*$"
readonly COMPUTE_UPGRADE_JSON="${CURRENT_PATH}/compute_upgrade.json"
readonly status_failed="/opt/fusionstorage/deploymanager/compute_backup/upgrade_failed"
readonly status_running="/opt/fusionstorage/deploymanager/compute_backup/upgrade_running"
readonly status_success="/opt/fusionstorage/deploymanager/compute_backup/upgrade_success"
X86_64="x86_64"
OS_VERSION="EulerOS"
SCRIPTNAME=$(basename $0)
ARCH=$(uname -m)

function log()
{
    local param1=$1
    local param2=$2
    local param3=$3

    echo "[$(date +'%F %T')][${param1}][${param2}]${param3}"
    if [ ! -f "${LOG_FILE}" ]; then
        log_path=`dirname "${LOG_FILE}"`
        mkdir -p "${log_path}"
        chmod 750 "${log_path}"
        touch "${LOG_FILE}"
        chmod 640 "${LOG_FILE}"
    fi
    echo "[$(date +'%F %T')][${param1}][${param2}]${param3}" >> "${LOG_FILE}" 2>/dev/null
}

function execute_oamx_script()
{
    micro_service=$1
    component_name=$2
    action_type=$3
    component_dir=`ls ${MICRO_PATH} | grep -v tar.gz | grep ^${component_name}_[0-9].* | sort -r | head -1`
    if [ -z "${component_dir}" ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] ${component_name} component is not installed."
        return 0
    fi
    shell_script="${UPGRADE_ROLLBACK_SCRIPT}${action_type}/${micro_service}/${component_name}.sh"
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] start execute ${action_type} of ${component_name} component:${shell_script}"
    bash ${shell_script} >> ${LOG_FILE} 2>&1
    ret=$?
    if [ ${ret} -ne 0 ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] execute ${action_type} of ${component_name} component failed:${ret}."
        return $ret
    fi
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] execute ${action_type} of ${component_name} component success."
    return 0
}

function download()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] download params: $1 $2"
    local url_path=$1
    local component=$2
    local token=$3
    cd ${MICRO_PATH}
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] download command : 'wget ${url_path} -O ${component} --tries=5 --retry-connrefused --connect-timeout=30 --read-timeout=1000 --no-check-certificate --proxy=off'"
    local count=0
    while true; do
        let count+=1
        wget ${url_path} -O ${component} --tries=5 --retry-connrefused --connect-timeout=30 --read-timeout=1000 --no-check-certificate --proxy=off --header="x-auth-token:${token}" --append-output=${LOG_FILE}
        if [ $? -eq 0 ];then
            log $scriptName $FUNCNAME "[INFO][Line:${LINENO}]download $component success."
            return 0
        fi
        if [ ${count} -gt 6 ]; then
            log $scriptName $FUNCNAME "[ERROR][Line:${LINENO}]download $component failed."
            return 1
        fi
        sleep 3
    done
    #需要校验sha256sum
}

function download_pkg()
{
    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to download ${component_name}"
    cd ${MICRO_PATH}
    local component_name=$1
    local token=$2
    local float_ip=`cat ${CLOUDA_CONFIG} | grep -w cloudm_float_ip | awk -F '=' '{print $2}'`
    is_ip_v6 ${float_ip}
    if [[ $? -eq 0 ]];then
        url_path="https://[${float_ip}]:17443"
    else
        url_path="https://${float_ip}:17443"
    fi
    local platform=`uname -m`
    download_url=""
    if [[ "X${platform}" == "X${X86_64}" ]];then
        component_url=`get_spec_component_info ${component_name} x86DownloadUrl`
        component_config_url=`get_spec_component_info ${component_name} x86CfgDownloadUrl`
    else
        component_url=`get_spec_component_info ${component_name} armDownloadUrl`
        component_config_url=`get_spec_component_info ${component_name} armCfgDownloadUrl`
    fi
    if [[ "X${component_url}" == "X" ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] ${component_name} component url is empty."
        retrun 1
    fi
    local comp=`echo ${component_url} | awk -F "/" '{print $NF}'`
    download_url="${url_path}${component_url}"
    download ${download_url} ${comp} ${token}
    if [[ $? -ne 0 ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] download ${component_name} ${component_url} failed."
        retrun 1
    fi
    if [[ "X${component_config_url}" != "X" ]];then
        component_config=`echo ${component_config_url} | awk -F "/" '{print $NF}'`
        download_url="${url_path}${component_config_url}"
        download ${download_url} ${component_config} ${token}
        if [[ $? -ne 0 ]];then
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] download ${component_name} ${component_config_url} failed."
            retrun 1
        fi
    fi
    local comp_version=`echo ${comp}|awk -F ".tar.gz" '{print $1}'`
    if [[ "X${component_config_url}" != "X" ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to assemble ${component_name}"
        component_config_version=`echo ${component_config}|awk -F ".tar.gz" '{print $1}'`
        tar --no-same-owner -zxmf ${MICRO_PATH}/${comp} && chmod o-rwx -R ${MICRO_PATH}/${comp_version} \
        && tar --no-same-owner -zxmf ${MICRO_PATH}/${component_config} && chmod o-rwx -R ${MICRO_PATH}/${component_config_version} \
        && mv ${MICRO_PATH}/${component_config_version}/${component_config_version}.rpm ${MICRO_PATH}/${comp_version}/config -f \
        && mv ${MICRO_PATH}/${component_config_version}/main.yml ${MICRO_PATH}/${comp_version}/vars -f \
        && rm ${MICRO_PATH}/${component_config} -rf && rm ${MICRO_PATH}/${component_config_version} -rf \
        && chmod 500 ${MICRO_PATH}/${comp_version}
        if [[ $? -ne 0 ]];then
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] assemble ${component_name} failed."
            return 1
        fi
        return 0
    else
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to assemble ${component_name}"
        tar --no-same-owner -zxmf ${MICRO_PATH}/${comp} && chmod o-rwx -R ${MICRO_PATH}/${comp_version} \
        && chmod 500 ${MICRO_PATH}/${comp_version}
        if [[ $? -ne 0 ]];then
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] assemble ${component_name} failed."
            return 1
        fi
        return 0
    fi
}

function execute_component_action()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Params: $1 $2 $3 $4"
    task_type=$1
    component=$2
    micro_service=$3
    component_name=$4
    token=$5
    if [ "${component}X" == "X" -o "${component_name}X" == "X" ];then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] param is not right."
        return 1
    fi
    if [[ "${task_type}"x != "${TASK_TYPE_UPGRADE}"x ]]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ${task_type} is not right."
        return 1
    fi
    for action in ${upgrade_actions[@]}; do
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] current execute step id  ${component_name} ${action}."
        result=`get_upgrade_step_from_file ${component_name} ${component_name}_${action}`
        if [[ "${result}"x == "${ACTION_SUCCESS}"x  ]]; then
            log ${SCRIPTNAME} ${FUNCNAME} "[Line:${LINENO}][INFO ] The ${action} of the ${component} component was successfully executed, no need to retry."
        else
            platform=`uname -m`
            if [[ "X${platform}" == "X${X86_64}" ]];then
                component_version=`get_spec_component_info ${component_name} curVersionX86`
            else
                component_version=`get_spec_component_info ${component_name} curVersionARM`
            fi
            if [[ "X${component_version}" == "X" ]];then
                log ${SCRIPTNAME} ${FUNCNAME} "[Line:${LINENO}][ERROR ] The ${component_version} is empty"
                return 1
            fi
            if [[ "X${component}" == "X${component_name}_${component_version}" ]];then
                log ${SCRIPTNAME} ${FUNCNAME} "[Line:${LINENO}][INFO ] The ${action} of the ${component} is same."
                touch_upgrade_step_to_file ${component_name} ${component_name}_${action} ${ACTION_SUCCESS}
                continue
            fi
            if [ "${action}"x == "upgrade"x ];then
                local count=0
                while true; do
                    let count+=1
                    download_pkg ${component_name} ${token}
                    ret=$?
                    if [ ${ret} -eq 0 ];then
                        break
                    fi
                    if [ ${count} -gt 3 ]; then
                        break
                    fi
                    sleep 5
                done
                if [[ ${ret} -ne 0 ]];then
                    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] download_pkg ${component_name} failed."
                    rm -rf ${MICRO_PATH}/${component_name}_${component_version}
                    rm -f ${MICRO_PATH}/${component_name}_${component_version}.tar.gz
                    return 1
                fi
            fi
            local pkg="${component_name}_${component_version}"
            if [ "${action}"x == "offline"x -o "${action}"x == "stop"x ]; then
                pkg=${component}
            elif [ "${action}"x == "start"x -o "${action}"x == "check_status"x -o "${action}"x == "online"x ]; then
                pkg=${component_name}_${component_version}
            fi
            shell_command="action/appctl.sh ${action}"
            if [[ "${action}"x == "${PRECHECK}"x  ]]; then
                shell_command="${UPGRADE_ROLLBACK_SCRIPT}${PRECHECK}/${micro_service}/${component_name}.sh"
            else
                cd ${MICRO_PATH}/${pkg}
            fi
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] start execute ${action} of ${pkg} component:${shell_command}"
            local count=0
            while true; do
                let count+=1
                bash ${shell_command} >> ${LOG_FILE} 2>&1
                ret=$?
                if [ ${ret} -eq 0 ];then
                    break
                fi
                log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] start execute ${action} of ${pkg} component:${shell_command} failed"
                if [ ${count} -gt 6 ]; then
                    break
                fi
                sleep 10
            done
            if [[ ${ret} -ne 0 ]]; then
                log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][ERROR ] Failed to execute ${action} of ${pkg} component."
                 if [[ "${action}"x == "upgrade"x ]]; then
                    rm -rf ${MICRO_PATH}/${component_name}_${component_version}.tar.gz
                    rm -rf ${MICRO_PATH}/${component_name}_${component_version}
                 fi
                return 1
            else
                log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Succeed to execute ${action} of ${pkg} component."
                if [[ "${action}"x == "upgrade"x ]]; then
                    rm -rf "${MICRO_PATH}/${component_name}_"*.tar.gz
                    find ${MICRO_PATH} -name "${component_name}_*" | grep -v "${pkg}" | while read oldcomp; do
                        rm -rf "${oldcomp}"
                    done
                fi
                touch_upgrade_step_to_file ${component_name} ${component_name}_${action} ${ACTION_SUCCESS}
            fi
        fi
    done
    return $?
}

function execute_upgrade_rollback_action()
{
    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to execute_upgrade_rollback_action."
    local token=$1
    task_type=`cat "${SCENES_INFO}" | grep scenes | awk -F '=' '{print $2}'`
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start executing the ${task_type} task."
    if [ "${task_type}"x != "upgrade"x -a "${task_type}"x != "rollback"x ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Scene error ${task_type}."
        return 1
    fi
    for comp in ${components[@]} ; do
        micro_service=`echo ${comp} | awk -F ':' '{print $1}'`
        component_name=`echo ${comp} | awk -F ':' '{print $2}'`
        component=`cat ${CURRENT_PATH}/component_old_version.ini | grep -v grep |grep -v "*.tar.gz" |  grep "${component_name}_"| head -n 1`
        if [ -z "${component}" ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ${task_type} ${component_name} components not exist."
            return 1
        fi
		backup_component "${component}" "${component_name}"
        execute_component_action "${task_type}" "${component}" "${micro_service}" "${component_name}" "${token}"
        ret=$?
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] execute ${task_type} of ${component_name} component:${ret}."
        if [ ${ret} -ne 0 ]; then
            return $ret
        fi
    done
    return $?
}

function backup_component()
{
    component_dir=$1
	component_name=$2
	backup_dir="/opt/fusionstorage/deploymanager/compute_backup/${ARCH}"
	log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] start backup component:${component_dir}."
	if [ ! -d ${backup_dir} ];then
	    mkdir -p ${backup_dir}
		chmod 777 ${backup_dir}
	fi
	component_old=`ls ${backup_dir} | grep ^${component_name}_* | sort -r | head -1`
    if [ -z "${component_old}" ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] ${component_name} component is need backup."
		cd ${MICRO_PATH}
	    tar -zcvf ${component_dir}.tar.gz ${component_dir}
	    \cp -f ${component_dir}.tar.gz  ${backup_dir}
	    rm -rf ${component_dir}.tar.gz
	    cd -
	    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Succeed to backup component:${component_dir}."
    else
	    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] ${component_name} component is exist. no need backup."
    fi
}

function is_valid_scenes()
{
    local scenes_param=$1
    for var in ${SCENES[@]}
    do
        if [ "${var}"x == "${scenes_param}"x ]; then
            return 0
        fi
    done
    return 1
}

function config_mark_info()
{
    task_type=$1
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start generating mark information."
    is_valid_scenes "${task_type}"
    if [[ $? -ne 0 ]];then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Unknown task type:${task_type}."
        return 1
    fi

    if [ ! -d "/opt/fusionstorage" ]; then
        mkdir /opt/fusionstorage
        chmod 755 /opt/fusionstorage
    fi
    echo "[deploy]" > ${SCENES_INFO}
    echo "scenes=${task_type}" >> ${SCENES_INFO}
    chmod 644 ${SCENES_INFO}
    if [ "${task_type}"x == "${TASK_TYPE_UPGRADE}"x ]; then
        echo "[upgrade_status]" > ${UPGRADE_STATUS_INFO}
        echo "in_upgrade_or_rollback=true" >> ${UPGRADE_STATUS_INFO}
        echo "[rollback_status]" >> ${UPGRADE_STATUS_INFO}
        echo "in_rollback=false" >> ${UPGRADE_STATUS_INFO}
    elif [ "${task_type}"x == "${TASK_TYPE_ROLLBACK}"x ]; then
        echo "[upgrade_status]" > ${UPGRADE_STATUS_INFO}
        echo "in_upgrade_or_rollback=true" >> ${UPGRADE_STATUS_INFO}
        echo "[rollback_status]" >> ${UPGRADE_STATUS_INFO}
        echo "in_rollback=true" >> ${UPGRADE_STATUS_INFO}
    fi
    chmod 644 ${UPGRADE_STATUS_INFO}

    return 0
}

function restore_mark_info()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start restoring mark information."
    rm -f ${SCENES_INFO}
    echo "[upgrade_status]" > ${UPGRADE_STATUS_INFO}
    echo "in_upgrade_or_rollback=false" >> ${UPGRADE_STATUS_INFO}
    echo "[rollback_status]" >> ${UPGRADE_STATUS_INFO}
    echo "in_rollback=false" >> ${UPGRADE_STATUS_INFO}
    chmod 644 ${UPGRADE_STATUS_INFO}

    return 0
}

function postcheck()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start post checking components status."
    for comp in ${components[@]} ; do
        micro_service=`echo ${comp} | awk -F ':' '{print $1}'`
        component_name=`echo ${comp} | awk -F ':' '{print $2}'`
        component_dir=`ls -t "${MICRO_PATH}" | grep -v grep |grep -v "*.tar.gz" |grep "${component_name}_"|head -n 1`
        if [ -z "${component_dir}" ]; then
            continue
        fi
        execute_oamx_script ${micro_service} ${component_name} ${POSTCHECK}
        ret=$?
        if [ ${ret} -ne 0 ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] post check ${component_name} failed."
            return $ret
        fi
    done
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] post check successful."
    return 0
}

function notify_complete()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start notifying components upgrade complete."
    for comp in ${components[@]} ; do
        micro_service=`echo ${comp} | awk -F ':' '{print $1}'`
        component_name=`echo ${comp} | awk -F ':' '{print $2}'`
        component_dir=`ls -t "${MICRO_PATH}" | grep -v grep |grep -v "*.tar.gz" |grep "${component_name}_"|head -n 1`
        if [ -z "${component_dir}" ]; then
            continue
        fi
        execute_oamx_script ${micro_service} ${component_name} ${NOTIFY_COMPLETE}
        ret=$?
        if [ ${ret} -ne 0 ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] notify ${component_name} upgrade complete failed."
            return $ret
        fi
    done
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] notify components upgrade complete successful."
    return 0
}

function clear()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start cleanup."

    restore_mark_info

    rm -rf /opt/fusionstorage/repository/components
    #3、卸载组件：fdsa、plog
}

function platform()
{
    PLATFORM=`uname -m`
}

function os_version()
{
    os_info=$(cat /etc/os-release | grep "Ubuntu")
    if [ -n "${os_info}" ];then
        OS_VERSION="Ubuntu"
    fi
}

function check_already_installed()
{
    os_version
    if [ "${OS_VERSION}"x == "Ubuntu"x ];then
        dpkg -l |grep clouda >> ${LOG_FILE} 2>&1
        if [ $? -eq 0 ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] Clouda installation completed."
           return 1
        fi
    else
        rpm -qa | grep clouda >> ${LOG_FILE} 2>&1
        if [ $? -eq 0 ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] Clouda installation completed."
            return 1
        fi
    fi
    return 0
}

function task_isolation()
{
    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to task_isolation."
    task_type=$1
    if [ "${task_type}"x == "${TASK_TYPE_PREINSTALL}"x ]; then
        check_already_installed
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] The software is installed, please perform the cleanup before installing!" && return 1
    fi
    if [ "${task_type}"x == "${TASK_TYPE_UPGRADE}"x ]; then
        if [ -f "${PREINSTALL_MAKR}" ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] The current node has not joined the cluster. Please join the cluster before upgrading!"
            echo "==================   If you need to use the new version, please uninstall the old version and reinstall the new version.   =================="
            return 1
        fi
        check_already_installed
        if [ $? -eq 0 ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] The node is not installed and cannot be upgraded."
            return 1
        fi
        if [ ! -f ${SCENES_INFO} ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] The ${SCENES_INFO} is empty."
            return 0
        fi
        scenes=`cat "${SCENES_INFO}" | grep scenes | awk -F '=' '{print $2}'`
        if [ "${scenes}"x == "${TASK_TYPE_ROLLBACK}"x ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] in rollback state,unable to perform upgrade currently."
            return 1
        fi
    fi
    if [ "${task_type}"x == "${TASK_TYPE_ROLLBACK}"x ]; then
        task_type=`cat "${SCENES_INFO}" | grep scenes | awk -F '=' '{print $2}'`
        if [ "${task_type}"x != "${TASK_TYPE_UPGRADE}"x -a "${task_type}"x != "${TASK_TYPE_ROLLBACK}"x ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] in ${task_type} state,unable to perform rollback currently."
            return 1
        fi
    fi
    return 0
}

function is_ip_v4()
{
    local host=$1
    echo "${host}" | grep -Eq "${IP_V4_REGULAR}"
    return $?
}

function is_ip_v6()
{
    local host=$1
    echo "${host}" | grep -Eq "${IP_V6_REGULAR}"
    return $?
}

function is_valid_ip()
{
    local ip=$1
    is_ip_v4 ${ip}
    if [ $? -eq 0 ]; then
        loop_ip="127.0.0.1"
        return 0
    fi
    if [ x"$1" == x"::" -o x"$1" == x"0:0:0:0:0:0:0:0" ]; then
        return 1
    fi
    is_ip_v6 ${ip}
    if [ $? -eq 0 ]; then
        loop_ip="::1"
        return 0
    fi
    return 1
}

function is_valid_action()
{
    local action_param=$1
    for var in ${INSTALL_ACTION[@]}
    do
        if [ "${var}"x == "${action_param}"x ]; then
            return 0
        fi
    done
    return 1
}

function is_valid_role()
{
    local role_param=$1
    if [ "${role_param}"x == "${COMPUTE}"x -o "${role_param}"x == "${DPC_COMPUTE}"x ]; then
        return 0
    else
        echo "For technical reasons, only pre-installation of compute or dpc_compute nodes is supported."
        return 1
    fi

    for var in ${ROLE[@]}
    do
        if [ "${var}"x == "${role_param}"x ]; then
            return 0
        fi
    done
    return 1
}

function clear_component()
{
    component=$1
    component_dirs=`ls ${MICRO_PATH} | grep -v tar.gz | grep ^${component}_[0-9].* | sort -r`
    if [ -z "${component_dirs}" ]; then
        continue
    fi
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Files match ${component}: ${component_dirs}."
    for component_dir in ${component_dirs[@]} ; do
        component_dir="${MICRO_PATH}/${component_dir}"
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] Try do clear with ${component_dir}."
        if [ -d $component_dir -a -f "$component_dir/action/appctl.sh" ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] bash ${component_dir}/action/appctl.sh stop 1"
            bash ${component_dir}/action/appctl.sh stop 1>> ${LOG_FILE} 2>&1
            ret=$?
            if [ $ret -ne 0 ]; then
                log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Stop ${component_dir} failed."
            fi
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] bash ${component_dir}/action/appctl.sh uninstall 1"
            bash ${component_dir}/action/appctl.sh uninstall 1>> ${LOG_FILE} 2>&1
            ret=$?
            if [ $ret -ne 0 ]; then
                log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Uninstall $component failed."
                return $ret
            fi
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] Clear success, try rm -rf ${MICRO_PATH}/${component}_[0-9].*"
            chattr -a ${MICRO_PATH}/${component}_[0-9].*
            rm -rf ${MICRO_PATH}/${component}_[0-9].* #upgrade scene, multiple package should be deleted
            break #first package clear success, no need do clear in other version packages
        else
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][WARN ] ${component_dir}/action/appctl.sh not exist, continue."
        fi
    done
    return $?
}

function print_result()
{
    result="$@"
    echo "   ###############################"
    echo "   #  --  ${result}  --  #"
    echo "   ###############################"
    is_success=`echo "${result}" | grep "success"`
    if [ -z "${is_success}" ]; then
        return 1
    fi
}

function generate_component_info()
{
    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to generate_component_info."
    local token=$1
    local url_path="api/v2/upgrade/compute_component?refresh=off"
    local float_ip=`cat ${CLOUDA_CONFIG} | grep -w cloudm_float_ip | awk -F '=' '{print $2}'`
    is_ip_v6 ${float_ip}
    if [[ $? -eq 0 ]];then
        url_path="https://[${float_ip}]:6098/${url_path}"
    else
        url_path="https://${float_ip}:6098/${url_path}"
    fi
    for((i=1;i<=3;i++))
    do
        log  ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][INFO ] begin to generate_component_info time ${i} ${url_path}. "
        json_body=`curl -sgkX GET -H "Content-type: application/json" -H "x-auth-token: ${token}" --noproxy "*" --retry 3 --connect-timeout 60 ${url_path}`
        json_body=`echo ${json_body} | awk -F "[" '{print $2}'| awk -F "]" '{print $1}'`
        if [[ "X${json_body}" =~ "sysconfig" ]];then
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Info ] generate_component_info ok ${json_body}."
            echo ${json_body} > ${COMPUTE_UPGRADE_JSON}
            chmod 600 ${COMPUTE_UPGRADE_JSON}
            return 0
        else
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] generate_component_info empty."
            sleep 5
        fi
    done
    log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] generate_component_info retry 3 times, failed."
    return 1
}

function get_upgrade_step_from_file()
{
    local filepath="${CURRENT_PATH}/compute_upgrade.ini"
    local section=$1
    local key=$2
    if [[ ! -f ${filepath} ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] ${filepath} not exist."
        echo ""
    else
        local value=$(cat ${filepath} |grep -Po '(?<=^'${key}'=).*' |awk 'NR==1{print}')
        if [[ "X${value}" != "X" ]];then
            echo "${value}"
        else
            echo ""
        fi
    fi
}

function touch_upgrade_step_to_file()
{
    local filepath="${CURRENT_PATH}/compute_upgrade.ini"
    local section=$1
    local key=$2
    local value=$3
    if [[ ! -f ${filepath} ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] ${filepath} not exist."
        return 1
    else
        local step_key=$(cat ${filepath} |grep -P '(?<=^'${key}'=).*' |awk -F = '{print $1}')
        if [[ "X${step_key}" == "X" ]];then
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] ${section} ${key} not exist."
            return 1
        else
            sed -i "s/^${step_key}=.*/${step_key}=${value}/g" ${filepath}
            if [[ $? -ne 0 ]];then
                log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] touch ${step_key} ${value} Error."
                return 1
            fi
            log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Info ] touch ${step_key} ${value} success."
            return 0
        fi
    fi
}

function get_spec_component_info()
{
    local filepath="${CURRENT_PATH}/compute_upgrade.json"
    local component=$1
    local section=$2
    if [[ ! -f ${filepath} ]];then
        log ${SCRIPTNAME} $FUNCNAME "[Line:${LINENO}][Error ] ${filepath} not exist."
        echo ""
    else
        local value=$(cat ${filepath} |tr  \{\} \\n |grep -w ${component} | tr , \\n|grep -w ${section}|awk -F ":" '{print $2}'|awk -F "\"" '{print $2}')
        if [[ "X${value}" != "X" ]];then
            echo "${value}"
        else
            echo ""
        fi
    fi
}

function set_failed()
{
    local key=$1
    if [[ "X${key}" != "X" ]];then
        touch_upgrade_step_to_file UPGRADE_STATUS "${key}" ${ACTION_FAILED}
    fi
    touch_upgrade_step_to_file UPGRADE_STATUS "status" "failed"
	if [ -d "/opt/fusionstorage/deploymanager/compute_backup" ];then
	    rm -rf /opt/fusionstorage/deploymanager/compute_backup/upgrade_*
	    touch ${status_failed}
		chmod 777 ${status_failed}
	fi
	
}