#!/bin/bash

CURRENT_PATH=$(cd "$(dirname "$0")"; pwd)
SCRIPTNAME=$(basename $0)
source ${CURRENT_PATH}/common.sh
readonly PREINSTALL_PATH=$(realpath "${CURRENT_PATH}/../deploymanager/preinstall/")
MANIFEST=$(realpath "${CURRENT_PATH}/../manifest.yml")
PARAM_CFG="${1}"
#全局参数
sense="preinstall"
action="reboot_init"
role="compute"
ha_role="primary"
ha_mode="double"
float_ip="192.168.57.3"
active_ip="192.168.57.3"
standby_ip="192.168.57.3"
loop_ip="127.0.0.1"
local_manager_ip="192.168.57.3"
local_storage_ip="192.168.57.3"

function parsing_preinstall_param()
{
    # 获取浮动IP
    server_float_ip=`grep "server_float_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}' | sed s/[[:space:]]//g`
    is_valid_ip "${server_float_ip}"
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server float ip error : ${server_float_ip}." && return 1

    # 获取主节点IP
    server_active_ip=`grep "server_active_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'  | sed s/[[:space:]]//g`
    if [ -z "${server_active_ip}" ];then
        server_active_ip=${server_float_ip}
        ha_mode="single"
    else
        is_valid_ip "${server_active_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server active ip error : ${server_active_ip}." && return 1
    fi

    # 获取备节点IP
    server_standby_ip=`grep "server_standby_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}' | sed s/[[:space:]]//g`
    if [ -z "${server_standby_ip}" ];then
        server_standby_ip=${server_float_ip}
        ha_mode="single"
    else
        is_valid_ip "${server_standby_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server standby ip error : ${server_standby_ip}." && return 1
    fi

    # 校验主备参数合法性
    if [ "${server_active_ip}"x == "${server_standby_ip}"x -a "${server_active_ip}"x != "${server_float_ip}"x ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] The active and standby node IPs cannot be the same."
        return 1
    fi

    # 获取本地管理IP
    local_manager_ip=`grep "local_manager_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}' | sed s/[[:space:]]//g`
    is_valid_ip "${local_manager_ip}"
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] local manager ip error : ${local_manager_ip}." && return 1
    local ret=$(ip address | grep -w "${local_manager_ip}")
    if [ -z "${ret}" ];then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ip ${local_manager_ip} is not bind local."
        return 2
    fi

    # 获取本地存储IP
    local_storage_ip=`grep "local_storage_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}' | sed s/[[:space:]]//g`
    if [ -z "${local_storage_ip}" ];then
        local_storage_ip=${local_manager_ip}
    else
        is_valid_ip "${local_storage_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] local storage ip error : ${local_storage_ip}." && return 1
        local ret=$(ip address | grep -w "${local_storage_ip}")
        if [ -z "${ret}" ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ip ${local_storage_ip} is not bind local."
            return 2
        fi
    fi
    return 0
}

function parsing_reboot_init_param()
{
    # 获取本地管理IP
    local_manager_ip=`grep "local_manager_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    is_valid_ip "${local_manager_ip}"
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] local manager ip error : ${local_manager_ip}." && return 1
    local ret=$(ip address | grep -w "${local_manager_ip}")
    if [ -z "${ret}" ];then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ip ${local_manager_ip} is not bind local."
        return 2
    fi

    # 获取本地存储IP
    local_storage_ip=`grep "local_storage_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    if [ -z "${local_storage_ip}" ];then
        local_storage_ip=${local_manager_ip}
    else
        is_valid_ip "${local_storage_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] local storage ip error : ${local_storage_ip}." && return 1
        local ret=$(ip address | grep -w "${local_storage_ip}")
        if [ -z "${ret}" ];then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] ip ${local_storage_ip} is not bind local."
            return 2
        fi
    fi
    return 0
}

function parsing_make_iso_param()
{
    # 获取浮动IP
    server_float_ip=`grep "server_float_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    is_valid_ip "${server_float_ip}"
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server float ip error : ${server_float_ip}." && return 1

    # 获取主节点IP
    server_active_ip=`grep "server_active_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    if [ -z "${server_active_ip}" ];then
        server_active_ip=${server_float_ip}
        ha_mode="single"
    else
        is_valid_ip "${server_active_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server active ip error : ${server_active_ip}." && return 1
    fi

    # 获取备节点IP
    server_standby_ip=`grep "server_standby_ip=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    if [ -z "${server_standby_ip}" ];then
        server_standby_ip=${server_float_ip}
        ha_mode="single"
    else
        is_valid_ip "${server_standby_ip}"
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] server standby ip error : ${server_standby_ip}." && return 1
    fi

    # 校验主备参数合法性
    if [ "${server_active_ip}"x == "${server_standby_ip}"x -a "${server_active_ip}"x != "${server_float_ip}"x ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] The active and standby node IPs cannot be the same."
        return 1
    fi
    return 0
}

function parsing_param()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start parsing parameter file."
    if [ ! -f "${PARAM_CFG}" ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Parameter file does not exist, please specify the correct parameter file."
        return 1
    fi

    line=`wc -l "${PARAM_CFG}" | awk '{print $1}'`
    if [[ "${line}" -gt 20 ]];then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Parameter file is too large, please reduce parameter file size."
        return 1
    fi

    # 获取节点角色
    role=`grep "role=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    is_valid_role "${role}"
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] role error : ${role}." && return 1
    if [ "${role}"x == "${COMPUTE}"x ]; then
        type=`cat ${MANIFEST} | grep "vbs" | sed 's/ //g' | sed 's/%//g'`
        if [ -z "${type}" ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] package error : ${role}." && return 1
        fi
    elif [ "${role}"x == "${DPC_COMPUTE}"x ]; then
        type=`cat ${MANIFEST} | grep "dpc" | sed 's/ //g' | sed 's/%//g'`
        if [ -z "${type}" ]; then
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] package error : ${role}." && return 1
        fi
    fi

    sense=`grep "sense=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
    if [ "${sense}"x == "preinstall"x ];then
        # preinstall 预装场景
        parsing_preinstall_param
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] parsing preinstall param error." && return 1
    elif [ "${sense}"x == "ram_os"x ];then
        # ram_os
        action=`grep "action=" "${PARAM_CFG}" | awk -F '=' '{print $2}'`
        if [ "${action}"x == "reboot_init"x ];then
            # reboot_init 获取下电重启参数
            parsing_reboot_init_param
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] parsing reboot init_param error." && return 1
        elif [ "${action}"x == "make_iso"x ];then
            # make_iso 获取制作镜像参数
            parsing_make_iso_param
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] parsing make iso param." && return 1
        else
            log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] action error : ${action}."
            return 1
        fi
    else
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] sense error : ${sense}."
        return 1
    fi

    float_ip=${server_float_ip}
    active_ip=${server_active_ip}
    standby_ip=${server_standby_ip}
    return 0
}

function generate_file()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start generating configuration file."
    if [ ! -f ${HA_PROPERTIES_PATH} ]; then
        mkdir -p /home
        touch ${HA_PROPERTIES_PATH}
    fi
    chmod 640 ${HA_PROPERTIES_PATH}

    echo ha_role=${ha_role} > ${HA_PROPERTIES_PATH}
    echo ha_mode=${ha_mode} >> ${HA_PROPERTIES_PATH}
    echo float_ip_for_ha=${float_ip} >> ${HA_PROPERTIES_PATH}

    echo service_float_ip=${float_ip} >> ${HA_PROPERTIES_PATH}
    echo service_local_ip=${active_ip} >> ${HA_PROPERTIES_PATH}
    echo service_remote_ip=${standby_ip} >> ${HA_PROPERTIES_PATH}

    echo manager_float_ip=${float_ip} >> ${HA_PROPERTIES_PATH}
    echo manager_local_ip=${active_ip} >> ${HA_PROPERTIES_PATH}
    echo manager_remote_ip=${standby_ip} >> ${HA_PROPERTIES_PATH}

    echo loop_ip=${loop_ip} >> ${HA_PROPERTIES_PATH}
    echo om_ip=${local_manager_ip} >> ${HA_PROPERTIES_PATH}
    echo local_manager_ip=${local_manager_ip} >> ${HA_PROPERTIES_PATH}
    echo local_storage_ip=${local_storage_ip} >> ${HA_PROPERTIES_PATH}

    return 0
}

function set_mark()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start generating scene identification file."
    if [ ! -d "/opt/fusionstorage" ]; then
      mkdir /opt/fusionstorage
      chmod 755 /opt/fusionstorage
    fi
    DEPLOY_INI=/opt/fusionstorage/deploy.ini
    echo "[deploy]" > ${DEPLOY_INI}
    echo "scenes="${sense} >> ${DEPLOY_INI}
    chmod 644 ${DEPLOY_INI}
    return 0
}

function install()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start software pre-installation."
    bash ${PREINSTALL_PATH}/install.sh ${role} ${HA_PROPERTIES_PATH}
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] install failed." && return 1
    local CLOUDA_CONFIG_FILE="/opt/fusionstorage/deploymanager/clouda/conf/cloudagent.ini"
    local DSWARE_CONFIG_FILE="/opt/fusionstorage/agent/conf/dsware_manager_ip"
    sed -i "s/^cloudm_local_ip=.*/cloudm_local_ip=${active_ip}/g" ${CLOUDA_CONFIG_FILE}
    sed -i "s/^cloudm_remote_ip=.*/cloudm_remote_ip=${standby_ip}/g" ${CLOUDA_CONFIG_FILE}
    sed -i "s/^dsware_manager_ip=.*/dsware_manager_ip=${float_ip}/g" ${DSWARE_CONFIG_FILE}
    return 0
}

function config_oamu()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start configuring OAM-U."
    local LOCALIP=${local_manager_ip}
    local FMA_CONF=/opt/dfv/oam/fma/conf/servercfg.cfg
    local PMA_CONF=/opt/dfv/oam/oam-u/pma/conf/servercfg.cfg
    local NMA_CONF=/opt/dfv/oam/oam-u/nma/conf/servercfg.cfg
    local OAMU_IPTABLES_CONF=/opt/dfv/oam/oam-u/data/iptables
    local DECEPTIVE_IP=127.0.0.111

    is_ip_v6 "${LOCALIP}"
    if [ $? -eq 0 ];then
        DECEPTIVE_IP=fff1:1::eeee
        OAMU_IPTABLES_CONF=/opt/dfv/oam/oam-u/data/ip6tables
    fi

    sed -i "s/^LOCALIP=.*/LOCALIP=${LOCALIP}/g" $FMA_CONF
    sed -i "s/^LOCALIP=.*/LOCALIP=${LOCALIP}/g" $PMA_CONF
    sed -i "s/^LOCALIP=.*/LOCALIP=${LOCALIP}/g" $NMA_CONF

    sed -i "s/${DECEPTIVE_IP}\//${LOCALIP}\//g" ${OAMU_IPTABLES_CONF}

    fma_pid=$(ps -axu | grep fma | grep -v grep |awk '{print $2}')
    kill -9 ${fma_pid}

    bash /opt/dfv/oam/oam-u/pma/scripts/pma_ctl.sh restart >>/dev/null 2>&1 &
    bash /opt/dfv/oam/oam-u/pma/scripts/perf_ctl.sh restart >>/dev/null 2>&1 &

    bash /opt/dfv/oam/oam-u/nma/scripts/nma_ctl.sh restart >>/dev/null 2>&1 &

    return 0
}

function config_fsm()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start configuring FSM."
    sed -i "s/node_manager_ip=.*/node_manager_ip=${local_manager_ip}/" /etc/node_manager_ip
    rm -rf /opt/fusionstorage/agent/tmp/agent_conf_restore.ini
    ps -ef | grep dsware_agent | grep -v grep | awk '{print $2}' | while read pid; do kill -9 ${pid}; done;
    return 0
}

function config_deploymanager()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start configuring DeployManager."
    local NETWORK_CONFIG="/opt/network/network_config.ini"
    local CLOUDA_CONFIG_FILE="/opt/fusionstorage/deploymanager/clouda/conf/cloudagent.ini"
    om_ip=${local_manager_ip}
    if [ -z "$om_ip" ]; then
        om_ip=$(cat ${CLOUDA_CONFIG_FILE} | grep clouda_om_ip | awk -F '=' '{print $2}')
    fi
    SN=$(dmidecode -t 1 | grep Serial | awk '{printf $3}')
    suffix=$(echo ${om_ip//./-})
    suffix=$(echo ${suffix//:/-})
    clouda_agent_sn=${SN}"-"${suffix}

    sed -i "s/^om_ip=.*/om_ip=${om_ip}/g" ${NETWORK_CONFIG}
    sed -i "s/^clouda_om_ip=.*/clouda_om_ip=${om_ip}/g" ${CLOUDA_CONFIG_FILE}

    sed -i "s/^sn=.*/sn=${clouda_agent_sn}/g" ${NETWORK_CONFIG}
    sed -i "s/^clouda_agent_sn=.*/clouda_agent_sn=${clouda_agent_sn}/g" ${CLOUDA_CONFIG_FILE}

    kill -9 $(ps -ef|grep /opt/fusionstorage/deploymanager/clouda/bin/cloud_agent|grep -v grep|awk '{print $2}') >>/dev/null 2>&1
    rm -rf /tmp/clouda/clouda.pid
    bash /opt/fusionstorage/deploymanager/clouda/bin/cloudamonitor.sh >>/dev/null 2>&1 &
    return 0
}

function config()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start configuring components."
    config_deploymanager
    config_oamu
    config_fsm
    return 0
}

function execute()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start execution."
    if [ "${sense}"x == "preinstall"x ];then
        generate_file
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] generate file failed." && return 1
        set_mark
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] set mark failed." && return 1
        install
        [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] install failed." && return 1
    else
        if [ "${action}"x == "reboot_init"x ];then
            config
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] config failed." && return 1
        else
            generate_file
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] generate file failed." && return 1
            set_mark
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] set mark failed." && return 1
            install
            [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] install failed." && return 1
        fi
    fi
    return 0
}

function main()
{
    log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][INFO ] Start the installation."
    task_isolation ${TASK_TYPE_PREINSTALL}
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Cannot perform ${TASK_TYPE_PREINSTALL} operation!" && return 1
    parsing_param
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Parsing parameter file failed." && return 1
    execute
    [[ $? -ne 0 ]] && log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] execute failed." && return 1
    return 0
}

{
    flock -n 3
    if [ $? -eq 1 ]; then
        log $SCRIPTNAME $FUNCNAME "[Line:${LINENO}][ERROR ] Another task is being executed, try it later."
        exit 1
    fi
    echo "$$" > ${LOCKFILE}

    main
    if [ $? -eq 0 ]; then
        print_result "install   success"
    else
        print_result "install    failed"
    fi

    flock -u 3
} 3<>${LOCKFILE}
