#!/bin/bash

########################################################################################
# 描述：检查Pool可用容量（FSA）
# 错误码：
#         0 正常
#         1 可用容量过低（小于10%）
#         11 该节点未部署MDC
#         12 管理集群异常，无法进行检测（未能获取到主MDC的ip）
########################################################################################



RST_PATH="/opt/dfv/oam/public/result/inspect/FusionStorage_index"
SCRIPT_NAME=$(basename $0)
INSPECT_ITEM=${SCRIPT_NAME%.sh}
RST_NAME="${INSPECT_ITEM}.sh.rst"
RST_FILE=${RST_PATH}/${RST_NAME}
LOG_FILE="/var/log/dsware/inspect.log"
EDS_LOG_FILE="/var/log/dsware/eds/run/dsware_eds_50.log"
EDS_F_LOG_FILE="/var/log/dsware/eds-f/run/dsware_java_11.log"
EDS_SERVER_FILE="/opt/dsware/eds/eds/conf/ebs_server.conf"
EDS_F_SERVER_FILE="/opt/dsware/eds/eds-f/conf/ebs_server.conf"

IS_FSA_NODE="no"
IS_801_VERSION="yes"
ROLE=""

readonly PRODUCT_VERSION="/opt/dfv/oam/public/conf/product.ini"
readonly NETWORK_CONF="/opt/network/network_config.ini"
readonly DEPLOY_VERSION="/opt/fusionstorage/deploymanager/version"
readonly HA_BIN="/opt/dfv/oam/oam-u/ha/ha/module/hacom/bin/ha.bin"
readonly VIDEO_SCENES="/opt/fusionstorage/storage_resource_info.ini"

NORMAL=0
INFO_HAS_NO_STORAGE=2
NODE_NO_MDC=3
ERROR_GET_MDC_IP=12


if [ ! -d "/var/log/dsware" ]; then
    mkdir -p /var/log/dsware
    chmod 750 /var/log/dsware
fi

if [ ! -f "${LOG_FILE}" ]; then
    touch ${LOG_FILE}
    chmod 640 ${LOG_FILE}
fi

if [ ! -d "${RST_PATH}" ]; then
    mkdir -p "${RST_PATH}"
    chmod 750 "/opt/dfv/oam/public/result/inspect/"
    chmod 750 "${RST_PATH}"
fi
    
if [ ! -f "${RST_FILE}" ]; then
    touch "${RST_FILE}"
    chmod 640 "${RST_FILE}"
fi
# 涉及到801，最小化改动
chown -Rh root:oam /opt/dfv/oam/public/result
chmod 750 /opt/dfv/oam/public/result
chmod 750 /opt/dfv/oam/public/result/inspect

function log()
{
    local logInfo=$1
    local functionName=$2
    local scriptName=$3

    echo "[""$(date +'%F %T')""]""[""${logInfo}""]""[""${functionName}""]""[""${scriptName}""]"  >> "${LOG_FILE}" 2>/dev/null

    return 0
}

function output_json() 
{
    local key=$1
    local value=$2
    echo "[{\"${key}\": [${value}]}]" > "${RST_FILE}" 2>/dev/null

    return 0
}

function is_agent_node()
{
    crontab -l | grep -v "^#" | grep dswareAgent_crontab_chk.sh >/dev/null 2>&1
    if [ $? -eq 0 ];then
        IS_AGENT_NODE="yes"
    else
        local node_role=$(cat /opt/network/network_config.ini | grep -w '^role' | awk -F'=' '{print $2}')
        for role in ${FSA_ROLE};
        do
            if [ -n "`echo $node_role | grep -i $role`" ];then
                IS_AGENT_NODE="yes"
                return 0
            fi
        done
        IS_AGENT_NODE="no"
    fi

    return 0
}

function check_801_version()
{
    version=`cat ${PRODUCT_VERSION} | grep "product_version" |  awk -F '=' '{print$2}'`
    current_version=`echo $version | cut -b 1-5`

    if [[ $current_version = "8.0.1" ]];then
        IS_801_VERSION="yes"
    else
        IS_801_VERSION="no"
    fi
    log "INFO: this node version is ${version}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}

    return 0
}

function check_diagnose_valid()
{
    echo -e "attach 273\nexit" |diagnose_usr --set-cli --auto | grep "large than 4"
    if [ $? -eq 0 ];then
        # 4:已有4个用户登录diagnose
        log "INFO: client process num cannot large than 4. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        output_json "${INSPECT_ITEM}-4"
        echo "result=1"
        return 4;
    fi
    echo -e "attach 273\nexit" |diagnose_usr --set-cli --auto|grep "Error attach"
    if [ $? -eq 0 ];then
        if [ -f "${EDS_SERVER_FILE}" ];then
            # 3:节点EDS进程故障
            local eds=`grep -ain "fsa_service_id" $EDS_SERVER_FILE`
            if [[ $eds != "" ]];then
                log "INFO: node fault. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
                output_json "${INSPECT_ITEM}-3"
                echo "result=${NODE_NO_MDC}"
                return 3;
            fi
        fi
        if [ -f "${EDS_F_SERVER_FILE}" ];then
            # 3:节点EDS进程故障
            local eds_f=`grep -ain "fsa_service_id" $EDS_F_SERVER_FILE`
            if [[ $eds_f != "" ]];then
                log "INFO: node fault. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
                output_json "${INSPECT_ITEM}-3"
                echo "result=${NODE_NO_MDC}"
                return 3;
            fi
        fi
        # 2:本节点没有pool
        log "INFO: this node is not in the pool. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        output_json "${INSPECT_ITEM}-2"
        echo "result=${NORMAL}"
        return 2;
    fi
    return 0;
}

# 801先使用db show db lunmap 来获取poolid，在使用gc show publicinfo 来获取
function show_801_freeratio()
{
new_ebs_cmd=`diagnose_usr <<EOF --set-cli --auto 
attach 273
db show db lunmap
exit
EOF`
    db_info=`echo ${new_ebs_cmd} | sed 's/ /\n/g' | grep -i name`

    # Name:[kvs_db_k15_p14] Name:[kvs_db_k13_p14] Name:[kvs_db_k14_p14] Name:[kvs_db_k12_p14]
    poolid=`echo $db_info | awk -F ' ' '{print$1}'| grep -Po "(?<=_p)\d+" `

    # 找到poolid
new_ebs_cmd=`diagnose_usr <<EOF --set-cli --auto 
attach 273
gc show publicinfo ${poolid}
exit
EOF`
    free_ratio=`echo $new_ebs_cmd | grep -Po "free ratio \d+" | awk -F ' ' '{print$3}'`
    if [ -z "$free_ratio" ] 
    then
        free_ratio=0
        log "pooid=${poolid} not find free ratio, free ratio ${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 1
    fi

    if [ $free_ratio -lt 12500 ]
    then
        log "pooid=${poolid} have too few usable space, free ratio ${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 1
    fi

    log "pooid=${poolid} have enough space. freeRatio=${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
    return 0
}

function check_video_scenes()
{
    if [ -e $VIDEO_SCENES ]
    then
        solution_scenario=`cat $VIDEO_SCENES | grep solution_scenario= | awk -F '=' '{ print $2 }'`
        if [ $solution_scenario = "video_scenes" ]
        then
            return 1
        else
            return 0
        fi
    else
        return 0
    fi
}

# 810和803都使用gc show freeratio 获取
function show_freeratio()
{
new_ebs_cmd=`diagnose_usr <<EOF --set-cli --auto 
attach 273
gc show freeratio
exit
EOF`
    free_ratio=`echo ${new_ebs_cmd} | sed 's/ /\n/g' | grep -Po "(?<=freeratio=)\d+"`
    check_video_scenes
    let ret=$?
    if [[ $ret -eq 0 && $free_ratio -lt 11000 ]]
    then
        log "pooid have too few usable space. freeRatio=${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 1
    elif [[ $ret -eq 1 && $free_ratio -lt 1700 ]]
    then
        log "In video_scenes, pooid have too few usable space. freeRatio=${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 1
    else
        log "pooid have enough space. freeRatio=${free_ratio}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 0
    fi
}

function inspect_info()
{

attach_ebs=`diagnose_usr <<EOF --set-cli --auto 
ls
exit
EOF`
    has_ebs=`echo ${attach_ebs} | grep -o "EBS"`
    if [ -z ${has_ebs} ];then
        log "INFO: this node diagnose doesn't have EBS. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        output_json "${INSPECT_ITEM}-0"
        return 0
    fi

    check_801_version
    if [[ $IS_801_VERSION = "yes" ]];then
        show_801_freeratio
    else
        show_freeratio
    fi

    judge_flag=$?

    if [ ${judge_flag} -eq 1 ];then
        output_json "${INSPECT_ITEM}-1" "\"pool free ratio is low\"" 
        log "check:unok [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 1
    else
        output_json "${INSPECT_ITEM}-0" "\"pool free ratio is ok\"" 
        log "check:ok [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return 0
    fi
}


function check_role()
{
    local_role=$(cat ${NETWORK_CONF}|grep -e ^role= |awk -F = '{print $2}')
    if [[ ${local_role}"X" == "X" ]];then
        if [ -f "${DEPLOY_VERSION}" -o -f "${HA_BIN}" ];then
            ROLE="FSM"
        else
            ROLE="FSA"
        fi
    elif [[ "${local_role}" =~ .*storage.* ]];then
        ROLE="FSA"
    else
        ROLE="FSM_OR_COMPUTE"
    fi
}

function main()
{
    log "INFO: begin to inspect ${INSPECT_ITEM}. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
    check_role
    # FSM节点不检查
    if [[ ${ROLE} == "FSM_OR_COMPUTE" ]];then
        echo "result=${NORMAL}"
        output_json "${INSPECT_ITEM}-0"
        log "INFO: inspect ${INSPECT_ITEM} no need in FSM node. [Line:${LINENO}]" ${FUNCNAME} ${SCRIPT_NAME}
        return ${NORMAL}
    fi

    # 检查diagnose是否可用：1、节点故障；2、节点不在存储池；3、diagnose登录超过4个用户
    check_diagnose_valid
    if [ $? -ne 0 ];then
        exit 0
    fi

    # 检查存储池空闲率
    inspect_info
    
    echo "result=$?"
    exit 0
}

main $@
