# -*- coding: UTF-8 -*-

import utils
import json
from constants import Result
from com.huawei.bundleupgrade.service.connection.fusionstrg import FusionStorageSshPool


def check_storage_software(version):
    """
    :function: 存储软件风险配套检查
    :param version: 当前存储软件版本
    :return: True：检查通过；False:存在风险软件配套关系
    """
    black_product_ver = ("8.1.0", "8.1.0.SPH1", "8.1.0.SPH2", "8.1.0.SPH3", "8.1.0.SPH5", "8.1.0.SPH6",
                         "8.1.1", "8.1.RC2", "8.1.1.SPH1", "8.1.1.SPH2", "8.1.1.HP2", "8.1.1.HP3", "8.1.1.HP5")
    # 服务器BIOS风险版本与存储软件配套检查
    for black_ver in black_product_ver:
        if black_ver == version:
            return False

    return True


def check_bios(target_bios_ver, uid):
    """
    :function: 存储待升级BIOS是否为风险BIOS版本
    :param target_bios_ver: 目标BIOS版本
    :return: True：检查通过；False:是风险版本
    """
    ve_bord_uid = ("0x0207b900", "0x0207b901", "0x0207a100", "0x0207a101")      # 3系列
    vd_bord_uid = ("0x0201b900", "0x0201b901", "0x0201b902", "0x0201b903", "0x0201a900", "0x0201a901", "0x0201a902",
                   "0x0201a903", "0x0201a500", "0x0201a501", "0x0207a900", "0x0207a901", "0x0207a902", "0x0207a903",
                   "0x0207a500", "0x0207a501", "0x0207a502", "0x0207a503")      # 1系列

    # 如果不是泰山V2的机型, 不用管了直接返回, 812之前都不涉及X86服务器
    if uid not in ve_bord_uid and uid not in vd_bord_uid:
        return True

    if uid in ve_bord_uid and "3.37" <= target_bios_ver:
        return False
    elif uid in vd_bord_uid and "1.83" <= target_bios_ver:
        return False

    return True


def check_driver(rdma_ver):
    """
    :function: 当前节点驱动是否为内核随机化条件下风险驱动
    :param rdma_ver: 目标BIOS版本
    :return: True：检查通过；False:是风险版本
    """
    min_rdma_ver_810 = "4.19-26-pacific-5a5-0"
    min_rdma_ver_x86 = "4.19-31-x86-7582-0"
    min_rdma_ver_arm = "4.19-31-arm-7582-0"

    if "pacific" in rdma_ver and rdma_ver < min_rdma_ver_810:
        return False
    elif "arm" in rdma_ver and rdma_ver < min_rdma_ver_arm:
        return False
    elif "x86" in rdma_ver and rdma_ver < min_rdma_ver_x86:
        return False

    return True


def check_result(software_result, driver_result, language):
    """"
    :function: 检查结果计算
    :param software_result: True存储检查通过; False检查不通过
    :param driver_result: True驱动检查通过; False检查不通过
    :param language: 语言类
    :return: True：检查通过；False:检查不通过
    """
    software_err_msg = "pass"
    driver_err_msg = "pass"

    # 如果存储软件和驱动都没有风险则返回PASS
    if software_result and driver_result:
        return Result.PASS, ""

    # 但凡有一个有风险
    if not software_result:
        software_err_msg = "not pass"

    if not driver_result:
        driver_err_msg = "not pass"

    return Result.NOT_PASS, utils.get_msg(language,
                                          "query.node.status.bios_match_risk").format(software_err_msg, driver_err_msg)


def execute(context):
    """
    :function: BIOS升级前检查入口函数
    :param context: 上下文信息
    :return: PASS：检查通过；NOT_PASS:检查不通过, 存在风险软件配套关系
    """
    logger = context.get("logger")
    lang = context.get("lang")
    dev_list = context.get("selectedDevices")
    content = json.loads(context.get("storagePoolContent"))
    storage_pool = content.get("storagePools")

    if not storage_pool:
        return Result.PASS, ""

    # 可以一起升级的集群, 存储软件/UID/DSM/目标BIOS版本都是一致的
    use_dsm = storage_pool[0].get("dsmSwitch")
    target_bios_ver = context.get("biosTargetVersion")
    product_ver = context.get("productVersion").strip().upper()
    dev_uid = dev_list[0].getDeviceEntity().getProductUniqueId().lower()

    logger.info("start bios pre update check.")
    # 若有DSM, 则必有BBU, 只需要看泰山服务器的检查
    if use_dsm != 1:
        return Result.PASS, ""

    # 如果BIOS是风险版本, 开了地址随机化
    if check_bios(target_bios_ver, dev_uid):
        return Result.PASS, ""

    software_result = check_storage_software(product_ver)
    driver_result = False
    for dev in dev_list:
        # 服务器BIOS风险版本与多节点的网卡驱动配套检查
        rdma_ver = "unknown"
        con = FusionStorageSshPool.getFsSshConnectionByBmcIp(dev.getDeviceEntity().getIp())
        lines = con.executeCmd("rdma_ver").splitlines()
        for line in lines:
            if "4.19-31" in line or "4.19-26" in line:
                rdma_ver = line.strip()
                break

        driver_result = check_driver(rdma_ver)
        if not driver_result:
            break

    return check_result(software_result, driver_result, lang)
