# -*- coding: UTF-8 -*-

import time
import utils
import rest_util
from constants import Result, Timeout, RestType, NodeStatus
from com.huawei.uMate.common import UMateException


def execute(context):
    """
    上线节点的入口函数
    :param context: 上下文信息
    :return: PASS：上线成功；NOT_PASS:上线失败
    """
    logger = context.get("logger")
    lang = context.get("lang")
    logger.info("start post check.")
    try:
        flag, err_msg = check_node_status(context)
        if not flag:
            logger.error("exec check node status failed.")
            return Result.NOT_PASS, err_msg
        waiting_node_management_process_startup(context)
        return check_task_result(context)
    except Exception as e:
        logger.error("post_check exception:%s" % e)
        return Result.NOT_PASS, utils.get_msg(lang, "start.service.failed")


def check_task_result(context):
    logger = context.get("logger")
    lang = context.get("lang")
    r_lock = context.get("rLock")

    try:
        r_lock.lock()
        logger.info("start exec start service:" + context.get("managementIP"))
        flag, task_id, err_msg = start_service(context)
        if not flag:
            logger.error("exec start service func failed.")
            return Result.NOT_PASS, utils.get_msg(lang, "start.service.failed")
        flag, err_msg = rest_util.query_task_status(context, task_id, "start")
        if not flag:
            logger.error("query start service task info failed.")
            return Result.NOT_PASS, err_msg
        return Result.PASS, err_msg
    finally:
        logger.info("end exec start service:" + context.get("managementIP"))
        r_lock.unlock()


def waiting_node_management_process_startup(context):
    logger = context.get("logger")
    start_time = time.time()
    time_out = context.get("nmaProcessRebootTimeOut")
    os_auth = context.get("osAuth")
    if not os_auth:
        return
    cmd = "ps -ef|grep /opt/dfv/oam/oam-u/nma/bin/nmagent.bin|grep -v grep |wc -l"
    node_ssh_pool = context.get("nodeSSHPool")
    bmc_ip = context.get("deviceEntity").getDeviceEntity().getIp()
    while time.time() - start_time < time_out:
        if node_management_process_is_running(node_ssh_pool, bmc_ip, cmd,
                                              logger):
            return
        time.sleep(Timeout.DEFAULT_INTERVAL)


def node_management_process_is_running(node_ssh_pool, bmc_ip, cmd, logger):
    try:
        cli_ret = node_ssh_pool.getFsSshConnectionByBmcIp(
            bmc_ip).executeCmd(cmd)
        for line in cli_ret.splitlines():
            if line.strip() == "1":
                return True
    except UMateException as e:
        logger.error("SSH connection failed:{}".format(e))
    except Exception as e:
        logger.error("check node nma status exception:{}".format(e))
    return False


def check_node_status(context):
    """
    查询节点状态是否正常
    :return: True：状态正常；False：状态异常
    """
    logger = context.get("logger")
    lang = context.get("lang")
    is_check_node_status = context.get("isCheckNodeStatus")

    url = "/api/v2/cluster/servers"

    node_ssh_pool = context.get("nodeSSHPool")
    bmc_ip = context.get("deviceEntity").getDeviceEntity().getIp()

    cmd_dict = dict(url=url, rest_type=RestType.GET,
                    retry=Timeout.DEFAULT_RETRY,
                    interval=Timeout.DEFAULT_INTERVAL)

    start_time = time.time()
    data = ""
    while True:
        if time.time() - start_time > Timeout.QUERY_NODE_STATUS_TIMEOUT:
            logger.info("query node status timeout")
            return _reconnect(cmd_dict, context, data, lang)

        try:
            # 单管理节点的场景，重启完成先登录
            if not _is_login_success(context):
                time.sleep(Timeout.DEFAULT_INTERVAL)
                continue
            flag, data, err_msg = rest_util.exe_rest(context, cmd_dict)
            if not flag:
                time.sleep(Timeout.DEFAULT_INTERVAL)
                continue
            if is_check_node_status:
                flag, err_msg = check_node_status_normal(context, data)
            else:
                flag = check_node_fsa_status_normal(node_ssh_pool, bmc_ip, logger)
            if flag:
                return True, ""
            time.sleep(Timeout.DEFAULT_INTERVAL)
        except Exception as e:
            logger.error("check node status exception:%s" % e)
            time.sleep(Timeout.DEFAULT_INTERVAL)

    err_msg = utils.get_msg(lang, "check.node.status.abnormal")
    return False, err_msg


def _is_login_success(context):
    """
    双管理节点连接正常，默认登陆成功；
    单管理节点环境，若管理节点重启，连接会中断一段时间，需重连
    :param context:上下文
    :return:是否登录成功
    """
    logger = context.get("logger")
    if context.get("isSingleManagementRestart"):
        logger.info("single management node restart.")
        connector = context.get("connector")
        logger.info("Reset connection")
        connector.releaseConn()
        connector.connect()
        return connector.isLoginSuccess()
    else:
        logger.info("is not single management node restart.")
        return True


def _reconnect(cmd_dict, context, data, lang):
    if not data:
        return False, utils.get_msg(lang, "query.node.status.timeout")
    context.get("connector").reLogin()
    flag, data, err_msg = rest_util.exe_rest(context, cmd_dict)
    if not flag:
        return False, utils.get_msg(lang, "query.node.status.timeout")
    return check_node_status_normal(context, data)


def check_node_status_normal(context, data):
    """
    检查当前节点状态是否正常
    :param context: 上下文
    :param data: 查询的节点信息数据
    :return:  True:节点正常；False：节点不正常
    """
    logger = context.get("logger")
    lang = context.get("lang")

    select_node_manage_ip = context.get("managementIP")
    for node_data in data:
        management_ip = node_data.get("management_ip", "")
        if management_ip != select_node_manage_ip:
            continue

        node_status = node_data.get("status", -1)
        if node_status == NodeStatus.NORMAL:
            return True, ""
        logger.error("current node(%s) status is:%s"
                     % (select_node_manage_ip, str(node_status)))
        err_msg = utils.get_msg(lang, "check.node.status.abnormal")
        return False, err_msg
    err_msg = utils.get_msg(lang, "check.node.status.abnormal")
    return False, err_msg


def check_node_fsa_status_normal(node_ssh_pool, bmc_ip, logger):
    cmd = "sh /opt/fusionstorage/agent/script/dsware_agent_tool --op status >/dev/null 2>&1; echo $?"
    try:
        cli_ret = node_ssh_pool.getFsSshConnectionByBmcIp(bmc_ip).executeCmd(cmd)
        if "0" in [line.strip() for line in cli_ret.splitlines()]:
            return True
    except UMateException as e:
        logger.error("SSH connection failed:{}".format(e))
    except Exception as e:
        logger.error("check node fsa status exception:{}".format(e))
    return False


def start_service(context):
    """
    上线节点
    :param context: 上下文信息
    :return: True：执行命令成功；False：执行命令失败
    """
    lang = context.get("lang")
    manage_ip = context.get("managementIP")

    url = "/api/v2/cluster/cluster_node"
    params = dict(start_service=manage_ip)

    cmd_dict = dict(url=url, params=params, rest_type=RestType.POST,
                    retry=Timeout.DEFAULT_RETRY,
                    interval=Timeout.DEFAULT_INTERVAL)

    flag, data, err_msg = rest_util.exe_rest(context, cmd_dict)
    if not flag:
        err_msg = utils.get_msg(lang, "start.service.failed")
        return False, "", err_msg
    task_id = data.get("task_id")
    return True, task_id, ""
