# -*- coding: UTF-8 -*-

import time
import json
import traceback

import utils
from constants import Timeout, RestType, ErorCode

from java.lang import Exception as JException


def exe_rest(context, cmd_dict):
    """
    执行rest命令公共接口
    :param context: 上下文信息
    :param cmd_dict: 执行命令的参数
    :return: 执行命令的结果
    """
    logger = context.get("logger")
    connector = context.get("connector")
    lang = context.get("lang")

    url = cmd_dict.get("url", "")
    params = cmd_dict.get("params", {})
    rest_type = cmd_dict.get("rest_type", RestType.GET)
    retry = cmd_dict.get("retry", Timeout.DEFAULT_RETRY)
    interval = cmd_dict.get("interval", Timeout.DEFAULT_INTERVAL)

    for index in range(retry):
        try:
            if rest_type == RestType.GET:
                content = connector.getRestResponse(url, params)
            elif rest_type == RestType.POST:
                content = connector.postRestResponse(url, params)
            else:
                err_msg = utils.get_msg(lang, "not.support.cmd.type")
                return False, "", err_msg
        except (JException, Exception):
            logger.error("exe cmd(%s) occured exception:%s" %
                         (url, traceback.format_exc()))
            time.sleep(interval)
            connector.releaseConn()
            connector.connect()
            continue

        logger.info("exe cmd(%s) params(%s) retry (%s) time result:%s" %
                    (url, params, index, content))
        flag, data, err_info = parse_rest_res(content)
        if flag:
            return True, data, ""

        if err_info.get("code") in [ErorCode.INSURFFICIENT_USER_RIGHTS,
                                    ErorCode.CONNECT_FAILED]:
            time.sleep(interval)
            connector.releaseConn()
            connector.connect()
            continue
        if err_info.get("code", "-1") != '0':
            time.sleep(interval)
            continue

    err_msg = utils.get_msg(lang, "exe.cmd.failed")
    return False, "", err_msg


def parse_rest_res(rest_content):
    """
    解析rest响应
    :param rest_content: rest 响应
    :return: 解析后的rest响应
    """
    try:
        res_json = json.loads(rest_content)
    except Exception:
        return False, "", {}
    result = res_json.get("result")
    if not isinstance(result, dict):
        code = str(result)
        if code != "0":
            error_code = res_json.get("errorCode", code)
            desc = res_json.get("description", "")
            sugg = res_json.get("suggestion", "")
            err_info = dict(code=error_code, desc=desc, sugg=sugg)
            return False, "",  err_info

        return True, res_json, ""

    code = str(result.get("code"))
    if code != "0":
        desc = result.get("description")
        sugg = result.get("suggestion")
        err_info = dict(code=code, desc=desc, sugg=sugg)
        return False, "", err_info
    data = res_json.get("data")
    return True, data, ""


def query_task_status(context, task_id, task_type):
    """
    查询任务进度和状态
    :param context: 上下文信息
    :param task_id: 任务ID
    :param task_type: 任务类型（离线/上线）
    :return: True：执行成功， False:执行失败
    """
    lang = context.get("lang")

    query_uri = "/api/v2/cluster/sub_tasks?sub_task_id=1&task_id=%s" % task_id
    cmd_dict = dict(url=query_uri)

    start_time = time.time()
    while time.time()-start_time < Timeout.QUERY_TASK_TIMEOUT:
        flag, data, err_msg = exe_rest(context, cmd_dict)
        if not flag:
            return False, err_msg
        server_detail = data.get("server_detail")
        server_task_status = server_detail[0].get("server_task_status")
        if server_task_status == "success":
            return True, ""
        if server_task_status == "failure":
            return False, utils.get_msg(lang, "%s.service.failed" % task_type)
        time.sleep(Timeout.DEFAULT_INTERVAL)
    return False, utils.get_msg(lang, "%s.service.timeout" % task_type)
