# -*- coding: UTF-8 -*-

import os
import traceback

from profile_xml_parser import ProfileParser
from resource_bundle import ResourceBundle
from upgrade_using_redfish import RedFishUpgrader
from user_sensible_exceptions import UserException

from constants import upg_constants
from file_uploader import FileUploader

SNMP_CMD_NOT_START_OR_DONE = "-1"

SNMP_STATE_FINISH = "100"


def execute(context):
    """
    the entrance of out-band module upgrade.
    :param context:  the data dict from Jython imported params ---refer to.
            OutBandUpgradeServiceServiceImpl buildScriptEnvs
    :return (result status, error message)
    """
    logger = context.get("logger")

    logger.info("now start to execute out-band"
                " module upgrade commands....")
    upgrade_executor = OutBandModuleUpgradeExecutor(context)
    return upgrade_executor.do_upgrade()


class OutBandModuleUpgradeExecutor(object):
    """
    the OutBandModuleUpgradeExecutor: init context and execute module upgrade.
    """

    def __init__(self, context):
        """
        data initiation.
        :param context: context value from Jython imported params
        """
        self.context = context
        self.logger = context.get("logger")
        self.device = context.get("device")
        self.ssh_sftp = context.get("ssh_and_sftp")
        self.res = ResourceBundle(context)

    def do_upgrade(self):
        """
        proceed the upgrade .
        :return: execution result, error_message
        """
        profile_parser = ProfileParser(self.context)
        upgrade_profile = profile_parser.get_profile_details()
        file_uploader = FileUploader(self.context)
        try:
            local_file_path = self._get_local_file_location(upgrade_profile)
            remote_path = upg_constants \
                .ServerConstants.REMOTE_TEMP_DIR
            file_uploader.upload_using_sftp(local_file_path, remote_path)
            self.logger.info("now try to upgrade using RedFish.")
            try:
                rest_conn = RedFishUpgrader(self.context)
                if rest_conn.do_upgrade(upg_constants.ServerConstants.REMOTE_HPM_FILE_PATH):
                    return upg_constants.ExecResult.PASS, ""
                return upg_constants.ExecResult.FAILED, self.res.get_resource("upgrade.failed")
            except BaseException as exp:
                self.logger.error("redfish upgrade exception: {} {}", str(exp), traceback.format_exc())
                return upg_constants.ExecResult.FAILED, exp.message
        except UserException as exception:
            return upg_constants.ExecResult.FAILED, exception.message

    def _get_local_file_location(self, upgrade_profile):
        """
        get the local hpm file location
        :param upgrade_profile:
        :raises UserException
        :return: local_file_path
        """
        local_file_path = upgrade_profile.get_firmware_path()
        converted_path = os.path.dirname(local_file_path) + os.sep + upg_constants.ServerConstants.HPM_FILE_NAME
        if not os.path.exists(local_file_path) and not os.path.exists(converted_path):
            self.logger.error("target hmp file locally does not exist!")
            raise UserException(self.res.get_resource("target.hpm.file.not.exist"))
        if not os.path.exists(converted_path) and os.path.exists(local_file_path):
            self.logger.info("rename file to %s..." % converted_path)
            os.rename(local_file_path, converted_path)
            self.logger.info("converted target file to a "
                             "recognized file name: %s." % local_file_path)
        local_file_path = converted_path
        return local_file_path
