# -*- coding: UTF-8 -*-
import json
import time
from datetime import datetime
from constants import upg_constants

import com.huawei.uMate.common.rest.RedfishResult as RedfishResult

from upg_constants import HttpMethods, RedFishResources, \
    RedFishTaskState, RedFishTaskStateType
from bundle_upgrade_util import get_upgrade_time_out_interval
from resource_bundle import ResourceBundle
from user_sensible_exceptions import UserException


class RedFishUpgrader:
    def __init__(self, context):
        """
        the restful connector.
        :param context: the context map passed to python from java.
        """
        self.context = context
        self.rest_con = self.context.get("rest_conn")
        if self.rest_con is None:
            raise UserException("connection does not exists!")
        self.logger = context.get("logger")
        self.res = ResourceBundle(context)
        self.device = context.get("device")

    def do_upgrade(self, remote_dir):
        """
        execute entrance
        :param remote_dir remote hpm file location.
        :return:
        """
        task_oid = self.inform_device_2_upgrade(remote_dir)
        if not task_oid:
            self.logger.error("[%s]task oid is invalid." %
                              self.device.getDeviceEntity().getIp())
            return False
        return self.looping_4_upgrade_status(task_oid)

    def looping_4_upgrade_status(self, task_oid,
                                 allow_continuous_err=10):
        """
        looping for upgrade status till upgrade finished.
        :param allow_continuous_err:
        :param task_oid: RedFish resource URI
        :return: is finished successfully.
        """
        start_date = datetime.now()
        current_module = self.context.get("module")
        time_out_interval = get_upgrade_time_out_interval(current_module)
        continuous_err = 0
        while (datetime.now() - start_date).seconds < time_out_interval:
            time.sleep(2)
            result = self.rest_con.request(HttpMethods.GET, task_oid, None,
                                           None, None)
            try:
                RedFishUpgrader \
                    .check_is_redfish_response_ok(result, self.logger)
                response_dict = json.loads(result.getResult())
                # 如果当前升级组件是bmc且是专有硬件则判断升级进度为100%则为升级完成
                task_state = response_dict.get("TaskState")
                if current_module == upg_constants.ModuleConstants.MODULE_BMC and \
                        self.device.getDeviceEntity().isSpecificDevice():
                    if RedFishTaskState.get(task_state, "") == RedFishTaskStateType.DONE:
                        return task_state == "Completed"
                    if task_state != "Running":
                        continue
                    task_key = response_dict.get("Oem", {}).get("Huawei", {})
                    if not task_key:
                        task_key = response_dict.get("Oem", {}).get("Public", {})
                    task_percentage = task_key.get("TaskPercentage", {})
                    if task_percentage == "100%":
                        self.logger.info("Exit Query, current TaskPercentage: %s" % task_percentage)
                        return True
                    else:
                        self.logger.info("Continue Query, current TaskPercentage: %s" % task_percentage)
                else:
                    if RedFishTaskState.get(task_state, "") == RedFishTaskStateType.DONE:
                        return task_state == "Completed"
            except BaseException as exception:
                self.logger.error("[%s]Get upgrade "
                                  "state failed: %s" % (
                                      self.device.getDeviceEntity().getIp(),
                                      exception))
                continuous_err += 1
                if continuous_err > allow_continuous_err:
                    self.logger.error("Too much continuous error!")
                    break
        self.logger.error("Error! Time out or too much continuous exception.")
        return False

    def inform_device_2_upgrade(self, remote_dir):
        """
        send RedFish resource to inform current device execute upgrade.
        :param remote_dir: the remote hpm file's location.
        :return: task oid
        """
        header = "{\"Content-Type\": \"application/json\"}"
        data_json = "{\"ImageURI\": \"%s\"}"
        data_json = data_json % remote_dir

        for _ in range(3):
            try:
                result = self.rest_con.request(HttpMethods.POST,
                                               RedFishResources.DO_UPGRADE,
                                               header, data_json, None)
                RedFishUpgrader \
                    .check_is_redfish_response_ok(result, self.logger)
                response_dict = json.loads(result.getResult())
                task_oid = response_dict.get("@odata.id")
                self.logger.info("task oid :" + task_oid)
                return task_oid
            except UserException as exception:
                self.logger.error("failed to inform upgrade on %d time" % _)
                self.logger.error("exception: %s" % exception)
                pass
        raise UserException(self.res.get_resource("rest.inform."
                                                  "response.invalid"))

    @staticmethod
    def check_is_redfish_response_ok(response_obj, logger):
        """
        check whether current response object is normal with no errors
        :param logger: logger instance
        :param response_obj: instance of RedfishResult
               ref. java.com.huawei.uMate.common.rest.RedfishResult
        :return: void
        :raises UserSensibleException
        """
        if not isinstance(response_obj, RedfishResult):
            logger.error("current RedFish response is not valid.")
            raise UserException("invalid response")
        logger.info("result:%s;\n"
                    "status:%s." % (response_obj.getResult(),
                                    response_obj.getStatusCode()))
        if not response_obj.getStatusCode().startswith("2"):
            logger.info("error Code:" + response_obj.getErrorCode())
            logger.info("errorMsg:" + response_obj.getErrorMsg())
            logger.error("HTTP response status is not ok.")
            raise UserException("http response invalid.")
