﻿# -*- coding: UTF-8 -*-
import time

from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import FuncFactory
from cbb.business.operate.fru.common import getCurrentFruRunningInfo
from cbb.frame.base import baseUtil
from cbb.frame.base import funcUtils
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from cbb.frame.util import ability_map_util

CHECK_TIME = 10 * 60


@funcUtils.fakeProgress(totalSecs=CHECK_TIME, intervalSec=5)
def execute(context, preCheck=True, **kwargs):
    """获取所有启动器信息（ETH和FC）

    :param context:
    :param preCheck:
    :return:
    """
    if preCheck:
        # 获取所有linkUp的端口信息
        linkedPortInfo = getLinkedPortsInfo(context, **kwargs)
        # 获取所有linkUp端口中Online的启动器wwn列表
        onlineHostInitiator = getPortInitiatorInfo(context, linkedPortInfo)

        if onlineHostInitiator is None:
            return

        # 保存信息
        BaseFactory.persist.setModule(context, "linkUpPortInfo", linkedPortInfo)
        BaseFactory.persist.setModule(context, "linkUpPortOnlineWwnInfo", onlineHostInitiator)

        BaseFactory.log.info(context, "linkedPortInfo :%s" % str(linkedPortInfo))
        BaseFactory.log.info(context, "onlineHostInitiator :%s" % str(onlineHostInitiator))
        BaseFactory.result.setResultPass(context)
        return
    # 更换后比对信息
    else:
        BaseFactory.log.info(context, "Check Host Initiators begin")
        # 获取已保存信息
        online_host_initiator_before = BaseFactory.persist.getModule(
            context,
            "linkUpPortOnlineWwnInfo")
        online_host_initiator_before = online_host_initiator_before if \
            online_host_initiator_before else {}
        BaseFactory.log.info(context, "HostInitiatorInfo before is :%s" %
                             str(online_host_initiator_before))

        if online_host_initiator_before is None:
            BaseFactory.result.setResultPass(context)
            return

        start_time = time.time()
        abnormal_port_info = {}
        while True:
            if time.time() - start_time > CHECK_TIME:
                err_msg_list = []
                for port in abnormal_port_info:
                    wwn_str = ", ".join(abnormal_port_info[port])
                    err_msg_list.append("{0}:{1}".format(port, wwn_str))

                BaseFactory.result.setResultFailByKey(
                    context,
                    FuncFactory.LangKey.HOST_INITIATOR_CHECK_CONSISTENT,
                    "<br>".join(err_msg_list),
                    ", ".join(abnormal_port_info.keys()))
                BaseFactory.log.info(context,
                                     "Check Host Initiators fail end")
                return
            time.sleep(30)
            abnormal_port_info = {}
            # 获取所有linkUp的端口信息
            linked_port_info_after = getLinkedPortsInfo(context, **kwargs)
            # 获取所有linkUp端口中Online的启动器wwn列表
            online_host_initiator_after = getPortInitiatorInfo(
                context,
                linked_port_info_after)

            BaseFactory.log.info(context, "HostInitiatorInfo after is :%s" %
                                 str(online_host_initiator_after))
            if online_host_initiator_after is None:
                return
            for port_id, before_wwns in online_host_initiator_before.items():
                if port_id not in online_host_initiator_after:
                    if port_id not in abnormal_port_info:
                        abnormal_port_info[port_id] = []

                after_wwns = online_host_initiator_after.get(port_id, [])
                # 更换前启动器，如果在更换后状态异常
                for wwn in before_wwns:
                    if wwn not in after_wwns:
                        if port_id not in abnormal_port_info:
                            abnormal_port_info[port_id] = []
                        abnormal_port_info[port_id].append(wwn)

            if not abnormal_port_info:
                BaseFactory.result.setResultPass(context)
                BaseFactory.log.info(context,
                                     "Check Host Initiators pass end")
                return


@ability_map_util.ability_mapping
def getLinkedPortsInfo(context, **kwargs):

    """获取LinkUp状态的端口信息

    :param context:
    :return:
    """
    linkedPortInfo = {}
    # 若接口卡包含ETH端口
    condition = restUtil.Tlv2Rest.getCondition(
        restData.Hardware.EthPort.LOGIC_TYPE,
        restData.Enum.ConditionTypeEnum.EQ,
        restData.Enum.EthPortLogicTypeEnum.HOST
    )
    condition_list = restUtil.Tlv2Rest.getConditionList(condition)

    linkedEthPorts = getCurrentFruRunningInfo.execute(
        context, restData.Enum.ObjEnum.ETH_PORT,
        restData.Enum.RunningStatusEnum.LINK_UP,
        condition_list=condition_list, **kwargs)
    linkedPortInfo["ETH"] = linkedEthPorts
    # 若接口卡包含FC端口
    linkedFCPorts = getCurrentFruRunningInfo.execute(context, restData.Enum.ObjEnum.FC_PORT,
                                                     restData.Enum.RunningStatusEnum.LINK_UP,
                                                     **kwargs)
    # 保存ETH和FC端口信息
    linkedPortInfo["FC"] = linkedFCPorts

    return linkedPortInfo


def get_linked_ports_info_for_micro_dev(context, **kwargs):
    """
    获取LinkUp状态的端口信息(微存储不支持ETH端口)
    :param context: 上下文
    :return:
    """
    linked_port_info = {}
    linked_fc_ports = getCurrentFruRunningInfo.execute(context, restData.Enum.ObjEnum.FC_PORT,
                                                       restData.Enum.RunningStatusEnum.LINK_UP,
                                                       **kwargs)
    # 保存ETH和FC端口信息
    linked_port_info["FC"] = linked_fc_ports

    return linked_port_info


def get_linked_ports_info_for_compute(context, **kwargs):
    """
    获取LinkUp状态的端口信息（计算型存储不支持FC端口）
    :param context: 上下文
    :return: 端口信息
    """
    linked_port_info = {}
    # 若接口卡包含ETH端口
    condition = restUtil.Tlv2Rest.getCondition(
        restData.Hardware.EthPort.LOGIC_TYPE,
        restData.Enum.ConditionTypeEnum.EQ,
        restData.Enum.EthPortLogicTypeEnum.HOST
    )
    condition_list = restUtil.Tlv2Rest.getConditionList(condition)

    linked_eth_ports = getCurrentFruRunningInfo.execute(
        context, restData.Enum.ObjEnum.ETH_PORT,
        restData.Enum.RunningStatusEnum.LINK_UP,
        condition_list=condition_list, **kwargs)
    linked_port_info["ETH"] = linked_eth_ports

    return linked_port_info


def getPortInitiatorInfo(context, linkedPortInfo):
    """获取LinkUp状态的端口的所有启动器信息

    :param context:
    :param linkedPortInfo:
    :return:
    """
    currentHostInitiator = {}
    for portType, portObjs in linkedPortInfo.items():
        portIds = portObjs.keys()
        for portId in portIds:
            onlineInitiatorWwns = queryOnePortInitiators(context, portType, portId)
            if onlineInitiatorWwns is None:
                BaseFactory.result.setResultFailByKey(context, FuncFactory.LangKey.EXEC_CLI_CMD_FAIL)
                return None
            # 记录端口的在线的启动器wwn信息（此处不需要区分ETH、FC端口类型）
            currentHostInitiator[portId] = onlineInitiatorWwns

    return currentHostInitiator


@ability_map_util.ability_mapping
def queryOnePortInitiators(context, portType, portId):
    """获取单个端口的所有启动器信息

    :param context:
    :param portType:
    :param portId:
    :return:
    """
    onlineInitiatorWwns = []
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)

    cmd = "show port initiator port_type={} port_id={}".format(portType,
                                                               portId)
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        BaseFactory.log.error(context, "Command execute failed!")
        return None
    for initiatorInfo in cliUtil.getHorizontalCliRet(cliRet):
        if initiatorInfo.get("Running Status") == "Online":
            onlineInitiatorWwns.append(initiatorInfo.get("WWN"))

    # Dorado 6.1新增支持NVMe接口卡，ETH端口普通命令查询为空时使用单独命令查询NVMe启动器
    if portType == "ETH" and not onlineInitiatorWwns \
            and baseUtil.is_support_NVMe_intf(context):
        cmd_nvme = "show port nvme_over_roce_initiator port_id={}".format(
            portId)
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd_nvme, True,
                                                          lang)
        if flag is not True:
            BaseFactory.log.error(context, "Command execute failed!")
            return None
        for initiatorInfo in cliUtil.getVerticalCliRet(cliRet):
            if initiatorInfo.get("Running Status") == "Online":
                onlineInitiatorWwns.append(initiatorInfo.get("NQN"))
    return onlineInitiatorWwns


def query_one_port_initiators_for_compute(context, port_type, port_id):
    """获取单个端口的所有启动器信息

    :param context:
    :param port_type:
    :param port_id:
    :return:
    """
    online_initiator_wwns = list()
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)

    cmd = "show port initiator port_type={} port_id={}".format(port_type,
                                                               port_id)
    flag, cli_ret, err_msg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        BaseFactory.log.error(context, "Command execute failed!")
        return None
    for initiator_info in cliUtil.getHorizontalCliRet(cli_ret):
        if initiator_info.get("Running Status") == "Online":
            online_initiator_wwns.append(initiator_info.get("iSCSI IQN"))

    return online_initiator_wwns
