# coding:utf-8
from cbb.frame.base import config
from cbb.business.product.product_adapter_base import ProductBaseAdapter
from cbb.business.product.expansion_doc_link import ExpansionDocLink
from cbb.business.product import interface_check
from cbb.common.conf.switch_config import CE8851_DORADO_HIGH
from cbb.common.conf.switch_config import CE8851_DORADO_MID
from cbb.common.conf.switch_config import CE8850_DORADO
from cbb.common.conf.new_switch_config import CE8855_DORADO_HIGH
from cbb.common.conf.new_switch_config import CE8855_DORADO_MID
from cbb.frame.dsl import adapter
from utils import Products

# 32Dorado 5600/6000支持32节点
V7_MAX_CTRL_NUM = 32

max_ctrl = 16
SLOT_KEY = "slot"
INTERF_MODEL_KEY = "interfModel"

# 不支持扩控
NOT_SUPPORT_EXPANSION_SPEC = (
    "OceanProtect X6000", "OceanProtect X3000", "OceanStor 2910", "OceanStor 2200", "OceanStor 2220"
)

expansion_spec_list = {
    # X6000不支持扩控
    NOT_SUPPORT_EXPANSION_SPEC: {},
    # E8000支持交换机组网扩控
    tuple(set(config.DORADO_DEVS_V6_LOW + config.OCEAN_PROTECT_MID + config.OCEAN_STOR_MICRO_DEVS).difference(
        set(NOT_SUPPORT_EXPANSION_SPEC + tuple(config.OCEAN_PROTECT_SUPPORT_SWITCH)))): {
        "2C": {
            "direct": {
                "supports_expansion_ctrls": [4],
                "scale_out_card_num": 1
            },
        },
    },
    tuple(config.OCEAN_PROTECT_HIGH): {
        "2C": {
            "none": {
                "supports_expansion_ctrls": [4],
                "scale_out_card_num": 0
            },
            "direct": {
                "supports_expansion_ctrls": [8],
                "scale_out_card_num": 1
            },
        },
        "4C": {
            "direct": {
                "supports_expansion_ctrls": [8],
                "scale_out_card_num": 1
            },
        },
    }
}

support_interface = {
    tuple(config.DORADO_DEVS_V6_LOW): {
        SLOT_KEY: ["IOM1"],
        INTERF_MODEL_KEY:
            {
                # 直连组网：4端口 SO 25Gb RDMA I/O模块
                "direct": [2309, 2338, 2361, 2399],
                # 交换机组网：4端口 SO 25Gb RDMA I/O模块
                "switch": [2338, 2361, 2405, 2457]
            }
    },
    # 微存储DISK中端设备，扩控卡槽位为IOM4
    tuple(config.OCEAN_DISK_DEVS_MID): {
        SLOT_KEY: ["IOM4"],
        INTERF_MODEL_KEY:
            {
                # 直连组网：4端口 SO 25Gb RDMA I/O模块
                "direct": [2309, 2338, 2361, 2399],
                # 交换机组网：4端口 SO 25Gb RDMA I/O模块
                "switch": [2338, 2361, 2351, 2362, 2405, 2457]
            }
    },
    tuple(set(config.DORADO_DEVS_V6_MID).difference(set(config.OCEAN_DISK_DEVS_MID))): {
        SLOT_KEY: ["IOM2"],
        INTERF_MODEL_KEY:
            {
                # 直连组网：4端口 SO 25Gb RDMA I/O模块
                "direct": [2309, 2338, 2361, 2399],
                # 交换机组网：4端口 SO 25Gb RDMA I/O模块
                "switch": [2338, 2361, 2351, 2362, 2405, 2457]
            }
    },
    tuple(config.DORADO_DEVS_V6_HIGH): {
        SLOT_KEY: ["H3", "L3", "H10", "L10"],
        INTERF_MODEL_KEY:
            {
                # 高端直连和交换组网都使用2端口 SO 100Gb RDMA I/O模块
                "direct": [2322, 2357, 2363, 2385],
                "switch": [2357, 2363, 2385]
            }
    },
}

support_switch = {
    tuple(config.DORADO_DEVS_V6_HIGH + config.DORADO_DEVS_V6_MID):
        {
            "CE8850": ("V200R005C10SPC800", "V200R005SPH017"),
            "CE8851": ("V300R020C10SPC200", ""),
            "CE8850-SAN": ("V300R020C10SPC200", ""),
            "CE8850E": ("V200R022C00", "")
        },
}

switch_config = {
    "CE8850": {
        tuple(config.DORADO_DEVS_V6): CE8850_DORADO
    },
    "CE8851": {
        tuple(config.DORADO_DEVS_V6_HIGH): CE8851_DORADO_HIGH,
        tuple(config.DORADO_DEVS_V6_MID): CE8851_DORADO_MID
    },
    "CE8850-SAN": {
        tuple(config.DORADO_DEVS_V6_HIGH): CE8851_DORADO_HIGH,
        tuple(config.DORADO_DEVS_V6_MID): CE8851_DORADO_MID
    },
    "CE8855": {
        tuple(config.DORADO_DEVS_V6_HIGH): CE8855_DORADO_HIGH,
        tuple(config.DORADO_DEVS_V6_MID): CE8855_DORADO_MID
    },
    # CE8850E和CE8850用同一个配置
    "CE8850E": {
        tuple(config.DORADO_DEVS_V6_HIGH): CE8850_DORADO,
        tuple(config.DORADO_DEVS_V6_MID): CE8850_DORADO
    }
}

# 组网图或者帮助文档配置
support_doc_link = {
    "manuallyConfigSwitch": {
        "doc_tag": {
            tuple(config.DORADO_DEVS_V6_HIGH): "manuallyConfiguringSwitches_CE8851_High",
            tuple(config.DORADO_DEVS_V6_MID): "manuallyConfiguringSwitches_CE8851_Middle"
        },
        "method": "get_help_doc"
    },
    "switchInit": {
        "doc_tag": {
            tuple(config.DORADO_DEVS_V6): "switchesInitConfig_CE8851_V6"
        },
        "method": "get_help_doc"
    },
    "highEnd": {
        "doc_tag": {
            tuple(config.DORADO_DEVS_V6): "Dorado612"
        },
        "method": "get_network_image"
    },
    "midAndLowEnd": {
        "doc_tag": {
            tuple(config.OCEAN_PROTECT_SUPPORT_SWITCH): "OceanProtect",
            tuple(set(config.DORADO_DEVS_V6).difference(set(config.OCEAN_PROTECT_SUPPORT_SWITCH))): "Dorado612"
        },
        "method": "get_network_image"
    }
}

interface_config = {
    "interfaceCompatibility": "is_compatibility"
}


class ProductAdapter(ProductBaseAdapter):
    def __init__(self, context):
        super(ProductAdapter, self).__init__(context)
        self.product_mode = adapter.get_dev_type(context)
        self.product_version = adapter.get_dev_version(context)

    def is_support(self):
        from cbb.frame.adapter.replace_adapter import compare_version
        from cbb.frame.adapter.replace_adapter import is_digital_ver
        if not is_digital_ver(self.product_version):
            return Products.compareVersion(self.product_version, "V700R001C00") >= 0
        if "dorado" in self.product_mode.lower():
            return compare_version(self.product_version, "6.1.2RC1") >= 0
        return True

    def get_support_switch(self):
        for key, item in support_switch.items():
            if self.product_mode in key:
                return item

    def get_switch_config(self, switch_ver):
        switch_dict = switch_config.get(switch_ver)
        if switch_dict:
            for key, item in switch_dict.items():
                if self.product_mode in key:
                    return item

    def get_support_interface(self):
        for key, item in support_interface.items():
            if self.product_mode in key:
                return item

    def get_expansion_spec(self, ctrl_num):
        from cbb.business.product import expansion_spec
        for key, item in expansion_spec_list.items():
            if self.product_mode in key:
                return item.get("{}C".format((str(ctrl_num))), {})
        # 5600/6000 V6 版本为V7，最大为32控。
        if self.product_mode in config.DORADO_32_CTRL_DEVS \
                and Products.compareVersion(self.product_version, "V700R001C00") >= 0:
            return expansion_spec.calculate_expansion_dict_2_ctrl(ctrl_num, V7_MAX_CTRL_NUM)

        if self.product_mode in config.DORADO_DEVS_V6_HIGH:
            return expansion_spec.calculate_expansion_dict_4_ctrl(ctrl_num, max_ctrl)
        else:
            return expansion_spec.calculate_expansion_dict_2_ctrl(ctrl_num, max_ctrl)

    def get_doc_link(self, doc_key):
        self.get_logger().info("doc key={}".format(doc_key))
        doc_tag, doc_method = self.parse_doc_key(str(doc_key))
        self.get_logger().info("doc_type={}, doc_tag={}".format(
            doc_key, doc_tag))
        if not doc_tag:
            self.get_logger().info("doc key is invalid.")
            return ""
        expansion_res = ExpansionDocLink(self.context)
        return expansion_res.get_doc_link(doc_tag, doc_method)

    def parse_doc_key(self, doc_key):
        doc_link_dict = support_doc_link.get(doc_key)
        if doc_link_dict:
            for key, item in doc_link_dict.get("doc_tag").items():
                if self.product_mode in key:
                    return item, doc_link_dict.get("method")
        return "", ""

    def get_interface_result(self, interface_key, param):
        return interface_check.execute(self.context,
                                       interface_config.get(interface_key),
                                       param)
