# -*- coding: UTF-8 -*-

#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

import os
import re
import time

import cbb.frame.base.config as config
import cbb.frame.base.logger as logger
import cbb.frame.base.resource as resource
from cbb.common.conf import productConfig
from cbb.frame.base.regex import Regex
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil
from com.huawei.ism.tool.protocol.tlv.exception import TLVException
from com.huawei.ism.exception import IsmException
from java.lang import Exception as JException
from cbb.frame.rest import restUtil
from cbb.frame.rest import restData
import cbb.business.operate.fru.common.BaseFactory as BaseFactory
from utils import Products

IPV4 = "v4"
IPV6 = "v6"
# IPv4正则表达式定义
IPV4_REGEX_DEFINE = r"^((25[0-5]|2[0-4]\d|[0-1]?\d\d?)\.){3}" \
                    r"(25[0-5]|2[0-4]\d|[0-1]?\d\d?)$"
# IPv6正则表达式定义
IPV6_REGEX_DEFINE = r"^\s*((([0-9A-Fa-f]{1,4}:){7}(([0-9A-Fa-f]{1,4})|:))|" \
                    r"(([0-9A-Fa-f]{1,4}:){6}(:|((25[0-5]|2[0-4]\d|[01]?" \
                    r"\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})|" \
                    r"(:[0-9A-Fa-f]{1,4})))|(([0-9A-Fa-f]{1,4}:){5}" \
                    r"((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]" \
                    r"\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|" \
                    r"(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){0,1}" \
                    r"((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]" \
                    r"\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|" \
                    r"(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){0,2}((:((" \
                    r"25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]" \
                    r"\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|" \
                    r"(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){0,3}((:((" \
                    r"25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]" \
                    r"\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|" \
                    r"(([0-9A-Fa-f]{1,4}:)(:[0-9A-Fa-f]{1,4}){0,4}((:((" \
                    r"25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]" \
                    r"\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|" \
                    r"(:(:[0-9A-Fa-f]{1,4}){0,5}((:((25[0-5]|2[0-4]\d|[01]?" \
                    r"\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|" \
                    r"((:[0-9A-Fa-f]{1,4}){1,2})))|(((25[0-5]|2[0-4]\d|[01]?" \
                    r"\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})))" \
                    r"(%.+)?\s*$"

NUM_VER_REGEX = r"(\d+\.\d+\.\w+).*"


def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    """资源国际化

    :param lang: 语言lang
    :param res: 资源
    :param args: 资源对应的参数
    :param resDict: 资源字典
    :return: 经过国际化处理后的资源
    """
    # noinspection PyBroadException
    try:
        key = "%s_%s" % (res, lang)
        if key not in resDict.keys():
            return "--", ""

        context = resDict.get(key)
        if "%s" in context or "%i" in context:
            context = context % args

        return context

    except Exception:
        return "--", ""


def getPyResource(lang, msg, args="", resource=resource.MESSAGES_DICT):
    """消息国际化

    :param lang: 语言lang
    :param msg: 消息
    :param args: 消息参数
    :param resource: 消息字典
    :return: 经过国际化处理后的消息
    """
    errMsg = "--"
    # noinspection PyBroadException
    try:
        if msg not in resource.keys():
            return errMsg

        localeDict = resource.get(msg)
        if lang not in localeDict.keys():
            return errMsg

        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except Exception:
        return errMsg


def getMsg(lang, msg, errMsgArgs="", suggestionArgs="",
           msgDict=resource.MESSAGES_DICT):
    """
    错误消息和修复建议国际化
    :param lang: 语言lang
    :param msg: 消息
    :param errMsgArgs: 错误对应的参数
    :param suggestionArgs: 修复建议对应的参数，有多个参数时，以元组方式传入，单个参数时，以字符串方式传入
    :param msgDict: 消息和修复建议字典，有多个参数时，以元组方式传入，单个参数时，以字符串方式传入
    :return: 经过国际化处理后的消息和修复建议
    """
    try:
        if msg not in msgDict:
            return ("--", "")

        localeDict = msgDict.get(msg)
        errMsg = localeDict.get("errMsg_%s" % lang, "--")
        suggestion = localeDict.get("suggestion_%s" % lang, "")

        if "%s" in errMsg or "%i" in errMsg:
            errMsg = errMsg % errMsgArgs
        if "%s" in suggestion or "%i" in suggestion:
            suggestion = suggestion % suggestionArgs

        return (errMsg, suggestion)

    except (Exception, JException):
        return ("--", "")


def getBaseName(filePath):
    """返回文件路径的文件名，不包含后缀

    :param filePath: 文件路径
    :return: 返回不包含后缀的文件名字符串
    """
    baseName, ext = os.path.splitext(os.path.basename(filePath))
    return baseName


def getLogger(loggerInstance, pyFilePath):
    """获取日志类

    :param loggerInstance: logger实例
    :param pyFilePath: py文件路径
    :return:
    """
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)


def cmpPorts(port1, port2):
    """端口字典比较

    :param port1: 端口字典1
    :param port2: 端口字典2
    :return: 根据location进行比较的结果
    """
    if port1["location"] < port2["location"]:
        return -1
    elif port1["location"] > port2["location"]:
        return 1
    return 0


def getPortSlot(portInfo):
    """根据端口信息，获取该端口的槽位号

    :param portInfo: 端口信息
    :return: 端口的槽位号
    """
    return portInfo["location"].split(".")[-1]


def getVRCVersion(version):
    """获取产品的C版本信息

    :param version: 产品版本
    :return: 产品C版本
    """
    matched = re.match(NUM_VER_REGEX, version)
    if matched:
        return matched.group(1)

    if len(version) <= 11:
        return version
    else:
        return version[0:11]


def bit2GB(bit):
    """比特容量转换为GB容量

    :param bit:
    :return: 转换后的GB容量
    """
    return bit * 1.0 / 1024 / 1024 / 1024


def isPureDigit(digitStr):
    """ 判断字符串是否为数字（如果字符串以0开头，不认为是数字）

    :param digitStr:
    :return:True：是数字，False：非数字
    """
    if not digitStr.isdigit():
        return False

    if len(digitStr) != len(str(int(digitStr))):
        return False

    return True


def getIpVer(accessIP):
    """获取IP地址版本号

    :param accessIP: IP
    :return: IP类型
    """
    if ":" in accessIP:
        return IPV6
    return IPV4


def isIllegalManIpAddr(manIpAddr, ipVer):
    """校验管理IP地址是否合法（要求为合法的主机IP地址）

    :param manIpAddr: 管理IP
    :param ipVer: ip类型
    :return: True: 管理IP合法 , False: 管理IP不合法
    """
    if ipVer is None:
        # IPv4地址
        if ":" not in manIpAddr:
            return isIllegalManIpv4Addr(manIpAddr)
        # 考虑IPv6地址有兼容IPv4的情况
        return isIllegalManIpv4Addr(manIpAddr) or \
            isIllegalManIpv6Addr(manIpAddr)

    if ipVer == IPV4:
        return isIllegalManIpv4Addr(manIpAddr)

    if ipVer == IPV6:
        return isIllegalManIpv6Addr(manIpAddr)

    return False


def isIllegalManIpv4Addr(manIpAddr):
    """校验管理IP地址是否为合法的IPv4地址

    :param manIpAddr: 管理IP
    :return: True: 合法的IPv4地址, False: 不合法的IPv4地址
    """
    if not isIpV4(manIpAddr):
        return False

    addrs = manIpAddr.split(".")

    if int(addrs[0]) == 0:
        return False

    return True


def isIllegalManIpv6Addr(manIpAddr):
    """校验管理IP地址是否为合法的IPv6地址

    :param manIpAddr: 管理IP
    :return: True: 合法的IPv6地址, False: 不合法的IPv6地址
    """
    return isIpV6(manIpAddr)


def isIllegalMask(mask, ipVer):
    """校验IPV4子网掩码/IPV6前缀是否合法

    :param mask: 子网掩码/前缀
    :param ipVer: ip类型
    :return: True: 子网掩码/前缀合法, False: 子网掩码/前缀不合法
    """
    if ipVer is None:
        return isIllegalSubnetMask(mask) or isIllegalPrefix(mask)

    if ipVer == IPV4:
        return isIllegalSubnetMask(mask)

    if ipVer == IPV6:
        return isIllegalPrefix(mask)

    return False


def isIllegalSubnetMask(mask):
    """校验IPV4子网掩码是否合法

    :param mask: IPV4子网掩码
    :return: True: IPV4子网掩码合法, False: IPV4子网掩码不合法
    """
    if not isIpV4(mask):
        return False

    addrs = mask.split(".")

    if int(addrs[0]) == 0:
        return False

    return True


def isIllegalPrefix(mask):
    """校验IPV6前缀是否合法

    :param mask: IPV6前缀
    :return: True: IPV6前缀合法, False: IPV6前缀不合法
    """
    if not isPureDigit(mask):
        return False

    if 0 <= int(mask) <= 128:
        return True
    else:
        return False


def isIllegalGateway(gateway, ipVer):
    """校验IPV4网关地址/IPV6网关地址是否合法

    :param gateway: 网关地址
    :param ipVer: ip类型
    :return: True: 网关地址, False: 网关地址
    """
    if ipVer is None:
        return isIllegalIPv4Gateway(gateway) or isIllegalIPv6Gateway(gateway)

    if ipVer == IPV4:
        return isIllegalIPv4Gateway(gateway)

    if ipVer == IPV6:
        return isIllegalIPv6Gateway(gateway)

    return False


def isIllegalIPv4Gateway(gateway):
    """校验IPv4网关地址是否合法

    :param gateway: IPv4网关地址
    :return: True: IPv4网关地址合法, False: IPv4网关地址不合法
    """
    return isIpV4(gateway)


def isIllegalIPv6Gateway(gateway):
    """校验IPv6网关地址是否合法

    :param gateway: IPv6网关地址
    :return: True: IPv6网关地址合法, False: IPv6网关地址不合法
    """
    return isIpV6(gateway)


# ---------New Frame---------
def isIpV4(ip):
    """检测IP地址是否为通用的IPv4地址

    :param ip: ip
    :return:True: IPv4地址合法, False: IPv4地址不合法
    """
    return Regex.find(IPV4_REGEX_DEFINE, ip)


def isIpV6(ip):
    """检测IP地址是否为通用的IPv6地址

    :param ip:ip地址
    :return:True: IPv6地址合法, False: IPv6地址不合法
    """
    return Regex.find(IPV6_REGEX_DEFINE, ip)


def getBaseUri(ip, devSN, port=config.REST_DEFAULT_PORT):
    """获取rest的基础链接（https://${ip}:${port}/deviceManager/rest/${deviceId}/）

    :param ip: 设备ip
    :param devSN: 设备SN
    :param port: rest使用的端口
    :return: rest及基础链接
    """
    return config.REST_BASE_URI % (ip, port, devSN)


def has_fan_confirm_tip(context, product_model):
    """更换风扇是否有高危提示

    :param product_model: 产品型号
    :return: True:有风险，False：无风险
    """
    # 超低端的风扇为普通风险
    if is_ultra_low_end(context, product_model):
        return False
    return isV5V6LowEnd(product_model) or isV5V6MidEnd(product_model)


def isHighEndDev(productModel):
    """判断是否为高端设备

    :param productModel:产品型号
    :return: True: 产品型号为高端, False: 产品型号不为高端
    """
    return productModel in config.HIGH_END_DEVS


def isNeedDrawGraph(productModel):
    """判断是否需要画设备图

    :param productModel:产品型号
    :return: True: 需要画设备图, False: 不需要画设备图
    """
    return productModel in config.NEED_DRAW_GRAPH_DEVS


def isArmDev(context, productModel):
    """判断是否为ARM设备

    :param context: 上下文信息
    :param productModel:产品型号
    :return:True：ARM设备，False：非ARM设备
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return productModel in config.ARM_DEVS_V5_NEW and "Kunpeng" in version


def isArmHighEndDev(productModel, productVersion):
    """判断是 ARM 高端设备.

    :param productModel:
    :param productVersion:
    :return:
    """
    return (
        productModel in config.ARM_HIGH_END_NEW and "Kunpeng" in productVersion
    )


def isArmDevLowEnd(context, productModel):
    """判断是否为ARM新硬件低端设备

    :param context:上下文信息
    :param productModel:产品型号
    :return:True：ARM新硬件低端设备，False：不是ARM新硬件低端设备
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return productModel in (config.ARM_LOWEND_NEW +
                            config.ARM_LOWEND_ENHANCED) and "Kunpeng" in version


def isArmDevMidEnd(context, productModel):
    """判断是否为ARM新硬件中端设备

    :param context: 上下文信息
    :param productModel:产品型号
    :return:True：ARM新硬件中高端设备，False：不是ARM新硬件中高端设备
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return productModel in config.ARM_MID_END_NEW and "Kunpeng" in version


def is_kunpeng_high_end(context, product_model):
    """判断时候为V5高端

    :param context:上下文信息
    :param product_model:产品型号
    :return:True：高端Kunpeng设备，False：非高端Kunpeng设备
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return product_model in config.ARM_HIGH_END_NEW and "Kunpeng" in version


def is_high_end_no_need_graph(context, product_model):
    """判断高端不画图

    :param context: 上下文信息
    :param product_model:产品型号
    :return:True：需要画图，False：不需要画图
    """
    high_end_models = config.DORADO_DEVS_V6_HIGH + config.ARM_HIGH_END_NEW
    return (product_model in high_end_models
            and product_model not in config.HIGH_END_NEED_GRAPH_MODELS)


def is_kunpeng_2810(context, product_model):
    """判断时候为新硬件2810（国内版、海外版、安可版）

    :param context:上下文信息
    :param product_model:产品型号
    :return: True：新硬件2810（国内版、海外版、安可版），False：不是新硬件2810（国内版、海外版、安可版）
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return product_model in config.ARM_2810_NEW and "Kunpeng" in version


def is_2810_hyper(context, product_model):
    """判断是否为支持双活的2810 V5 （C72及以后）

    :param context:上下文信息
    :param product_model:产品型号
    :return: True：2810 V5 C72及以后，False：不是2810 V5 C72及以后
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return product_model == "2810 V5" and \
        Products.compareVersion(version, "V500R007C72 Kunpeng") >= 0


def is_ultra_low_end(context, product_model):
    """判断是否为2210 ,2200 V5
    或者新增背板型号如下："5110ES_V5_16G_P", "5110ES_V5_P", "5210ES_V5_P"
    或者 OceanStor Dorado 2000，OceanStor Dorado 2020， OceanStor Dorado 2100, OceanProtect X3000，新融合入门级型号
    :param context:上下文信息
    :param product_model:产品型号
    :return: True：在范围内，False：不是范围内
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    if is_short_enc_low_end(product_model):
        return True
    if product_model in config.ULTRA_LOW_END_NEW and version.startswith("V500R007C73"):
        return True
    # 5110E，5210E新增背板型号，新增背板型号如下："5110ES_V5_16G_P", "5110ES_V5_P", "5210ES_V5_P"
    if get_internal_product_model(context) in config.ULTRA_LOW_END_NEW_INTERNAL_PRODUCT_MODEL:
        return True
    return False


def is_kunpeng_middle_end(context, product_model):
    """判断时候为V5中低端

    :param context:上下文信息
    :param product_model:产品型号
    :return:True：高端Kunpeng设备，False：非高端Kunpeng设备
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return product_model not in config.ARM_HIGH_END_NEW \
        and "Kunpeng" in version


def is_sup_vm_by_context(context, product_model):
    """判断是否为支持虚拟机的Kunpeng新硬件

    :param context:上下文信息
    :param product_model:产品型号
    :return: True：支持虚拟机，False：不支持虚拟机
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)
    return is_sup_vm_by_version(product_model, version)


def is_sup_vm_by_version(product_model, product_version):
    """判断是否为支持虚拟机的Kunpeng新硬件

    :param product_model: 产品型号
    :param product_version: 产品版本
    :return:
    """
    return (product_model in config.ARM_SUP_VIRTUAL_MACHINE
            and "Kunpeng" in product_version)


def is_v5_middle_new(product_model):
    """判断是否为V5中低端（支持交换机型号）

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.V5_MIDDLE_NEW


def is_v5_middle_c71_new(context, product_model):
    """判断是否为V5中低端（支持交换机型号）且支持C71版本

    :param context:上下文信息
    :param product_model: 产品型号
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)

    return product_model in config.V5_MIDDLE_NEW \
        and Products.compareVersion(version, "V500R007C71 Kunpeng") >= 0


def is_dorado_612_high(context, product_model):
    """判断是否为V6高端且版本大于6.1.2RC1

    :param context:上下文信息
    :param product_model: 产品型号
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    _, version, _ = cliUtil.getProductVersion(cli, lang)

    return product_model in config.DORADO_DEVS_V6_HIGH \
        and Products.compareVersion(version, "6.1.2RC1") >= 0


def is_ocean_protect_high(product_model):
    """判断是否为OceanProtect 高端

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.OCEAN_PROTECT_HIGH


def is_ocean_protect(product_model):
    """判断是否为OceanProtect

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.OCEAN_PROTECT


def is_ocean_protect_support_switch(product_model):
    """判断是否为支持交换机组网的OceanProtect

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.OCEAN_PROTECT_SUPPORT_SWITCH


def is_support_16_ctrl_with_nas(product_model, product_version):
    """
    判断是否为支持带NAS扩容至16控（D5600/D6000 V7及以后版本）
    :param product_model: 产品型号
    :param product_version: 软件版本
    :return:
    """
    return product_model in config.DORADO_32_CTRL_DEVS and product_version.startswith("V700")


def is_ocean_protect_x(product_model):
    """判断是否为OceanProtect X系列的设备

    :param product_model: 产品型号
    :return: 是否为X系列的设备。
    """
    return product_model in config.OCEAN_PROTECT_X


def is_new_oceanstor(product_model):
    """判断是否为新融合

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.NEW_DORADO


def is_ocean_protect_mid(product_model):
    """判断是否为OceanProtect 中端

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.OCEAN_PROTECT_MID


def is_v5_high_new(product_model):
    """判断是否为V5高端

    :param product_model: 产品型号
    :return:
    """
    return product_model in config.V5_HIGH_NEW


def isArmDevV5New(productModel):
    """判断是否为ARM设备

    :param productModel: 产品型号
    :return: True：ARM新硬件设备，False：非ARM新硬件
    """
    return productModel in config.ARM_DEVS_V5_NEW


def isArmDev2U(productModel):
    """判断是否为2U ARM设备

    :param productModel: 产品型号
    :return:True：2U ARM设备，False：非2U ARM设备
    """
    return productModel in config.ARM_DEVS_2U


def isDoradoNAS(productModel):
    """判断是否稳Dorado NAS

    :param productModel: 产品型号
    :return: True：Dorado NAS设备，False：非Dorado NAS设备
    """
    if "DORADO NAS" in productModel.upper():
        return True
    return False


def isDoradoDev(productModel):
    """判断是否为Dorado设备

    :param productModel: 产品型号
    :return:True：Dorado设备，False：非Dorado设备
    """
    return productModel in config.DORADO_DEVS


def is_micro_dev(product_model):
    """
    判断是否为微存储设备
    :param product_model:
    :return:
    """
    return product_model in config.OCEAN_STOR_MICRO_DEVS


def is_micro_flashing_dev(pdt_inter_model):
    """
    判断是否为微存储混闪形态
    @param pdt_inter_model: 设备内部型号
    @return: 是否为混闪形态
    """
    return pdt_inter_model in config.MICRO_FLASHING_INTER_MODEL


def is_micro_1300(product_model):
    """
    判断是否为OceanStor Micro 1300
    :param product_model: 设备型号
    :return: True：是OceanStor Micro 1300， False： 不是OceanStor Micro 1300
    """
    return product_model in ("OceanStor Micro 1300", "OceanDisk 1300")


def no_support_ip_scale_out_end(product_model):
    """
    判断是否为不支持ip scale out 组件的设备
    :param product_model: 设备型号
    :return: True：是 False： 不是
    """
    return any([is_micro_1300(product_model),
                is_ocean_protect_x3000(product_model),
                product_model in ("OceanStor 2200", "OceanStor 2220")])


def is_computing_dev(product_model):
    """
    判断是否为微存储设备
    :param product_model:
    :return:
    """
    return product_model in config.OCEAN_STOR_COMPUTING_DEVS


def is_big_clusters_ai_dev(product_model):
    """
    判断是否为大集群AI存储设备
    :param product_model: 产品型号
    :return: True: 是 False:不是
    """
    return product_model in config.BIG_CLUSTERS_AI_DEVS


def is_dorado_2000(product_model):
    """
    判断是否为Dorado 2000、Dorado 2100、Dorado 2020
    :param product_model: 设备型号
    :return: True: 是 False:不是
    """
    return product_model in ("OceanStor Dorado 2000", "OceanStor Dorado 2100", "OceanStor Dorado 2020")


def is_ocean_protect_x3000(product_model):
    """
    判断是否为OceanProtect X3000
    :param product_model: 设备型号
    :return: True: 是 False:不是
    """
    return product_model == "OceanProtect X3000"


def is_short_enc_low_end(product_model):
    """
    判断是否为超低端短框设备
    （OceanProtect X3000，OceanStor Dorado 2000，OceanStor Dorado 2020，OceanStor Dorado 2100, 新融合入门级型号），
    该型号硬件和融合V5超低端保持一致
    :param product_model: 设备型号
    :return: True: 是 False:不是
    """
    return is_dorado_2000_or_protect_x3000(product_model) or product_model in config.HYBRID_V6_ENTRY_LEVEL_END


def is_dorado_2000_or_protect_x3000(product_model):
    """
    判断是否为OceanProtect X3000 或 OceanStor Dorado 2000, OceanStor Dorado 2020, OceanStor Dorado 2100
    :param product_model: 设备型号
    :return: True: 是 False:不是
    """
    return product_model in config.DORADO_2000_AND_PROTECT_X3000


def isDoradoV6Dev(productModel):
    """判断是否为DoradoV6设备

    :param productModel: 产品型号
    :return:True：Dorado V6设备，False：非Dorado V6设备
    """
    return productModel in config.DORADO_DEVS_V6


def is_dorado_v3(product_model):
    """判断是否为DoradoV3设备

    :param product_model: 产品型号
    :return:True：Dorado V3设备，False：非Dorado V3设备
    """
    return product_model in config.DORADO_DEVS_V3


def isDoradoV6HighEnd(productModel):
    """判断是否为DoradoV6高端设备

    :param productModel: 产品型号
    :return:True：Dorado V6高端设备，False：非Dorado V6高端设备
    """
    return productModel in config.DORADO_DEVS_V6_HIGH


def isDoradoV6Mid(productModel):
    """判断是否为DoradoV6中端设备

    :param productModel: 产品型号
    :return:True：Dorado V6中端设备，False：非Dorado V6中端设备
    """
    return productModel in config.DORADO_DEVS_V6_MID


def isV5V6LowEnd(productModel):
    """ 判断是否为V6/V5(仅限于带Kunpeng的V5)新硬件低端设备

    :param productModel: 产品型号
    :return:True：为V6/V5(仅限于带Kunpeng的V5)新硬件低端设备，
            False：不是V6/V5(仅限于带Kunpeng的V5)新硬件低端设备
    """
    return productModel in config.DORADO_DEVS_V6_LOW or \
        productModel in config.ARM_LOWEND_NEW or \
        productModel in config.ARM_LOWEND_ENHANCED


def isV5LowEndEnhanced(productModel):
    """ 判断是否为V5(仅限于带Kunpeng的V5)超低端设备

    :param productModel: 产品型号
    :return:True：为VV5(仅限于带Kunpeng的V5)超低端设备
            False：不是V6/V5(仅限于带Kunpeng的V5)超低端设备
    """
    return productModel in config.ARM_LOWEND_ENHANCED


def isV5V6MidEnd(productModel):
    """ 判断是否为V6/V5(仅限于带Kunpeng的V5)新硬件中端设备

    :param productModel: 产品型号
    :return:True：为V6/V5(仅限于带Kunpeng的V5)新硬件中端设备，
            False：不是V6/V5(仅限于带Kunpeng的V5)新硬件中端设备
    """
    return productModel in config.DORADO_DEVS_V6_MID or \
        productModel in config.ARM_MID_END_NEW


def isV5V6HighEnd(productModel):
    """ 判断是否为V6/V5(仅限于带Kunpeng的V5)新硬件高端设备

    :param productModel: 产品型号
    :return:True：为V6/V5(仅限于带Kunpeng的V5)新硬件高端设备，
            False：不是V6/V5(仅限于带Kunpeng的V5)新硬件高端设备
    """
    return productModel in config.DORADO_DEVS_V6_HIGH or \
        productModel in config.ARM_HIGH_END_NEW


def isFullBayDilverProduct(productModel):
    """ 判断是否为整柜交付产品

    :param productModel: 产品型号
    :return:True：整柜交付产品， False：不是整柜交付产品
    """
    return productModel in (config.V6_SOLID_FULL_BAY_DELIVER_PDT_MODEL +
                            config.V5_HYBRID_FULL_BAY_DELIVER_PDT_MODEL)


def is_full_bay_support_4u(product_model, internal_product_model):
    """判断是否是支持4U框的整柜型号

    :param product_model:产品型号
    :param internal_product_model:内部型号
    :return:True：支持4U框的整柜设备，False：不支持4U框的整柜设备
    """
    return product_model in config.V5_FULL_BAY_SUP_4U_ENC or \
        internal_product_model in config.HYBRID_V6_FULL_BAY_SUP_4U_ENC


def is_f_series_product(product_model):
    """判断是否是F系列整柜型号

    :param product_model:产品型号
    :return:True：F系列的整柜设备，False：不是F系列的整柜设备
    """
    return product_model in config.V5_HIGH_END_NOT_SUP_4U_ENC


def isBigSasDev(productModel):
    """ 判断是否为SAS大卡设备

    :param productModel: 产品型号
    :return:True：SAS大卡设备，False：非SAS大卡设备
    """
    return productModel in config.HIGH_MID_BIGSAS


def isV5_Dev(productModel):
    """判断是否为V5设备

    :param productModel: 产品型号
    :return:True：V5设备，False：不是V5设备
    """
    return productModel in config.DEVS_V5


def is_v5_high_end_devs(productModel):
    """是否为V5高端设备

    :param productModel: 产品型号
    :return:True：V5高端设备，False：非V5高端设备
    """
    return productModel in config.HIGH_END_DEVS_V5


def safeSleep(seconds, logger=None):
    """安全睡眠时间

    :param seconds: seconds为睡眠时间，单位：秒；数据类型：整数或小数
    :return:无
    """
    # noinspection PyBroadException
    try:
        if type(seconds) not in [int, long, float] or seconds <= 0:
            return

        startTime = time.clock()
        while True:
            if (time.clock() - startTime) >= seconds:
                return

            # 睡眠一下，避免长时间占用cpu，该时间设置过长会影响睡眠时间精度
            # noinspection PyBroadException
            time_sleep(logger)
    except Exception as ex:
        if logger:
            logger.error(ex)


def time_sleep(logger_param):
    try:
        time.sleep(0.1)
    except Exception as ex:
        if logger_param:
            logger_param.error(ex)


def getRedundCtrlId(curCtrlId):
    """获取冗余控制器id(格式：*A, *B； 或：0, 1)

    :param curCtrlId: 当前控制器id
    :return:冗余控制器ID
    """
    if len(curCtrlId) == 1:
        curCtrlName = curCtrlId
        curEnginId = ""
    else:
        curCtrlName = curCtrlId[-1].upper()
        curEnginId = curCtrlId[0:-1]
    redundCtrlName = config.CONTROLLER_PAIR.get(curCtrlName)
    # pair对的控制器ID是相邻的
    redundCtrlId = curEnginId + redundCtrlName
    return redundCtrlId


def cntrlIdFormat(cntrlId):
    """控制器ID格式转换： 207.0.0 ->  0A

    :param cntrlId:控制器ID
    :return:转换后的控制器ID
    """
    if cntrlId.startswith("207."):
        encId = cntrlId.split(".")[1]
        cntrlName = config.NODE_ID_MAP.get(cntrlId.split(".")[2])
        return encId + cntrlName
    return cntrlId


def getRegPatternList(patternList):
    """获取正则匹配项

    :param patternList: 规则列表
    :return:正则匹配项
    """
    return [re.compile(pattern) for pattern in patternList]


def checkRegValue(checkValue, regList, indexList):
    """单个告警匹配过滤，过滤字段

    :param checkValue: 待过滤的告警
    :param regList: 用于过滤的正则，其中该列表中的每条记录的每个元素都会和checkIndex中对应位置上的值做匹配。
    :param indexList: 配置的需要检查的告警id、检查那个字段，检查该字段中哪些值。
        如64438009898: {"obj":restData.Alarm.EVENTPARAM,
                      "checkIndex": [1,2]},
    :return: True 存在告警， False 不存在告警
    """
    checkValue = checkValue.split(",")
    valueList = []
    # 取出对应位置上的值，存入valueList
    for index in indexList:
        if index >= len(checkValue):
            return False
        else:
            valueList.append(checkValue[index])

    for i in range(len(regList)):
        # 只要有一个找不到就退出当前备件进入下一个记录，如从级联板DAE000.A记录的检查跳到DAE001.A
        if regList[i].search(valueList[i]) is None:
            return False

    # 如果True表示该告警匹配目标备件所有规则都成功。
    return True


def formatVersion(version):
    """将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）

    :param version: 原始版本号
    :return: 有效的版本号
    """
    matched = re.match(NUM_VER_REGEX, version)
    if matched:
        return matched.group(1)
    startIndex = version.index("V")
    if Regex.find("SPC", version, Regex.IGNORECASE):
        endIndex = startIndex + 17  # 格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  # 格式：V100R005C02

    return version[startIndex:endIndex]


# 加载资源文件
try:
    from com.huawei.ism.tool.obase.utils import IniResourceUtil
    IniResourceUtil.loadPyRes("cbb/res/upgrade", "resource")
    IniResourceUtil.loadPyRes("cbb/res/upgrade", "resource_dorado")
    IniResourceUtil.loadPyRes("cbb/res/collect/host", "hostinfo")
except (Exception, JException):
    pass


def getString(key):
    """资源国际化

    :param key: 待国际化的资源项
    :return: 经过国际化处理后的资源
    """
    return IniResourceUtil.getString(key)


def getStringBy(path, name, key):
    """指定资源文件获取资源国际化

    :param path: 查找资源文件路径
    :param name: 查找资源文件名称
    :param key: 待国际化的资源项
    :return: 经过国际化处理后的资源
    """
    return IniResourceUtil.getString(path, name, key)


class UpgradeRes:
    """
    升级类工具使用的资源查找类
    """

    def __init__(self, dev_type):
        self.dev_type = dev_type

    def get_res(self, key):
        if is_dorado_series_devs(self.dev_type):
            # 如果是Dorado，优先从dorado里面去找，找不到再从全局里面找
            val = getStringBy("cbb/res/upgrade", "resource_dorado", key)
            if val and val != "--":
                return val
        return getString(key)


def is_dorado_series_devs(dev_type):
    """
    判断是否为Dorado ,二级存储，新融合，微存储。
    :param product_model: 设备型号
    :return: 结果。
    """
    return any(["dorado" in dev_type.lower(),
                is_ocean_protect(dev_type),
                is_new_oceanstor(dev_type),
                is_micro_dev(dev_type),
                is_computing_dev(dev_type),
                is_big_clusters_ai_dev(dev_type)])


def isSupportRest(devType, devVersion):
    """
    判断是否支持rest连接
    :param devType: 产品类型
    :param devVersion: 产品版本
    :return:True：支持Rest连接，False：不支持Rest连接
    """
    if isDoradoDev(devType):
        return True
    if devVersion >= "V300R006C00":
        return True
    return False


def get_logic_eng_id(ctrl_height, ctrl_list):
    """根据给出的控制器列表，计算对应的逻辑引擎列表

    :param ctrl_height: 控制框高度
    :param ctrl_list: 控制器列表
    :return: 逻辑引擎列表
    """
    logic_eng_list = []
    for ctrl_id in ctrl_list:
        eng_id = int(ctrl_id[0])
        node = ctrl_id[-1]
        if str(ctrl_height) != "6":
            logic_eng_list.append(str(eng_id))
        else:
            logic_eng_list.append(str(eng_id * 2 +
                                      productConfig.NODE_MAP.get(node, 0)))
    return list(set(logic_eng_list))


def get_ctrl_by_logic_eng_id(ctrl_height, logic_eng_id):
    """根据逻辑引擎ID，计算对应的控制器列表

    :param ctrl_height:控制框高度
    :param logic_eng_id:逻辑引擎id
    :return:控制器列表
    """
    ctrl_list = []
    if str(ctrl_height) != "6":
        ctrl_list.append(str(logic_eng_id) + "A")
        ctrl_list.append(str(logic_eng_id) + "B")
    else:
        phy_eng_id = int(logic_eng_id) / 2
        if int(logic_eng_id) % 2 == 0:
            ctrl_list.append(str(phy_eng_id) + "A")
            ctrl_list.append(str(phy_eng_id) + "B")
        else:
            ctrl_list.append(str(phy_eng_id) + "C")
            ctrl_list.append(str(phy_eng_id) + "D")
    return ctrl_list


def get_fan_num_per_ctrl(product_model):
    """V5/V6新硬件获取每控冗余风扇数量

    :param product_model: 产品型号
    :return:冗余风扇数量
    """
    if isV5V6LowEnd(product_model):
        pdt_type = "low_end"
    elif isV5V6MidEnd(product_model):
        pdt_type = "mid_end"
    else:
        pdt_type = "high_end"
    return productConfig.PDT_FAN_NUM_PER_CTRL.get(pdt_type, 4)


def has_svp_module(context):
    """ 是否带SVP
        针对融合高端V5R7C60及之后版本SVP选配场景，兼容老工具箱+新子工具箱场景
    :param context: 上下文对象
    :return:
    """
    dev_type = contextUtil.getProductModel(context)
    version = contextUtil.getCurVersion(context)
    dev_node = contextUtil.get_base_dev(context)
    log = getLogger(context.get("logger"), __file__)
    log.logInfo("cbb version:{}".format(version))
    log.logInfo("cbb dev_type:{}".format(dev_type))
    if isHighEndDev(dev_type):
        if version < "V500R007C60":
            return True
        try:
            svp_module_info = dev_node.getHighDevSVPModuleInfo()
            log.logInfo("SVP module info result: %s" % svp_module_info)
            if svp_module_info == config.SvpResult.NEW_TOOLBOX_NO_SVP:
                return False
        except (JException, Exception) as ex:
            log.logException(ex)
            log.logInfo("The toolkit may not be latest.")
    return False


def get_slot_id_list(context):
    """获取系统节点类别

    :param context:
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    slot_id_list = list()
    cli_ret = cliUtil.executeCmdInDebugMode(
        cli, cliUtil.DIAGNOSE_SYS_SHOW_CLS_CMD, True, lang)
    if cli_ret[0] is not True:
        return False, "", slot_id_list
    vertical_ret = cliUtil.getVerticalCliRet(cli_ret[1])
    horizontal_ret = cliUtil.getHorizontalNostandardCliRet(cli_ret[1])
    current_node = vertical_ret[0].get("local node id", "")
    for horizontal_info in horizontal_ret:
        slot_id = horizontal_info.get("id", "")
        slot_id_list.append(slot_id)
    return True, current_node, slot_id_list


def open_tlv_channel(context):
    """打开设备tlv通道

    :param context:
    :return:
    """
    log = getLogger(context.get("logger"), __file__)
    try:
        pdt_model = contextUtil.getProductModel(context)
        pdt_version = contextUtil.getCurVersion(context)
        # 需要开启tlv通道的版本
        need_open_tlv_ver = ["V300R003", "V300R005", "V300R006"]
        if not isHighEndDev(pdt_model) \
                and len(pdt_version) >= 11 \
                and pdt_version[:8] in need_open_tlv_ver:
            # 打开tlv通道
            cli = contextUtil.getCli(context)
            lang = contextUtil.getLang(context)
            result = cliUtil.openTlvChannel(cli, lang)
            log.logInfo("Open tlv channel result:{}".format(result))
            return
    except (JException, Exception) as ex:
        log.logException(ex)
        return


def is_support_NVMe_intf(context):
    """
    是否支持NVMe前端接口卡
    条件：判断DoradoV6型号且版本在6.1.及之后
    例外：新融合系列，D2000

    :param context:
    :return:
    """
    pdt_model = contextUtil.getProductModel(context)
    pdt_version = contextUtil.getCurVersion(context)
    return all((
        not is_dorado_2000(pdt_model),
        pdt_model not in config.HYBRID_V6_ENTRY_LEVEL_END,
        isDoradoV6Dev(pdt_model),
        pdt_version.startswith("6.1.")
    ))


def get_ctrl_num_one_engine(cli, lang):
    """
    获取单引擎控制器数量
    :return:
    """
    flag, cli_ret, err_msg, node_tuples \
        = cliUtil.getControllerEngineTopography(cli, lang)
    ctrl_num = len(node_tuples[2].get("0"))
    return ctrl_num


# 控制器名称在单引擎上对应的节点号
CTRL_NAME_TO_NODE_MAP = {
    "A": 0,
    "B": 1,
    "C": 2,
    "D": 3
}


def get_node_id_by_ctrl_id(ctrl_id, engine_height):
    eng_id = int(ctrl_id[:-1])
    ctrl_name = ctrl_id[-1]
    node_id = eng_id * engine_height + CTRL_NAME_TO_NODE_MAP.get(ctrl_name)
    return str(node_id)


def get_ctrl_id_by_node_id(node_id, ctrl_num):
    ctrl_trans_dict = {"0": "A", "1": "B", "2": "C", "3": "D"}
    return str(int(node_id / ctrl_num)) + ctrl_trans_dict[
        str(node_id % ctrl_num)]


def is_errcode_in_exception(err_code, exception):
    """ 判断异常信息中是否包含指定错误码

    :param err_code: 错误码
    :param exception: 异常信息
    :return:
    """
    try:
        if isinstance(exception, Exception) and exception.args:
            return str(err_code) in [str(arg) for arg in exception.args]
        if isinstance(exception, (IsmException, TLVException)):
            return str(exception.getErrorId()) == str(err_code)
        return False
    except (JException, Exception):
        return False


def is_v5v6_support_switch(pdt_model, full_pdt_ver):
    """ 判断V5Kunpeng/DoradoV6版本是否支持交换机

    :param pdt_model: 产品型号
    :param full_pdt_ver: 产品版本
    :return:
    """
    if is_ocean_protect_support_switch(pdt_model):
        return True

    return any([all([is_v5_high_new(pdt_model),
                     full_pdt_ver >=
                     productConfig.VersionManager.V5_SWITCH]),
                all([is_v5_middle_new(pdt_model),
                     full_pdt_ver >=
                     productConfig.VersionManager.V5_MIDDLE_SWITCH]),
                all([isDoradoV6Dev(pdt_model),
                     productConfig.compare_dorado_version(
                         full_pdt_ver,
                         productConfig.VersionManager.V6_SWITCH)[0],
                     productConfig.compare_dorado_version(
                         full_pdt_ver,
                         productConfig.VersionManager.V6_SWITCH)[1]])])


def is_support_sas_nvme_mix(context, product_model):
    """ 判断是否为支持NVme和SAS混插的设备

    :param context
    :param product_model:产品型号
    """
    params = {}
    rest = contextUtil.getRest(context)
    uri_param_dict = restUtil.CommonRest.getUriParamDict(
        restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO)
    record = restUtil.CommonRest.execCmd(rest, uri_param_dict, params,
                                         restData.RestCfg.RestMethod.GET)
    if not record:
        BaseFactory.log.error(context, "get internal productModel failed.")
        # 如果取得为空的话，直接返回None
        return False
    else:
        internal_product_model = restUtil.CommonRest.getRecordValue(
            restUtil.CommonRest.getData(record),
            restData.InternalDeviceInfo.INTERNAL_PRODUCT_MODEL)
        BaseFactory.log.info(context, "Product internal model:{}".format(
            internal_product_model))
        return internal_product_model.upper() in config.\
            PDT_MODEL_SUPPORT_SAS_NVME_MIX


def is_special_patch(patch_version):
    """
    是否特殊补丁
    :param patch_version:
    :return: True 是，False 不是
    """
    return re.search(r"SPH\d[6-9]\d", patch_version) is not None


def group_by(dict_list, target_key):
    result = {}

    def compute_if_absent(key):
        if key not in result:
            result[key] = []
        return result.get(key)
    for item in dict_list:
        val = item.get(target_key)
        compute_if_absent(val).append(item)
    return result


def get_internal_product_model(context):
    """通过rest接口查询内部产品型号(FRU移植)

    :param context:
    :return:
    """
    logger = getLogger(context.get("logger"), __file__)
    devObj = contextUtil.getDevObj(context)
    pdtVersion = devObj.get("version")
    pdtType = devObj.get("type")
    logger.logInfo("Product version:{}, and product model:{}".format(pdtVersion, pdtType))
    if not ((pdtType and 'Dorado' in pdtType and (pdtVersion > 'V300R002C10' or '.' in pdtVersion))  # For dorado
            or pdtType in config.OCEAN_PROTECT or pdtType in config.OCEAN_STOR_MICRO_DEVS
            or (pdtType and 'V5' in pdtType and pdtVersion >= 'V500R007C60')):  # 5X10 V5
        logger.logInfo("Internal productModel not supported.")
        return ''

    params = {}
    rest = contextUtil.getRest(context)
    uriParamDict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO)
    record = restUtil.CommonRest.execCmd(rest, uriParamDict, params, restData.RestCfg.RestMethod.GET)
    if not record:
        logger.logInfo("get internal productModel failed.")
        # 如果取得为空的话，直接返回None
        return None
    else:
        internalProductModel = restUtil.CommonRest.getRecordValue(restUtil.CommonRest.getData(record),
                                                                  restData.InternalDeviceInfo.INTERNAL_PRODUCT_MODEL)
        logger.logInfo("Product internal model:{}".format(internalProductModel))
        return internalProductModel
