# -*- coding: UTF-8 -*-
import re
from cbb.frame.base import config

KUNPENG = "Kunpeng"


def isKunpeng(version):
    """
    根据产品版本判断是否为Kunpeng设备
    :param version: 产品版本号
    :return: True：Kunpeng设备，False：非Kunpeng设备
    """
    if KUNPENG in version and version >= "V500R007C60":
        return True
    return False


def isDigitalVer(version):
    """
    判断是否为数字版本号
    :param version: 版本号
    :return: True：数字版本号，False：非数字版本号
    """
    if re.compile(r"\d+\.\d+.*").match(version):
        return True
    return False


def compare_version(version1, version2):
    """
    判断是否为dorado系列的产品，版本的特点位数字版本和V7版。
    :param version1: 版本号
    :param version2: 版本号
    :return: 结果 1：大于，0：等于，-1：小于
    """
    from utils import Products
    return Products.compareVersion(version1, version2)


def is_dorado_series_version(version):
    """
    判断是否为dorado系列的产品，版本的特点位数字版本和V7版。
    :param version: 版本号
    :return: True：数字版本号，False：非数字版本号
    """
    return isDigitalVer(version) or compare_version(version, "V700R001C00") >= 0


class KunpengProduct:
    @staticmethod
    def getFullVersion(demo, version):
        """
        获取完整版本号（版本号+Kunpeng）
        :param demo:
        :param version: 版本号
        :return: 完整版本号
        """
        if KUNPENG in demo and KUNPENG not in version:
            version += " " + KUNPENG
        return version


class Feature:
    @staticmethod
    def isSupportFastUpgrade(devType, version):
        """
        根据产品型号和版本号判断是否支持快速升级
        :param devType: 产品型号
        :param version: 产品版本
        :return:True：支持快速升级，False：不支持快速升级
        """
        if "dorado" in devType.lower() and isDigitalVer(version):
            return True
        if compare_version(version, "V700R001C00") >= 0:
            return True
        if "oceanprotect" in devType.lower():
            return True
        if devType in config.NEW_DORADO:
            return True
        if devType in config.OCEAN_STOR_MICRO_DEVS:
            return True
        if devType in config.OCEAN_STOR_COMPUTING_DEVS:
            return True
        return False

    @staticmethod
    def isSupportInnerCheck(devType, version):
        """
        根据产品型号和版本号判断是否支持在阵列执行升级前检查
        :param devType:产品型号
        :param version:产品版本
        :return:True：支持在阵列执行升级前检查，False：不支持在阵列执行升级前检查
        """
        if Feature.isSupportFastUpgrade(devType, version):
            return True
        if isKunpeng(version):
            return True
        return False

    @staticmethod
    def is_support_cross_engine_ssh(dev_ver, dev_type):
        """
        通过版本型号判断是否支持跨ssh连接
        :return: True/False
        """
        support_start_version = {"V3": "V300R006C00SPC100",
                                 "V5": "V500R007C00SPC100"}
        if isDigitalVer(dev_ver):
            return True
        if compare_version(dev_ver, "V700R001C00") >= 0:
            return True
        if "Dorado" in dev_type:
            if dev_ver < "V300R001C20":
                return False
            return True
        for version_start in support_start_version:
            version = support_start_version.get(version_start)
            if dev_ver.startswith(version_start) and dev_ver < version:
                return False
        return True


def is_dorado_v3(dev_type, dev_version):
    """
    是否Dorado V3
    :param dev_type:
    :param dev_version:
    :return:
    """
    if "dorado" in dev_type.lower() and "v3" in dev_version.lower() \
            and "dorado nas" != dev_type.lower():
        return True
    return False
