# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

import os
import re
import traceback
from com.huawei.ism.exception import IsmException
from java.lang import Exception as JException
from java.lang import Throwable

from cbb.frame.base import baseUtil
from cbb.frame.base.config import STOR_DEV_INFO_DICT
from cbb.frame.base.funcUtils import wrap_all_exception_logged
from cbb.frame.base.regex import Regex
from cbb.frame.cli import cliResource
from cbb.frame.context import contextUtil
from cbb.frame.rest import restUtil

MAX_RETRYS = 5
CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEBUG_MODEL_FLAG = ":/diagnose>"
DEVELOPER_MODEL_FLAG = "developer@*.*?:/>"
CLI_RET_END_FLAG = ":/>"
DIAGNOSE_RET_END_FLAG = "diagnose>"
MINISYSTEM_MODEL_FLAG = "minisystem>"
STORAGE_MODEL_FLAG = "Storage:~ #"
DIAGNOSE_SYS_SHOW_CLS_CMD = 'sys showcls'
MINISYSTEM_SYS_SHOW_CLS_CMD = 'showsysstatus'
VSTORE_MODEL_FLAG = ".*@.*:/>$"

SERIES_T = "T_SERIES"
SERIES_V3 = "V3_SERIES"
SERIES_18000 = "18000_SERIES"

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"
RESULT_DICT = {
    True: "0",
    False: "1",
    RESULT_NOCHECK: "2",
    RESULT_NOSUPPORT: "3",
    RESULT_WARNING: "4",
}
CLI_CAN_NOT_CONNECT_CODE = "1073949185"  # 与设备通信异常，请检查网络连接或设备状态是否正常

ROLE_ID_MAP = {"super_admin": "1", "admin": "2", "monitor": "13"}


class CheckStatus():
    ERROR = 'EEROR'


# 优先级从高到底
RESULT_LIST = [False, RESULT_WARNING, RESULT_NOCHECK, RESULT_NOSUPPORT, True]


def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    """消息国际化

    :param lang: 语言lang
    :param msg: 消息
    :param args: 消息参数
    :param resource: 消息字典
    :return: 经过国际化处理后的消息
    """
    errMsg = "\n--"

    try:
        if not resource.has_key(msg):
            return errMsg

        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg

        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg


def isChinese(lang):
    """判断lang是否为中文

    :param lang: 语言lang
    :return: bool: True - 中文 False - 非中文
    """
    if lang == "zh":
        return True
    return False


def isNoneLicense(cliRet):
    """判断回显结果里面是否包含需要license

    :param cliRet: cli回显
    :return: True: cli回显包含需要license, False: cli回显不包含需要license
    """
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True

    return False


def hasCliExecPrivilege(cliRet):
    """判断是否具有执行cli命令的权限

    :param cliRet: cli回显
    :return:
    """
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "use the tab key to check that the command format is correct and then try again" in line.lower():
            return False
        if "does not have the permission" in line:
            return False
    return True


def queryResultWithNoRecord(cliRet):
    """判断回显是否为Command executed successfully

    :param cliRet: cli回显
    :return:
    """
    if Regex.find(CLI_EXECUTE_CMD_SUCCESS, cliRet, Regex.IGNORECASE):
        return True
    return False


def execCmdInCliMode(cli, cmd, logResult, lang):
    """获取cli模式下执行命令后的回显

    :param cli: cli对象
    :param cmd: 待执行命令
    :param logResult: 是否需要以有log的方式执行cli命令下发
    :param lang: 语言lang
    :return: flag: True: 执行命令正常, False: 执行命令不正常
             cliRet: cli回显
             errMsg: 错误消息
    """
    errMsg = ""
    cliRet = ""

    if cli is None:
        raise Exception(CLI_CAN_NOT_CONNECT_CODE, "")
    try:
        if cmd.lower() in ['y', 'n']:
            cliRet = execCliYesorNoCmd(cli, cmd, logResult)
        else:
            cliRet = execCliCmd(cli, cmd, logResult)
    except:
        raise Exception(CLI_CAN_NOT_CONNECT_CODE, "")

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    if isSupportFeature(cliRet):
        return (RESULT_NOSUPPORT, cliRet, errMsg)

    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)

    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    if isNoneLicense(cliRet):
        return (True, cliRet, errMsg)

    if isInternalError(cliRet):
        return (True, cliRet, errMsg)

    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang, cmd)
        if checkRet[0]:
            errMsg = getMsg(lang, "system.status.abnormal")
            return (False, cliRet, errMsg)

    if "TOOLKIT_SEND_CMD_TIME_OUT" in cliRet:
        cli.reConnect()
        errMsg = getMsg(lang, "cmd.execute.timeout")
        return RESULT_NOCHECK, cliRet, errMsg

    return (True, cliRet, errMsg)


def isInternalError(cliRet):
    """判断回显结果是否包含内部错误信息

    :param cliRet:
    :return: cli回显
    """
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False


def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    user_name = ""

    try:
        enterCliModeFromSomeModel(cli, lang)
        cmd = "show system general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] is not True:
            return checkRet

        cliRet = checkRet[1]
        line_list = cliRet.splitlines()

        if len(line_list) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)

        last_line = line_list[-1]
        if CLI_RET_END_FLAG in last_line:
            user_name = last_line.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            splits = re.split(r'^\[zone\d+\]', user_name)
            if len(splits) == 2:
                user_name = splits[1]
            return (True, user_name, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)

    return (False, cliRet, errMsg)


class PrivilegeType:
    USER_LEVEL = "userLevel"
    ROLE_ID = "roleId"


def getUserPrivilege(cli, lang, type=PrivilegeType.USER_LEVEL):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang)
        if checkRet[0] != True:
            return checkRet

        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return checkRet

        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)

        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, errMsg)

        for line in cliRetLinesList:
            name = line["Name"]
            if name == userName:
                if type == PrivilegeType.USER_LEVEL:
                    userPrivilege = line["Level"]
                    return (True, userPrivilege, errMsg)
                if type == PrivilegeType.ROLE_ID:
                    userPrivilege = line["Role ID"]
                    return (True, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)

    return (False, cliRet, errMsg)


def get_user_level_and_role_id(cli, lang):
    """
    获取用户级别及role_id
    :param cli: cli链接
    :param lang: 中英文
    :return:
    """
    cli_ret = ""

    try:
        flag, user_name, err_msg = getUserName(cli, lang)
        if not flag:
            return flag, user_name, "", "", err_msg

        cmd = "show user user_name=%s" % user_name
        flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)
        if not flag:
            return flag, cli_ret, "", "", err_msg
        cli_ret_lines_list = getHorizontalCliRet(cli_ret)

        if len(cli_ret_lines_list) == 0:
            err_msg = getMsg(lang, "cannot.get.user.info")
            return False, cli_ret, "", "", err_msg

        for line in cli_ret_lines_list:
            name = line.get("Name", "")
            if name == user_name:
                level = line.get("Level", "")
                role_id = line.get("Role ID", "")
                return True, cli_ret, level, role_id, err_msg
        return False, cli_ret, "", "", err_msg
    except:
        err_msg = getMsg(lang, "cannot.get.user.info")
        return False, cli_ret, "", "", err_msg


def is_super_administrator(cli, user_name, lang):
    """
    是否超级管理员角色
    :param cli:
    :param user_name:
    :param lang:
    :return: 命令是否执行成功  第二个参数：True: 是超级管理员， False: 非超级管理员
             第三个参数 如果命令执行失败，为错误信息，否则为执行的回显
    """
    cmd = "show user user_name=%s" % user_name
    flag, ret, msg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return False, False, msg
    user_dict_list = getHorizontalCliRet(ret)
    for user_item in user_dict_list:
        # 兼容老版本不存在“Role ID”场景"Super_admin"
        if user_item.get("Role ID") == '1' or user_item.get("Level") == "Super_admin":
            return True, True, ret
    return True, False, ret


def hasAdminOrSuperAdminPrivilege(
        cli, lang, privilege_type=PrivilegeType.USER_LEVEL):
    '''
    @summary: 判断用户是否具有管理员或超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @param privilege_type: 权限类型 Dorado V6 & V5R8 userLevel字段裁剪，使用RoleId 判断权限
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时
                True: 具有管理员或超级管理员权限
                False： 不具有管理员或超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    privileges = ["super_admin", "admin"]
    return has_privileges(cli, lang, privileges, privilege_type)


def hasSuperAdminPrivilege(cli, lang, privilege_type=PrivilegeType.USER_LEVEL):
    '''
    @summary: 判断用户是否具有超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @param privilege_type: 权限类型 Dorado V6 & V5R8 userLevel字段裁剪，使用RoleId 判断权限
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时
                True: 具有超级管理员权限
                False： 不具有超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    privileges = ["super_admin"]
    return has_privileges(cli, lang, privileges, privilege_type)


def hasMonitorOrAdminOrSuperAdminPrivilege(
        cli, lang, privilege_type=PrivilegeType.USER_LEVEL):
    """判断用户是否拥有监控/管理员/超级管理员权限

    :param cli:
    :param lang:
    :param privilege_type:
    :return:
    """
    privileges = ["super_admin", "admin", "monitor"]
    return has_privileges(cli, lang, privileges, privilege_type)


def has_privileges(cli, lang, privileges, privilege_type):
    """判断用户是否具有权限

    :param cli: cli对象
    :param lang: 语言lang
    :param privilege_type: 权限类型
    :param privileges: 权限列表
    :return:
    """
    if privilege_type == PrivilegeType.ROLE_ID:
        privileges = map(lambda x: ROLE_ID_MAP.get(x), privileges)

    check_ret = getUserPrivilege(cli, lang, privilege_type)
    if not check_ret[0]:
        return check_ret

    user_level = check_ret[1]
    if user_level is None or len(
            user_level) == 0 or user_level.lower() not in privileges:
        return True, False, ""
    else:
        return True, True, ""


def enterDeveloperMode(cli, lang, debugPasswd=None):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    errMsg = ""
    cliRet = ""
    cliRetAll = ""
    cmd = "change user_mode current_mode user_mode=developer"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]:
        return checkRet
    cliRet = checkRet[1]

    # 需要确认直接输入y
    cnt = 1
    while "y/n" in cliRet and cnt <= 3:
        cliRet = cli.execCmd("y")
        cnt += 1
    cliRetAll += cliRet

    if isInDeveloperMode(cliRet):
        return (True, cliRetAll, errMsg)

    if "password:" in cliRet.lower():
        errMsg = getMsg(lang, "cannot.access.developer.mode")
        for i in range(0, 3):
            checkRet = excuteCmdInCliMode(cli, debugPasswd, False, lang)
            cliRet = checkRet[1]
            cliRetAll += cliRet
            if isInDeveloperMode(cliRet):
                return (True, cliRetAll, errMsg)
            if "password is wrong" in cliRet.lower():
                errMsg = getMsg(lang, "developer.password.is.wrong")

    return (False, cliRetAll, errMsg)


def developerMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cli.execCmd("n")

    return None


def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return:
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if Regex.find(DEVELOPER_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False


def checkLineInBlackList(cliLine, lang, execlude_cmd=None):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @:param cmd 排除的命令
    @return:
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    if execlude_cmd:
        # 有些命令的回显包含Error: 之类的字段也属于正常，就不再判断是否在黑名单中了
        # 由于后面还要再继续判断，此处就不单独写白名单了
        for execlude_pattern in cliResource.EXECLUDE_CMD:
            if re.search(execlude_pattern, execlude_cmd):
                return False, errMsg
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)


def checkLineInWaiteWhiteList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在等待回显中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return:
        True: cli语句行在名单中
        False: cli语句行不在名单中
    '''
    errMsg = ""

    for dictItems in cliResource.WAIT_WHITELIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)


def isSupportFeature(cliRet):
    '''
    @summary: 判断是否支持某特性
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "use the tab key to check that the command format is correct" in lowerLine:
            return True
        if "error:" in lowerLine and "not support" in lowerLine:
            return True
        # 【注意】升级类工具专用：全是使用admin执行，所以不存在权限不足情况
        if lowerLine.strip() == "^":
            return True

    return False


def excuteCmdInCliMode4Privilege(cli, cmd, isHasLog, lang):
    '''
    @summary: 将没有权限的命令结果"未检查"转换成"不涉及"
    '''
    checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
    if not hasCliExecPrivilege(checkRet[1]):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOSUPPORT, checkRet[1], errMsg)
    return checkRet


def excuteCmdInDeveloperMode(cli, cmd, isHasLog, lang, debugPasswd=None, preExec=True):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        if preExec:
            # 兼容部分命令视图切换，先尝试在普通用户模式下执行，再尝试在研发模式下执行
            checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
            if checkRet[0] is not True:
                return checkRet

        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened is True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] is not True:
                return checkRet

        enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
        if enterDeveloperCheckRet[0] is not True:
            return enterDeveloperCheckRet

        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]

        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened is True:
            closeDeveloperSwitch(cli, lang)


def enterDebugModeFromCliMode(cli, lang):
    '''
    @summary: 从cli模式进入debug模式
    '''
    allCliRet = ""

    # 进入devoper模式
    checkRet = enterDeveloperMode(cli, lang)
    allCliRet += checkRet[1]
    if checkRet[0] != True:
        return checkRet

    cmd = "debug"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    allCliRet += "\n" + checkRet[1]
    if checkRet[0] != True:
        return (checkRet[0], allCliRet, checkRet[2])

    if isInDebugMode(checkRet[1]):
        return (True, allCliRet, "")
    else:
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (False, allCliRet, errMsg)


def enterCliModeFromDebugModel(cli, lang):
    '''
    @summary: 从debug模式进入cli模式
    '''
    allCliRet = ""
    cmd = "exit"
    # 切换到developer
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    allCliRet += checkRet[1]
    if checkRet[0] != True:
        return checkRet
    # 切换到CLI
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    return checkRet


def excuteCmdInDebugModel(cli, cmd, isHasLog, lang):
    """
    @summary: 在debug模式下执行命令
    """
    cliRet = ''
    try:
        cliRet = execCliCmd(cli, cmd, isHasLog)

        if isInDebugMode(cliRet):
            return (True, cliRet, "")

        if isInMinisystemMode(cliRet):
            cliRet = cli.execCmd("exit")
            if "(y/n)" in cliRet:
                cliRet = cli.execCmd("y")

        if isInDeveloperMode(cliRet):
            cliRet = cli.execCmd("debug")
            if isInDebugMode(cliRet):
                cliRet = execCliCmd(cli, cmd, isHasLog)
                return (True, cliRet, "")

            return (False, cliRet, "")

        flag, cliRet, errMsg = enterDebugModeFromCliMode(cli, lang)
        if flag != True:
            return flag, cliRet, errMsg

        cliRet = execCliCmd(cli, cmd, isHasLog)
        enterCliModeFromDebugModel(cli, lang)
        return (True, cliRet, "")
    except:
        return (False, cliRet, "")


def excuteCmdInDebugModelAndAttach(cli, cmd, isHasLog, lang):
    """
    @summary: 在debug模式 -> 先执行 attach 21 后执行命令
    """
    cli_ret = ""
    err_msg = ""
    try:
        cli_ret = execCliCmd(cli, cmd, isHasLog)

        if isInMinisystemMode(cli_ret):
            cli_ret = cli.execCmd("exit")
            if "(y/n)" in cli_ret:
                cli_ret = cli.execCmd("y")

        if isInDeveloperMode(cli_ret):
            cli_ret = cli.execCmd("debug")

        if not isInDebugMode(cli_ret):
            flag, cli_ret, err_msg = enterDebugModeFromCliMode(cli, lang)
            if flag is not True:
                return flag, cli_ret, err_msg

        # 高级版本支持免attach命令 直接在命令后加上进程名称
        attach_cmd = cmd + " -devm"
        cli_ret = execCliCmd(cli, attach_cmd, isHasLog)
        if Regex.search("not found", cli_ret, Regex.IGNORECASE):
            # 不支持免attach命令时 attach进程号
            cli_ret = execCliCmd(cli, "attach 21", isHasLog)
            if not Regex.search("success", cli_ret, Regex.IGNORECASE):
                enterCliModeFromDebugModel(cli, lang)
                return False, cli_ret, ""
            cli_ret = execCliCmd(cli, cmd, isHasLog)
        enterCliModeFromDebugModel(cli, lang)
        return True, cli_ret, ""
    except:
        return False, cli_ret, ""


def getProductModelVersion(cli, lang):
    '''
    @summary: 获取产品类型和产品版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，产品类型 产品版本
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show system general"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]:
        return checkRet

    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = restUtil.CommonRest.getOriginProductModel(cli.getdNode())
    product_version = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if not product_model and fieldName == "Product Model":
            product_model = fieldValue
        elif fieldName == "Product Version":
            product_version = fieldValue

        if len(product_model) != 0 and len(product_version) != 0 and product_model != "--" and product_version != "--":
            return (True, " ".join((product_model, product_version)), "")

    return (False, cliRet, getMsg(lang, "cannot.get.product.version.info"))


def getProductVersion(cli, lang):
    '''
    @summary: 获取产品版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show system general"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]:
        return checkRet

    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Product Version":
            product_model = fieldValue

        if len(product_model) != 0 and product_model != "--":
            return (True, product_model, "")

    return (False, cliRet, getMsg(lang, "cannot.get.product.version.info"))


def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    product_model = restUtil.CommonRest.getOriginProductModel(cli.getdNode())
    if product_model:
        return True, product_model, ""
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Product Model":
            product_model = fieldValue

        if len(product_model) != 0 and product_model != "--":
            return (True, product_model, "")

    return (False, cliRet, getMsg(lang, "cannot.get.product.model.info"))


def get_product_model_with_ret(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, product_model, err_msg, cli_ret)
        flag:
            True: 获取成功
            False: 获取失败
        product_model:
            产品类型
        err_msg: 错误消息
        cli_ret: 回显
    '''
    cmd = "show system general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return flag, "", err_msg, cli_ret

    cli_ret_list = cli_ret.splitlines()
    product_model = restUtil.CommonRest.getOriginProductModel(cli.getdNode())
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if not product_model and field_name == "Product Model":
            product_model = field_value

        if len(product_model) != 0 and product_model != "--":
            return True, product_model, "", cli_ret

    return False, "", getMsg(lang, "cannot.get.product.model.info"), cli_ret


def getStorageType(cli, lang):
    '''
    @summary: 获取存储设备类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，存储设备类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = getProductModel(cli, lang)
    if getProductModelRet[0] != True:
        return getProductModelRet

    pdtModel = getProductModelRet[1]

    if not STOR_DEV_INFO_DICT.has_key(pdtModel):
        errMsg = getMsg(lang, "cannot.get.storage.type.info")
        return (False, "", errMsg)

    storageType = STOR_DEV_INFO_DICT.get(pdtModel)
    return (True, storageType, "")


def get_horizontal_cli_ret_without_dot(cli_ret):
    """ 按逐行字典的方式获取水平表格形式(无虚线分隔的回显)的cli回显集合

    :param cli_ret:
    :return:
    """
    try:
        dict_list = []
        if queryResultWithNoRecord(cli_ret):
            return dict_list

        ret_list = cli_ret.encode("utf8").splitlines()
        if len(ret_list) < 4:
            return dict_list

        # 第一行为命令，第二行为表头
        keys = ret_list[1].split()
        # 第三行开始为每一项的值
        for line in range(2, len(ret_list)):
            # 最后一行为结束符，不处理
            if line == (len(ret_list) - 1):
                break
            values = ret_list[line].split()
            dict_list.append(dict(zip(keys, values)))
        return dict_list
    except Exception:
        return []


def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1

        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}

    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if Regex.find("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(":")
        if len(fields) < 2:
            continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        dictList.append(lineDict.copy())

    return dictList


def getHorizontalDebugRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        dictList = []
        for line in field_words:
            if DIAGNOSE_RET_END_FLAG in line:
                break

            if len(line.strip()) == 0:
                continue

            vals = line.split()
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def get_system_partition(cli_ret):
    '''
    Storage: minisystem> du -m /startup_disk/image/boot_osp
    1       /startup_disk/image/boot_osp/conf
    1       /startup_disk/image/boot_osp/coldpatch
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cli_ret.encode("utf8").splitlines()
    lineDict = {}

    for line in cliRetList:
        if MINISYSTEM_MODEL_FLAG in line:
            break

        fields = line.split()
        if len(fields) < 2:
            continue

        key = fields[1].strip().decode("utf8")
        value = fields[0].strip().decode("utf8")
        lineDict[key] = value
    return lineDict


def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    splitedCliRet = cliRetlist[1]

    match = Regex.search("^.*:(/>)*\r*\n*$", splitedCliRet, Regex.MULTILINE)
    if match != None:
        splitedCliRet = splitedCliRet[0:match.start()]

    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet


def isEngine0(value):
    '''
    @summary: 根据id判断是否为引擎0
    @param value: 回显中的ID
    @return:
        True: 是引擎0
        False: 不是引擎0
    '''
    if "ENG0" in value or "CTE0" in value:
        return True
    else:
        return False


def isEngineId(value):
    '''
    @summary: 根据id判断是否为引擎ID
    @param value: 回显中的ID
    @return:
        True: 是引擎ID
        False: 不是引擎ID
    '''
    if "ENG" in value or "CTE" in value:
        return True
    else:
        return False


def isDiskId(value):
    '''
    @summary: 根据id判断是否为硬盘ID
    @param value: 回显中的ID
    @return:
        True: 是硬盘ID
        False: 不是硬盘ID
    '''
    if "DAE" in value:
        return True
    else:
        return False


def isDataSwitchId(value):
    '''
    @summary: 根据id判断是否为交换机ID
    @param value: 回显中的ID
    @return:
        True: 是交换机ID
        False: 不是交换机ID
    '''
    if "DSW" in value:
        return True
    else:
        return False


def isHighDensityDisk(enclosureId, cli, lang):
    '''
    @summary: 判断硬盘框是否为高密框
    @param enclosureId: 框ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg)
        flag:
            True： 判断时正常
            False： 判断时异常
        ret:
            flag为True时，是高密框
            flag为False时，不是高密框
        errMsg: 错误时的消息
    '''
    if not isDiskId(enclosureId):
        return (True, False, "")

    checkRet = getHighDensityDiskEnclosureIdList(cli, lang)
    if not checkRet[0]:
        return checkRet

    highDensityDiskEnclosureIdList = checkRet[1]
    if enclosureId in highDensityDiskEnclosureIdList:
        return (True, True, "")
    else:
        return (True, False, "")


def getEnclosureHeight(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框高度(U)
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，框ID对应的高度(U)
            flag为False时，cli回显
        errMsg: 错误消息
    '''

    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Height(U)" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Height(U)":
            enclosureHeight = fieldValue
            return (True, enclosureHeight, "")

    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)


def get_kernel_version(cli, lang):
    """获取内核固件版本

    :param cli: cli连接
    :param lang: 语言
    :return:
    """
    cmd = "upgrade.sh kernel showversion"
    cli_ret = excuteCmdInMinisystemModel(cli, cmd, lang)
    if cli_ret[0] is not True:
        return ""
    kernel_info_list = getVerticalCliRet(cli_ret[1])
    ret = enterCliModeFromSomeModel(cli, lang)
    if not ret[0]:
        return ""
    if kernel_info_list:
        return kernel_info_list[0].get("kernel version", "")
    return ""


def getEnclosureType(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框类型
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，框ID对应的类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''

    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Type" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Type":
            enclosureType = fieldValue
            return (True, enclosureType, "")

    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)


def getCtrlInfoById(cli, ctrlId, lang):
    '''
    @summary: 获取指定控制器的信息
    '''
    cmd = "show controller general controller=%s" % ctrlId
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]:
        return (checkRet[0], {}, checkRet[1])

    ctrlList = getVerticalCliRet(checkRet[1])
    if len(ctrlList) > 0:
        return (True, ctrlList[0], checkRet[1])

    return (False, {}, checkRet[1])


def getControllerIdList(cli, lang):
    '''
    @summary: 获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，设备所有控制器ID列表
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show controller general|filterColumn include columnList=Controller"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    cliRetLinesList = cliRet.splitlines()
    controlList = []

    for line in cliRetLinesList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Controller":
            ctrlId = fieldValue
            controlList.append(ctrlId)

    if len(controlList) == 0:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return (False, cliRet, errMsg)

    return (True, controlList, "")


def get_controller_id_list_with_ret(cli, lang):
    """
    获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return:
    """
    cmd = "show controller general|filterColumn include columnList=Controller"
    check_ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if check_ret[0] is not True:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_lines_list = cli_ret.splitlines()
    control_list = []

    for line in cli_ret_lines_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Controller":
            ctrl_id = field_value
            control_list.append(ctrl_id)

    if len(control_list) == 0:
        err_msg = getMsg(lang, "cannot.get.controller.info")
        return False, control_list, err_msg, cli_ret

    return True, control_list, "", cli_ret


def get_controller_ip_list(cli, lang):
    """获取所有控制器的ip列表

    :param cli:
    :param lang:
    :return:
    """
    controller_ip_list = []
    cmd = "show upgrade package"
    flag, cli_ret, error_msg = execCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return False, controller_ip_list

    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, ""
    software_list = getHorizontalNostandardCliRet(
        cli_ret[software_begin_index:hot_patch_begin_index])
    for software in software_list:
        controller_ip = software.get("IP")
        controller_ip_list.append(controller_ip)

    return True, controller_ip_list


def get_ctrl_ip_name_dict(cli, lang):
    """
    获取所有控制器的ip和控制器名称的对应关系
    :param cli:
    :param lang:
    :return:
    """
    ctrl_ip_name_dict = {}
    cmd = "show upgrade package"
    flag, cli_ret, error_msg = execCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return False, cli_ret, error_msg, ctrl_ip_name_dict

    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, cli_ret, error_msg, ctrl_ip_name_dict
    software_list = getHorizontalNostandardCliRet(
        cli_ret[software_begin_index:hot_patch_begin_index])
    for software in software_list:
        controller_ip = software.get("IP")
        controller_name = software.get("Name")
        if controller_ip != '--' and controller_ip != "":
            ctrl_ip_name_dict[controller_ip] = controller_name

    return True, cli_ret, error_msg, ctrl_ip_name_dict


def get_cli_for_svp(context, log):
    """获取cli连接 屏蔽高端设备的svp差异

    :param context:
    :param log:
    :return:
    """
    has_svp = baseUtil.has_svp_module(context)
    if has_svp:
        # 有svp 先获取控制器的ip 然后创建连接到该ip的连接
        cli = contextUtil.getCli(context)
        lang = contextUtil.getLang(context)
        flag, controller_ip_list = get_controller_ip_list(cli, lang)
        if flag:
            # 返回第一个成功连接的控制器
            for controller_ip in controller_ip_list:
                cli = contextUtil.createCliConnection(context, controller_ip)
                if cli:
                    log.logInfo("init cli connection successfully")
                    break
        else:
            log.logInfo("can't get controller ip")

    return contextUtil.getCli(context)


def checkCliConnectionIsNormal(ssh):
    '''
    升级完成后，cli连接有可能处于upgrade: /> 模式，需要重新建立cli连接
    '''
    try:
        cmd = "show system general"
        flag, cliRet, errorMsg = excuteCmdInCliMode(ssh, cmd, True, "zh")
        if "admin:/>" not in cliRet:
            # 由于非admin用户的场景很少，为了简单即使命令正常返回也重连一次
            ssh.close()
            ssh.reConnect()
    except:
        return False
    return True


def getAlarmIdList(ssh, lang):
    alarmIdList = []
    cmd = "show alarm |filterColumn include columnList=Name,ID"
    (ok, cliRet, errMsg) = excuteCmdInCliMode(ssh, cmd, True, lang)
    if not ok:
        return (False, alarmIdList, "")

    cliRet2List = getHorizontalCliRet(cliRet)
    for item in cliRet2List:
        alarmIdList.append(str(item.get("ID")))
    return (True, alarmIdList, cliRet)


def isExistAlarm(ssh, lang, alarmId):
    cmd = "show alarm |filterRow column=ID predict=equal_to value=%s" % alarmId
    (ok, cliRet, errMsg) = excuteCmdInCliMode(ssh, cmd, True, lang)
    if not ok:
        return ("Error", "")
    if queryResultWithNoRecord(cliRet):
        return ("NotExist", cliRet)

    cliRet2List = getHorizontalCliRet(cliRet)
    if len(cliRet2List) == 0:
        return ("Error", cliRet)
    return ("Exist", cliRet)


def getEng0Id(cli, lang):
    '''
    @summary: 获取引擎0ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret:
            flag为True时，引擎0ID
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    cliRetLinesList = getHorizontalCliRet(cliRet)

    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        enclosureId = line["ID"]
        if isEngine0(enclosureId):
            return (True, enclosureId, "")

    return (True, "", "")


def getHighDensityDiskEnclosureIdList(cli, lang):
    '''
    @summary: 获取高密硬盘框ID集合
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True： 判断时正常
            False： 判断时异常
        ret:
            flag为True时，高密硬盘框ID集合
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    highDensityDiskEnclosureIdList = []
    cliRetLinesList = getHorizontalCliRet(cliRet)

    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        enclosureType = line["Type"]
        enclosureId = line["ID"]
        if "4U 75" in enclosureType:
            highDensityDiskEnclosureIdList.append(enclosureId)

    return (True, highDensityDiskEnclosureIdList, "")


def rollbackExpansion(cli, lang, logger, orinCtrlsNum):
    '''
    @summary: 执行扩容回退
    '''
    isOpened = False
    try:
        # 进入developer模式
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet

        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet
        # 下发扩容命令
        cmd = "change controllers_expansion cancel controller_total=%s" % orinCtrlsNum
        checkRet = execCmdInCliMode(cli, cmd, True, lang)

        cliRet = checkRet[1]
        tempRet = cliRet

        cnt = 0
        while ("y/n" in tempRet and cnt < 3):
            cmd = "y"
            checkRet = execCmdInCliMode(cli, cmd, True, lang)
            tempRet = checkRet[1]
            cliRet += tempRet
            if checkRet[0] != True:
                break
            cnt += 1

        developerMode2CliMode(cli)

        if CLI_EXECUTE_CMD_SUCCESS in cliRet:
            return (True, cliRet, "")
        else:
            return (False, cliRet, "")
    finally:
        # 关闭开关
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)


def expandCtrlCapacityFor2800(cli, lang):
    '''
    @summary: 执行扩容
    @param cli: cli对象
    @param lang: 语言lang
    '''
    isOpened = False
    try:
        # 进入developer模式
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet

        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet
        # 下发扩容命令
        cmd = "change system config_model"
        checkRet = execCmdInCliMode(cli, cmd, True, lang)

        cliRet = checkRet[1]
        tempRet = cliRet

        cnt = 0
        while ("y/n" in tempRet and cnt < 3):
            cmd = "y"
            checkRet = execCmdInCliMode(cli, cmd, True, lang)
            tempRet = checkRet[1]
            cliRet += tempRet
            if checkRet[0] != True:
                break
            cnt += 1

        developerMode2CliMode(cli)

        if CLI_EXECUTE_CMD_SUCCESS in cliRet:
            return (True, cliRet, "")
        elif "in multi-controller model" in cliRet.lower():
            return (True, cliRet, "")
        else:
            return (False, cliRet, "")
    finally:
        # 关闭开关
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)


def execCliCmd(cli, cmd, logResult):
    '''
    @summary: 获取cli回显，回显异常（回显中包含Error）时，进行重试
    @param cli: cli对象
    @param cmd: cli命令
    @param logResult: cli命令日志打印标志
    '''
    cliRet = ""
    retryNum = 3  # 回显异常（回显中包含Error）时，重试次数
    retryIntervalTime = 30  # 每次重试间隔时间（秒）

    for index in range(retryNum):
        if logResult:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)

        # 正常情况无需重试
        if "Error:" not in cliRet:
            break

        baseUtil.safeSleep(retryIntervalTime)

    return cliRet


def exec_cyber_engine_cmd(cli, cmd, log_result):
    """
    cyberEngine 连接有效时长为5分钟，在执行试试命令之前需要检查是否连接。如果连接断开重新建立连接后执行命令。

    :param cli: cli对象
    :param cmd: cli命令
    :param log_result: cli命令日志打印标志
    :return: 回显信息。
    """
    end_chars = ["]$", "]#$", "Password:", "]#", "]# "]
    if not cli.isConnected():
        cli.reConnect()
    if log_result:
        cli_ret = cli.execCmd(cmd, end_chars)
    else:
        cli_ret = cli.execCmdNoLog(cmd, end_chars)
    return cli_ret


def execCliYesorNoCmd(cli, cmd, logResult):
    cliRet = ""
    if logResult:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
    return cliRet


# ------------------------------------------------------------------------------
# debug mode factory.           time:2015/8/25 10:26
def isInDebugMode(cliRet):
    """
    @summary: check the ":/diagnose>" is in the return strings.
    """
    if Regex.find(DEBUG_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False


def enterDebugMode(cli, lang, debugPasswd=None):
    """
    @summary: first enter the developer mode,then the debug mode.
    """
    enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
    if not enterDeveloperCheckRet[0]:
        developerMode2CliMode(cli)
        return enterDeveloperCheckRet

    cmd = "debug"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return checkRet

    cliRet = checkRet[1]
    if isInDebugMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)


def enterMinisystemFromDebugMode(cli, lang):
    """
    @summary: firstly enter the developer mode,then enter minisystem mode.
    """

    all_cli_ret = ""
    cli_ret = cli.execCmdNoLog("test")
    if DEBUG_MODEL_FLAG not in cli_ret:
        return False, all_cli_ret

    cli_ret = cli.execCmd("exit")
    all_cli_ret += cli_ret
    if not isInDeveloperMode(cli_ret):
        return False, all_cli_ret

    cli_ret = cli.execCmd("minisystem")
    all_cli_ret += cli_ret
    if not isInMinisystemMode(cli_ret):
        return False, all_cli_ret
    return True, all_cli_ret


def executeCmdInDebugMode(cli, cmd, isHasLog, lang, debugPasswd=None):
    """@summary: execute cmd in debug mode
    """
    from cbb.frame.cli import cli_helper
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet

        enterDebugCheckRet = cli_helper.enter_target_mode(cli, cli_helper.CliMode.DIAGNOSE, lang, debugPasswd)
        if enterDebugCheckRet[0] != True:
            return enterDebugCheckRet
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)

        cliRet = checkRet[1]
        if isInDebugMode(cliRet):
            exitDebugModeToCliMode(cli)

        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)


def exitDebugModeToCliMode(cli):
    """
    @summary: first exit the debug mode,then exit the developer mode.
    """
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cliRet = cli.execCmd("n")
    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)


def exitMinisystemToCliMode(cli):
    """
    @summary: first exit the debug mode,then exit the developer mode.
    """
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cliRet = cli.execCmd("n")

    if isInMinisystemMode(cliRet):
        cliRet = cli.execCmd("exit")
        if "y/n" in cliRet:
            cliRet = cli.execCmd("y")

    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)


def GetCliRecsInDebugMode(cli, cmdList, logger, lang, debugPasswd=None):
    """
    @summary: Execute the cli command in the debug mode.
    @param cmdList:one or more cmd.
    @return: echo list.
    """
    cliRecsList = list()
    for cmd in cmdList:
        try:
            logger.logInfo("Start to execute the cli cmd:%s" % cmd)
            checkRet = executeCmdInDebugMode(cli, cmd, False, lang, debugPasswd)
            cmdRecs = checkRet[1]
        except Exception as msg:
            logger.logInfo("The Exception is:%s" % msg)
            cmdRecs = ""
        logger.logInfo("The cliRecs is %s" % unicode(cmdRecs))
        cliRecsList.append(cmdRecs)
    return cliRecsList


def getCtrlNames(cli, logger, lang):
    """
    @summary: 获取系统所有控制器名称
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag,ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlId,ctrlIp)])
    """
    cliRet = ""
    ctrlNames = []
    try:
        cmd = "show controller general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (False, [])

        cliRet = checkRet[1]
        ctrlNameList = getVerticalCliRet(cliRet)
        if len(ctrlNameList) == 0:
            return (False, [])

        for line in ctrlNameList:
            ctrlName = line["Controller"]
            ctrlNames.append(ctrlName)

        return (True, ctrlNames)
    except Exception as exception:
        logger.logException(exception)
        return (False, [])


def getEngineIdList(cli, lang):
    """
    @summary: 获取引擎ID
    """
    engineIdList = []
    cmd = "show enclosure|filterRow column=Logic\sType predict=equal_to value=Engine"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (flag, cliRet, errMsg, engineIdList)

    infoDictList = getHorizontalCliRet(cliRet)
    if not infoDictList:
        errMsg = getMsg(lang, "query.result.abnormal")
        return (False, cliRet, errMsg, engineIdList)

    for infoDict in infoDictList:
        engineId = infoDict.get("ID")
        engineIdList.append(engineId)

    return (True, cliRet, errMsg, engineIdList)


def getCtrlIps(cli, logger, lang):
    """
    @summary: 获取控制器的ip（中低端为管理ip，高端为内部ip）
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag,ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlId,ctrlIp)])
    """
    flag = True
    ctrlIps = []
    checkedCtrls = []
    try:
        cmdIpv4 = "show upgrade package"
        checkIpv4Ret = excuteCmdInCliMode(cli, cmdIpv4, True, lang)
        if checkIpv4Ret[0] != True:
            return (False, [])

        ipv4CliRet = checkIpv4Ret[1]
        ipv4cliRetLinesList = getHorizontalCliRet(ipv4CliRet)
        if len(ipv4cliRetLinesList) == 0:
            return (False, [])

        ctrlsIpv6Ret = getIpv4andIpv6Map(cli, logger, lang)
        ctrlsIpv6 = ctrlsIpv6Ret[1]

        for line in ipv4cliRetLinesList:
            ctrlName = line["Name"]
            ipv4 = line["IP"]

            if not Regex.find("^[0-9][A-Z]", ctrlName):
                continue

            # 如果已经获取了的控制器，无需再获取
            if ctrlName in checkedCtrls:
                continue

            checkedCtrls.append(ctrlName)
            ipv6 = ctrlsIpv6.get(ipv4, "")
            ctrlIps.append((ctrlName, ipv4, ipv6))

            # 按照控制器ID（如：0A）从小到大排序
        ctrlIps = sorted(ctrlIps, key=lambda x: x[0])
        return (flag, ctrlIps)

    except Exception as exception:
        logger.logException(exception)
        return (False, [])


def getIpv4andIpv6Map(cli, logger, lang):
    """
    @summary: 获取控制器的ipv4与ipv6的映射字典
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag,ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlIpv4,ctrlIpv6)])
    """
    flag = True
    ctrlsIpv6 = {}
    try:
        cmd = "show port general logic_type=Management_Port"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (False, {})

        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        if len(cliRetLinesList) == 0:
            return (False, {})

        for line in cliRetLinesList:
            ipv4 = line["IPv4 Address"]
            ipv6 = line["IPv6 Address"]
            ctrlsIpv6[ipv4] = ipv6
        return (flag, ctrlsIpv6)

    except Exception as exception:
        logger.logException(exception)
        return (False, {})


@wrap_all_exception_logged(logger=None)
def closeCliConnection(cli):
    """
    @summary: 关闭cli连接
    @param cli:cli连接
    """
    cli.close()


def execCmdInDeveloperModePrompt(cli, cmd, isHasLog, lang, prompt="y"):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param prompt: 确认命令(y/n)
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet

        cliRetAll = ""
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet

        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        cliRetAll += cliRet

        cnt = 0
        while ("y/n" in cliRet and cnt < 5):
            ret = excuteCmdInCliMode(cli, prompt, isHasLog, lang)
            cliRet = ret[1]
            cliRetAll += cliRet
            cnt += 1

        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)

        return (checkRet[0], cliRetAll, checkRet[2])
    finally:
        # 关闭开关
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)


def changeTlvChannel(cli, lang, switch):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @param switch: 开关状态
    @return: (falg, ret)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
    '''

    cliRet = ""
    try:

        cmd = "change system external_TLV_channel enabled=%s" % switch
        checkRet = execCmdInDeveloperModePrompt(cli, cmd, True, lang)
        cliRet = checkRet[1]
        if not hasCliExecPrivilege(cliRet):
            return (True, cliRet)

        if checkRet[0] != True:
            return (False, cliRet)

        if CLI_EXECUTE_CMD_SUCCESS in cliRet:
            return (True, cliRet)
        else:
            return (False, cliRet)
    except:
        return (False, cliRet)


def openTlvChannel(cli, lang="zh"):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @param switch: 开关状态
    @return: (falg, ret, errMsg, suggestion)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
        errMsg: 错误消息
        suggestion: 修复建议
    '''

    ret = hasSuperAdminPrivilege(cli, lang)
    if not ret[1]:
        # 当用户没有超级管理员权限时，打开tlv端口肯定失败
        raise Exception("DO_NOT_HAS_SUPER_ADMIN_RIGHTS", "")
    flag, cliRet = changeTlvChannel(cli, lang, "yes")

    if flag:
        return (True, cliRet, "", "")
    else:
        errMsg = ""
        suggestion = ""
        if lang == "zh":
            errMsg = u"开启TLV通道失败。"
            suggestion = u"请联系技术支持工程师协助处理。"
        else:
            errMsg = "Change the status of the external TLV channel failed."
            suggestion = "Contact technical support engineers for help."
        return (False, cliRet, errMsg, suggestion)


def closeTlvChannel(cli, lang):
    '''
    @summary: 打开或关闭tlv端口，关闭失败不受影响。
    @param cli: cli对象
    @return: (falg, cliRet)
        flag:
            True:  执行成功
            False: 执行失败
        cliRet: cli回显
    '''
    flag, cliRet = changeTlvChannel(cli, lang, "no")
    return (True, cliRet)


def getSystemVersion(cli, lang):
    cmd = "show upgrade package"
    (ok, cliRet, errorMsg) = execCmdInCliMode(cli, cmd, True, lang)
    if not ok:
        return (False, "", "")

    hotPatchBeginIndex = cliRet.find("HotPatch Version")
    softwareBeginIndex = cliRet.find("Software Version")
    if softwareBeginIndex == -1 or hotPatchBeginIndex == -1:
        return (False, "", "")
    softwareDict = getHorizontalCliRet(cliRet[softwareBeginIndex:hotPatchBeginIndex])
    softwareVersion = softwareDict[0].get('Current Version')

    hotPatchDict = getHorizontalCliRet(cliRet[hotPatchBeginIndex:])
    validHotpatchVers = list(filter(lambda hdict: hdict.get('Current Version') != '--'
                                                  and hdict.get('Current Version'), hotPatchDict))
    if list(validHotpatchVers):
        hotPatchVersion = validHotpatchVers[0].get('Current Version')
    else:
        hotPatchVersion = ''

    return True, softwareVersion, hotPatchVersion


def get_system_version_with_ret(cli, lang):
    cmd = "show upgrade package"
    (ok, cli_ret, error_msg) = execCmdInCliMode(cli, cmd, True, lang)
    if not ok:
        return False, "", "", cli_ret

    hot_patch_begin_index = cli_ret.find("HotPatch Version")
    software_begin_index = cli_ret.find("Software Version")
    if software_begin_index == -1 or hot_patch_begin_index == -1:
        return False, "", "", cli_ret
    software_dict = getHorizontalCliRet(
        cli_ret[software_begin_index:hot_patch_begin_index])
    software_version = software_dict[0].get('Current Version')

    hot_patch_dict = getHorizontalCliRet(cli_ret[hot_patch_begin_index:])
    valid_hotpatch_vers = list(filter(
        lambda hdict: hdict.get('Current Version') != '--' and
                      hdict.get('Current Version'), hot_patch_dict))
    if list(valid_hotpatch_vers):
        hot_patch_version = valid_hotpatch_vers[0].get('Current Version')
    else:
        hot_patch_version = ''

    return True, software_version, hot_patch_version, cli_ret


def exist_sas_interface_card(cli, lang):
    """
    检查设备是否存在sas接口卡
    :param cli: cli
    :param lang: lang
    :return: （命令执行成功，有SAS卡，回显）
    """
    cmd = "show interface_module"
    flag, cli_ret, error_msg = execCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        return False, False, cli_ret
    interface_modules = getHorizontalCliRet(cli_ret)
    sas_module = [item for item in interface_modules if "SAS" in item.get("Model", "")]
    if sas_module:
        return True, True, cli_ret
    return True, False, cli_ret


def excuteCmdInCliMode(cli, cmd, isHasLog, lang, cmdEndingSignList=None, timeOutSecs=20 * 60):
    """执行CLI命令的接口

    :param cli: cli对象
    :param cmd: 待执行命令
    :param isHasLog: 是否需要以有log的方式执行cli命令下发
    :param lang: 语言lang
    :param cmdEndingSignList: 命令结束符列表
    :param timeOutSecs: 超时时间
    :return: (falg, cliRet, errMsg)
             flag:
                    True: 执行命令正常
                    False: 执行命令不正常
                cliRet: cli回显
                errMsg: 错误消息
    """
    errMsg = ""

    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return RESULT_NOCHECK, "", errMsg

    try:
        if cmdEndingSignList and isHasLog:
            cliRet = cli.execCmd(cmd, cmdEndingSignList, timeOutSecs)
        elif cmdEndingSignList and not isHasLog:
            cliRet = cli.execCmdNoLogTimout(cmd, timeOutSecs, cmdEndingSignList)
        elif not cmdEndingSignList and isHasLog:
            cliRet = cli.execCmdWithTimout(cmd, timeOutSecs)
        else:
            cliRet = cli.execCmdNoLogTimout(cmd, timeOutSecs)
    except:
        _cmd = cmd if isHasLog else "***"
        exceptionArg = (
            'execute cmd exception with params, [cmd:{cmd}, '
            'isHasLog:{isHasLog}, cmdEndingSignList:{cmdEndingSignList},  '
            'timoutSecs: {timeOutSecs}, traceback: {traceback}]'
            .format(cmd=_cmd, isHasLog=isHasLog,
                    cmdEndingSignList=cmdEndingSignList,
                    timeOutSecs=timeOutSecs,
                    traceback=str(traceback.format_exc())))
        raise Exception(exceptionArg)

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    if isSupportFeature(cliRet):
        return (RESULT_NOSUPPORT, cliRet, errMsg)

    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)

    if isNoneLicense(cliRet):
        errMsg = getMsg(lang, "has.not.license")
        return (RESULT_NOSUPPORT, cliRet, errMsg)

    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    if isInternalError(cliRet):
        return (False, cliRet, errMsg)

    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang, cmd)
        if checkRet[0]:
            errMsg = getMsg(lang, "system.status.abnormal")
            return (False, cliRet, errMsg)

    if "TOOLKIT_SEND_CMD_TIME_OUT" in cliRet:
        cli.reConnect()
        errMsg = getMsg(lang, "cmd.execute.timeout")
        return RESULT_NOCHECK, cliRet, errMsg

    return (True, cliRet, errMsg)


def execute_cmd_when_power_on_fail_mode(
        cli, cmd, isHasLog, lang, cmdEndingSignList=None, timeOutSecs=20 * 60):
    """
    开工失败场景，执行cli命令
    :param cli:
    :param cmd:
    :param isHasLog:
    :param lang:
    :param cmdEndingSignList:
    :param timeOutSecs:
    :return:
    """
    errMsg = ""

    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return RESULT_NOCHECK, "", errMsg

    try:
        if cmdEndingSignList and isHasLog:
            cliRet = cli.execCmd(cmd, cmdEndingSignList, timeOutSecs)
        elif cmdEndingSignList and not isHasLog:
            cliRet = cli.execCmdNoLogTimout(
                cmd, timeOutSecs, cmdEndingSignList)
        elif not cmdEndingSignList and isHasLog:
            cliRet = cli.execCmdWithTimout(cmd, timeOutSecs)
        else:
            cliRet = cli.execCmdNoLogTimout(cmd, timeOutSecs)
    except (Exception, JException):
        _cmd = cmd if isHasLog else "***"
        exceptionArg = (
            'execute cmd exception with params, [cmd:{cmd}, '
            'isHasLog:{isHasLog}, cmdEndingSignList:{cmdEndingSignList},  '
            'timoutSecs: {timeOutSecs}]'
            .format(cmd=_cmd, isHasLog=isHasLog,
                    cmdEndingSignList=cmdEndingSignList,
                    timeOutSecs=timeOutSecs))
        raise Exception(exceptionArg)

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    if isSupportFeature(cliRet):
        return (RESULT_NOSUPPORT, cliRet, errMsg)

    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)

    if isNoneLicense(cliRet):
        errMsg = getMsg(lang, "has.not.license")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    return (True, cliRet, errMsg)


def openDeveloperSwitch(cli, lang):
    '''
            打开切换developer模式开关
    '''
    opencmd = "change user_mode enabled user_mode=developer enabled=yes"
    return excuteCmdInCliMode(cli, opencmd, True, lang)


def open_developer_switch_v(cli, lang):
    '''
            打开切换developer模式开关
    '''
    switch = True
    # 兼容V3R6版本，查看切developer模式开关是否打开
    developercmd = "show user_mode enabled"
    checkRet = excuteCmdInCliMode(cli, developercmd, True, lang)
    # 执行异常等均默认不需要开启模式开关
    if checkRet[0] != True:
        return (True, checkRet[1], checkRet[2]), switch

    cliRetDictList = getVerticalCliRet(checkRet[1])
    developerswitch = ""
    for dict in cliRetDictList:
        if "Developer View" in dict:
            developerswitch = dict.get("Developer View", "")
            break

    if developerswitch == "Disabled":
        switch = False
        # 开关关闭则打开
        opencmd = "change user_mode enabled user_mode=developer enabled=yes"
        checkRet = excuteCmdInCliMode(cli, opencmd, True, lang)
        if checkRet[0] != True:
            return checkRet, switch

    return checkRet, switch


def needOpenDeveloperSwitch(cli, lang):
    '''
    @summary: #兼容V3R6版本，查看切developer模式开关是否打开
    '''
    developercmd = "show user_mode enabled"
    checkRet = excuteCmdInCliMode(cli, developercmd, True, lang)
    if checkRet[0] is not True:
        return (CheckStatus.ERROR, checkRet)

    cliRetLinesList = getVerticalCliRet(checkRet[1])
    for line in cliRetLinesList:
        if "Disabled" == line.get("Developer View", ""):
            return (True, checkRet)

    return (False, checkRet)


@wrap_all_exception_logged(logger=None)
def closeDeveloperSwitch(cli, lang):
    '''
    @summary: 关闭developer视图开关
    '''
    closecmd = "change user_mode enabled user_mode=developer enabled=no"
    excuteCmdInCliMode(cli, closecmd, True, lang)


def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,此方法用来解析CLI回显未对其情况
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1

        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        lenkeys = len(keys)
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0:
                break

            if Regex.find("^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            vals = []
            valueList = line.strip().split("  ")

            valueSpaceList = []
            for value in valueList:
                if value != "":
                    vals.append(value.strip().decode("utf8"))
                    valueSpaceList.append(value)
            lenvalue = len(valueSpaceList)
            if lenvalue == lenkeys:
                dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []


def isInStorageMode(cliRet):
    '''
    @summary: 判断当前是否在Storage模式下，debug 版本.
    @param cliRet: cli回显
    @return:
        True: 当前在Storage模式下
        False: 当前不在Storage模式下
    '''
    if Regex.find(STORAGE_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False


def excuteCmdCommon(cli, cmd, lang):
    '''
    @summary: 执行cli命令，只判断是否拿到回显信息
    '''
    cliRet = ""
    errMsg = ""

    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)

    try:
        cliRet = cli.execCmd(cmd)
    except:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)

    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    return (True, cliRet, errMsg)


def getLunIdListWithCliRet(cli, lang):
    """
    @summary: 获取设备上所有LUN的ID
    """
    lunIdList = []

    cmd = "show lun general |filterColumn include columnList=ID"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, False, lang)
    if flag != True:
        return (flag, cliRet, errMsg, lunIdList)

    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg, lunIdList)

    lunInfoDictList = getHorizontalNostandardCliRet(cliRet)
    if len(lunInfoDictList) == 0:
        errMsg = getMsg(lang, "query.result.abnormal")
        return (False, cliRet, errMsg, lunIdList)

    for lunInfoDict in lunInfoDictList:
        lunIdList.append(lunInfoDict.get("ID"))

    return (True, cliRet, errMsg, lunIdList)


def getPresentCtrlNodeIdList(cli, lang):
    '''
    @summary: Get all present controller node ID List, such as, [0,1,2,3], not ['0A', '0B']
    @param cli: CLI connection.
    @param lang: Language.
    @return: (isQrySucc, cliRet, errMsg, nodeIdList)
    '''

    # Implement this method by execute diangose command 'sys showcls'
    isExeSucc, cliRet, _ = executeCmdInDebugMode(cli, 'sys showcls', True, lang)
    if not isExeSucc:
        return False, cliRet, getMsg(lang, "failed.to.get.present.nodeid"), []

    presentNodeIdList = []
    for line in cliRet.splitlines():
        fields = line.split()
        if len(fields) < 4 or not fields[0].isdigit():
            continue
        else:
            presentNodeIdList.append(fields[0].strip())

    return True, cliRet, '', presentNodeIdList


def getControllerEngineTopography(cli, lang):
    """
    @summary: fetch all enginId and corrosponding controllerIdlist info:

    :returns
    isOK:
    cliRet:
    erMsg:
    ctlrsTopographyTuple:(currentEngineId,currentCtlrId, {engineID: [ctlrIds]}

    """
    ctlrs_infos = ("", "", {})
    # 先尝试执行小系统下的指令
    cli_ret = cli.execCmd(MINISYSTEM_SYS_SHOW_CLS_CMD)
    if isInMinisystemMode(cli_ret):
        is_exe_succ = True
    else:
        is_exe_succ, cli_ret, _ = \
            executeCmdInDebugMode(cli, DIAGNOSE_SYS_SHOW_CLS_CMD, True, lang)
    if not is_exe_succ:
        err_msg = getMsg(lang, "failed.to.get.present.nodeid")
        return False, cli_ret, err_msg, ctlrs_infos

    engine_node_dict = {}
    cur_engine_id = ""
    current_node = ""
    vertical_ret = getVerticalCliRet(cli_ret)
    horizontal_ret = getHorizontalNostandardCliRet(cli_ret)

    current_node = vertical_ret[0].get("local node id")
    for horizontal_info in horizontal_ret:
        engine_id = horizontal_info.get("engine")
        controller_id = horizontal_info.get("id")
        if engine_id not in engine_node_dict:
            engine_node_dict[engine_id] = []
        if current_node == controller_id:
            cur_engine_id = engine_id
        engine_node_dict[engine_id].append(controller_id)

    ctlrs_topography_tuple = (cur_engine_id, current_node, engine_node_dict)

    return True, cli_ret, "", ctlrs_topography_tuple


def getControllerEngineTopographyMiniSys(cli, lang):
    """
    @summary: fetch all enginId and corrosponding controllerIdlist info:

    :returns
    isOK:
    cliRet:
    erMsg:
    ctlrsTopographyTuple:(currentEngineId,currentCtlrId, {engineID: [ctlrIds]}

    """
    cmd = "showsysstatus"
    ctlrs_infos = ("", "", {})
    is_exe_succ, cli_ret, _ = \
        excuteCmdInMinisystemModel(cli, cmd, lang)
    if not is_exe_succ:
        err_msg = getMsg(lang, "failed.to.get.present.nodeid")
        return False, cli_ret, err_msg, ctlrs_infos

    engine_node_dict = {}
    cur_engine_id = ""
    vertical_ret = getVerticalCliRet(cli_ret)
    horizontal_ret = getHorizontalNostandardCliRet(cli_ret)
    current_node = vertical_ret[0].get("local node id")
    for horizontal_info in horizontal_ret:
        engine_id = horizontal_info.get("engine")
        controller_id = horizontal_info.get("id")
        if engine_id not in engine_node_dict:
            engine_node_dict[engine_id] = []
        if current_node == controller_id:
            cur_engine_id = engine_id
        engine_node_dict[engine_id].append(controller_id)

    ctlrs_topography_tuple = (cur_engine_id, current_node, engine_node_dict)
    return True, cli_ret, "", ctlrs_topography_tuple


def get_current_ctrl_id(cli, lang):
    """查询用户节点
    :return current_node
    """
    flag, cli_ret, err_msg, ctrl_topography_tuple = getControllerEngineTopographyMiniSys(cli, lang)
    if not flag:
        raise Exception("query system status failed:%s" % str(err_msg))
    return ctrl_topography_tuple[1]


def getPresentCtrlNodeIdListForSshToRemote(cli, lang, engine):
    """
    @summary: 获取指定引擎下的节点（控制器）ID，然后转换为心跳参数
    """
    isExeSucc, cliRet, _ = executeCmdInDebugMode(cli, DIAGNOSE_SYS_SHOW_CLS_CMD, True, lang)
    if not isExeSucc:
        return False, cliRet, getMsg(lang, "failed.to.get.present.nodeid"), []

    presentNodeIdList = []
    for line in cliRet.splitlines():
        fields = line.split()
        if len(fields) < 4 or not fields[0].isdigit():
            continue

        enigeId = fields[-1].strip()
        if enigeId == engine:
            nodeId = fields[0].strip()
            nodeId = int(nodeId) % 4
            presentNodeIdList.append(nodeId)

    return True, cliRet, '', presentNodeIdList


def excute_cmd_in_minisystem_with_end_sign(cli, cmd, lang, end_with_sign, pwd=""):
    """
    @summary: 在minisystem模式下执行命令
    """

    cliRet = exec_cmd_has_end_with_sign(cli, cmd, end_with_sign)
    if "password:" in cliRet.lower():
        flag, ret, errMsg = excuteCmdInCliMode(cli, pwd, False, lang)
        return flag, ret, errMsg

    if isInMinisystemMode(cliRet):
        return True, cliRet, ""

    if isInDebugMode(cliRet):
        cliRet = cli.execCmd("exit")
        if "(y/n)" in cliRet:
            cliRet = cli.execCmd("y")

    if isInDeveloperMode(cliRet):
        cliRet = cli.execCmd("minisystem")
        if isInMinisystemMode(cliRet):
            cliRet = cli.execCmd(cmd)
            return True, cliRet, ""
        if not hasCliExecPrivilege(cliRet):
            return False, cliRet, getMsg(lang, "has.not.cli.privilege")
        return False, cliRet, ""

    flag, cliRet, errMsg = enterDeveloperMode(cli, lang)
    if flag is not True:
        return flag, cliRet, errMsg

    cliRet = cli.execCmd("minisystem")
    if not isInMinisystemMode(cliRet):
        if not hasCliExecPrivilege(cliRet):
            return False, cliRet, getMsg(lang, "has.not.cli.privilege")
        return False, cliRet, ""

    cliRet = exec_cmd_has_end_with_sign(cli, cmd, end_with_sign)
    if "password:" in cliRet.lower():
        flag, ret, errMsg = excuteCmdInCliMode(cli, pwd, False, lang)
        return flag, ret, errMsg
    return True, cliRet, ""


def exec_cmd_has_end_with_sign(cli, cmd, end_with_sign):
    """
    @summary: 查询执行命令时 是否含有指定结束符
    """
    if end_with_sign:
        cliRet = cli.execCmd(cmd, end_with_sign)
    else:
        cliRet = cli.execCmd(cmd)
    return cliRet


def isInMinisystemMode(cliRet):
    '''
    @summary: 判断当前是否在minisystem模式下
    '''
    if Regex.find(MINISYSTEM_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False


def testIsInMinisystemMode(cli):
    cliRet = cli.execCmd("\n")
    return isInMinisystemMode(cliRet)


def execute_cmd_y(cli, ret):
    """
    当回文中存在 y/n 时，执行y
    :param cli:
    :param ret:
    :return: ret_str: 全部回文
            ret: 带结束符的回文
    """
    ret_str = ''
    while ret.strip().endswith('(y/n)') or ret.strip().endswith('[y/n]'):
        ret = cli.execCmd('y')
        ret_str += "\n{}".format(ret)
    return ret_str, ret


def excuteCmdInMinisystemModel(cli, cmd, lang, pwd=""):
    """
    @summary: 在minisystem模式下执行命令
    """
    all_ret = ''
    cliRet = cli.execCmd(cmd)
    all_ret += cliRet
    ret_str, cliRet = execute_cmd_y(cli, cliRet)
    all_ret += "\n{}".format(ret_str) if ret_str else ''
    if "password:" in cliRet.lower():
        flag, ret, errMsg = excuteCmdInCliMode(cli, pwd, False, lang)
        all_ret += "\n{}".format(ret)
        return flag, all_ret, errMsg

    if isInMinisystemMode(cliRet):
        return True, all_ret, ""

    if isInDebugMode(cliRet):
        cliRet = cli.execCmd("exit")
        if "(y/n)" in cliRet:
            cliRet = cli.execCmd("y")
    if isInDeveloperMode(cliRet):
        cliRet = cli.execCmd("minisystem")
        if isInMinisystemMode(cliRet):
            cliRet = cli.execCmd(cmd)
            all_ret = cliRet
            ret_str, cliRet = execute_cmd_y(cli, cliRet)
            all_ret += "\n{}".format(ret_str) if ret_str else ''
            return True, all_ret, ""
        if not hasCliExecPrivilege(cliRet):
            return False, all_ret, getMsg(lang, "has.not.cli.privilege")
        return False, all_ret, ""

    flag, cliRet, errMsg = enterDeveloperMode(cli, lang)
    if flag is not True:
        return flag, cliRet, errMsg

    cliRet = cli.execCmd("minisystem")
    if not isInMinisystemMode(cliRet):
        if not hasCliExecPrivilege(cliRet):
            return False, cliRet, getMsg(lang, "has.not.cli.privilege")
        return False, cliRet, ""

    cliRet = cli.execCmd(cmd)
    all_ret = cliRet
    ret_str, cliRet = execute_cmd_y(cli, cliRet)
    all_ret += "\n{}".format(ret_str) if ret_str else ''
    if "password:" in cliRet.lower():
        flag, ret, errMsg = excuteCmdInCliMode(cli, pwd, False, lang)
        all_ret += "\n{}".format(ret)
        return flag, all_ret, errMsg
    return True, all_ret, ""


def enterCliModeFromSomeModel(cli, lang):
    '''
    @summary: 从其他模式进入cli模式
    '''
    excuRet = excuteCmdCommon(cli, "show system general", lang)
    if excuRet[0] == False:
        return excuRet

    if isInDebugMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if isInMinisystemMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if isInDeveloperMode(excuRet[1]) or isInStorageMode(excuRet[1]):
        excuRet = excuteCmdCommon(cli, "exit", lang)
        if "(y/n)" in excuRet[1]:
            excuRet = excuteCmdCommon(cli, "y", lang)

    if excuRet[1].endswith(":/>"):
        return (True, excuRet[1], "")

    return (False, excuRet[1], excuRet[2])


def sshToRemoteContr(cli, cmd, passWord, lang):
    """
    @summary: 在minisystem模式执行心跳命令到指定的控制器
    """
    allCliRet = ""
    flag, cliRet, errMsg = excuteCmdInMinisystemModel(cli, cmd, lang)
    allCliRet += cliRet
    if flag != True:
        return (flag, cliRet, errMsg)

    if '(yes/no)?' in cliRet:  # 首次心跳问题.
        allCliRet += "\n" + cliRet
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, 'yes', True, lang)

    if "password:" in cliRet:
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, passWord, False, lang)
        allCliRet += "\n" + cliRet
        if flag != True:
            return (flag, allCliRet, errMsg)

    if "System Name" not in cliRet:
        errMsg = getMsg(lang, "ssh.remote.failure")
        return (False, allCliRet, errMsg)

    return (True, allCliRet, errMsg)


def ssh_remote_ctrl_even_mini_sys(cli, cmd, passWord, lang):
    """
    跳控制器，即使目标控制器启动异常进入小系统
    :param cli:
    :param cmd:
    :param passWord:
    :param lang:
    :return:
    """
    allCliRet = ""
    flag, cliRet, errMsg = excuteCmdInMinisystemModel(cli, cmd, lang)
    allCliRet += cliRet
    if flag is not True:
        return (flag, cliRet, errMsg)

    if '(yes/no)?' in cliRet:  # 首次心跳问题.
        allCliRet += "\n" + cliRet
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, 'yes', True, lang)

    if "password:" in cliRet:
        flag, cliRet, errMsg = execute_cmd_when_power_on_fail_mode(
            cli, passWord, False, lang)
        allCliRet += "\n" + cliRet
        if flag is not True:
            return (flag, allCliRet, errMsg)

    if "System Name" not in cliRet and 'System Information' not in cliRet:
        errMsg = getMsg(lang, "ssh.remote.failure")
        return (False, allCliRet, errMsg)

    return (True, allCliRet, errMsg)


def exitHeartbeatCli(cli, lang):
    '''
    @summary: Exit from minisystem mode of peer controller to local controller CLI mode.
    @param cli: CLI connection
    @return:None
    '''
    try:
        for _ in range(3):
            (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)

            while "(y/n)" in cliRet:
                _, cliRet, _ = excuteCmdCommon(cli, "y", lang)

            # Compatible for debug version.
            if isInStorageMode(cliRet):
                (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)
    except:
        traceback.format_exc(None)
    finally:
        flag, ret, msg = excuteCmdCommon(cli, "show system general", lang)
        if flag is not True:
            cli.reConnect()


def exit_heart_beat_mini_sys(cli, exit_to_mini_sys, lang):
    """
    从远端退出，可能直接退出到登录控制器的mini system
    :param cli:
    :param lang:
    :return:
    """

    try:
        for _ in range(3):
            (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)

            while "(y/n)" in cliRet:
                _, cliRet, _ = excuteCmdCommon(cli, "y", lang)

            # Compatible for debug version.
            if isInStorageMode(cliRet):
                (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)
            if exit_to_mini_sys and isInMinisystemMode(cliRet):
                break
    except (Exception, JException):
        traceback.format_exc(None)


def getManagementPortInfo(cli, lang):
    """
    @summary: 获取设备管理端口信息（包含所在的引擎，连接状态，管理IP等）
    """
    managementPortInfo = []
    cmd = "show port general physical_type=ETH logic_type=Management_Port"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (flag, cliRet, errMsg, managementPortInfo)

    managementPortInfo = getHorizontalCliRet(cliRet)
    if len(managementPortInfo) == 0:
        errMsg = getMsg(lang, "query.result.abnormal")
        return (False, cliRet, errMsg, managementPortInfo)

    return (True, cliRet, errMsg, managementPortInfo)


def getDevPatchVersion(ssh, lang):
    cmd = "show upgrade package"
    (ok, cliRet, errorMsg) = excuteCmdInCliMode(ssh, cmd, True, lang)
    if not ok:
        return (False, "")

    beginIndex = cliRet.find('HotPatch Version')
    if beginIndex == -1:
        return (False, "")
    cliRetDictList = getHorizontalCliRet(cliRet[beginIndex:])
    hotPatchVersion = cliRetDictList[0].get('Current Version')
    return (True, hotPatchVersion)


def isSigleModel(cli, lang):
    '''
          控制器模式检查：
                  执行show system config_model命令，查看返回结果：Configuration Model为Multi-Controller时多多控，为Single-Controller时为单控
    '''
    cmd = "show system config_model"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    cliRet = checkRet[1]
    if "single" in cliRet.lower():
        return True
    else:
        return False


def getDevVersion(cli, lang, logger):
    '''
    @summary: 获取设备当前运行的系统版本版本
    @param context: 上下文
    @return:
        flag:
            True:获取设备系统版本成功
            False:获取设备系统版本失败
        devVersion:设备当前运行的系统版本
    '''
    cliRet = ""
    devVersion = ""
    try:
        cmd = "show upgrade package"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (False, devVersion, cliRet)

        cliRet = checkRet[1]
        endIndex = cliRet.find('HotPatch Version')
        cliRetDictList = getHorizontalCliRet(cliRet[:endIndex])
        devVersion = cliRetDictList[0].get('Current Version')
        return (True, devVersion, cliRet)

    except Exception as exception:
        logger.logException(exception)
        return (False, devVersion, getMsg(lang, "query.result.abnormal"))


def get_container_version(cli, lang, logger):
    '''
    @summary: 获取设备当前运行的容器版本
    @param context: 上下文
    @return:
        flag:
            True:获取设备容器版本成功
            False:获取设备容器版本失败
        container_version:设备当前运行的容器版本
    '''
    cli_ret = ""
    container_version = ""
    try:
        cmd = "show container_application general"
        check_ret = excuteCmdInCliMode(cli, cmd, True, lang)
        if check_ret[0] is not True:
            return (False, container_version, cli_ret)

        cli_ret = check_ret[1]
        data = getHorizontalCliRet(cli_ret)[0]
        container_version = data.get("Revision")
        return (True, container_version, cli_ret)

    except Exception as exception:
        logger.logException(exception)
        return (False, container_version, getMsg(lang, "query.result.abnormal"))


def getAvaliableCli(cli):
    '''
    @summary: 获取工具与阵列的SSH连接对象,并判断是否可用,不可用进行重连
    '''
    cmd = "show system general"
    for _ in range(3):
        try:
            cliRet = cli.execCmd(cmd)
            if "Health Status" in cliRet:
                return cli
            raise Exception()
        except:
            try:
                baseUtil.safeSleep(5)
                cli.close()
                cli.reConnect()
                return cli
            except:
                return cli
    return cli


def getSystemDate(cli, lang):
    '''
    @summary: 获取阵列系统时间，时间格式：2017-7-19/12:00:00
    '''
    sysDate = ""
    cmd = "show system general"
    ret = execCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return (sysDate, ret[1])

    cliRet = ret[1]
    sysInfoList = getVerticalCliRet(cliRet)
    for sysInfo in sysInfoList:
        sysDateInfo = sysInfo.get("Time", "")
        if sysDateInfo == "":
            continue
        sysDateInfoList = sysDateInfo.split(" ")
        if len(sysDateInfoList) < 1:
            return (sysDate, cliRet)

        sysDate = sysDateInfoList[0]
        return (sysDate, cliRet)

    return (sysDate, "")


def getAlarm(cli, lang, startTime, endTime):
    """获取某个时间段产生的告警
    当开始时间与结束时间相同时 获取当前系统所有告警

    :param cli: cli连接
    :param lang: 语言
    :param startTime: 开始时间 时间格式：2017-7-19/12:00:00
    :param endTime: 结束时间 时间格式：2017-7-19/12:00:00
    :return:
    """
    alarmIdList = []
    if startTime == endTime:
        cmd = "show alarm|filterColumn include columnList=ID"
    else:
        cmd = "show alarm from_time=%s to_time=%s" \
              "|filterColumn include columnList=ID" % (startTime, endTime)

    ret = execCmdInCliMode(cli, cmd, True, lang)
    cliRet = ret[1]
    infoList = getHorizontalCliRet(cliRet)
    for info in infoList:
        alarmId = info.get("ID", "")
        if alarmId == "":
            continue
        alarmIdList.append(alarmId)
    return alarmIdList


def getFreeDisks(cli, lang):
    cmd = "show disk general|filterColumn include columnList=ID,Capacity,Type|filterRow " \
          "column=Role predict=equal_to value=Free\sDisk " \
          "logicOp=and column=Health\sStatus predict=equal_to value=Normal " \
          "logicOp=and column=Running\sStatus predict=equal_to value=Online"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    cliRetLinesList = getHorizontalCliRet(checkRet[1])
    return cliRetLinesList


def getVerticalCliRetFilterElabel(cliRet, isParseElcLabel=True):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合（只处理表格中过滤显示电子标签场景）
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    elcLabel = ""
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break

        if Regex.search("^-+\r*\n*$", line):
            if not isParseElcLabel and lineDict.has_key("Electronic Label"):
                lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
                elcLabel = ""

            dictList.append(lineDict.copy())
            lineDict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            if isParseElcLabel and "=" in line:
                fields = line.split("=")
            elif "=" in line and Regex.search("^\s{16,}", line):
                elcLabel += line
                continue
            else:
                continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)

    if len(lineDict) > 0:
        if not isParseElcLabel and "Electronic Label" in lineDict:
            lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
        dictList.append(lineDict.copy())

    return dictList


def get_enclo_max_depth(cli, lang):
    """获取硬盘框最大级联深度

    :param cli: cli对象
    :param lang: 语言
    :return: 级联深度
    """
    # 默认4级级联
    default_depth = 4
    cmd = "show enclosure_max_depth general"
    check_ret = execCmdInCliMode(cli, cmd, True, lang)
    # 命令不支持，返回默认级联深度
    if not hasCliExecPrivilege(check_ret[1]):
        return default_depth

    # 其他查询失败场景，返回None
    if check_ret[0] is not True:
        return None
    cli_ret_lines_list = getVerticalCliRet(check_ret[1])
    if len(cli_ret_lines_list) == 0:
        return None

    deep = cli_ret_lines_list[0].get("Enclosure Max Depth", "4").strip()
    return int(deep)


def getLinkupSasPorts(cli, lang):
    '''
    @summary: 查询状态为link_up的SAS端口
    '''
    cmd = "show port general physical_type=SAS running_status=link_up"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    cliRetLinesList = getHorizontalCliRet(checkRet[1])
    linkupSasPorts = [line.get("ID") for line in cliRetLinesList]
    return linkupSasPorts


def getMgmtIpInfo(cli, lang):
    '''
    @summary: 获取管理IP信息
    '''
    cmd = "show system management_ip"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return (False, [])

    mgmtIpInfo = getVerticalCliRet(checkRet[1])
    if len(mgmtIpInfo) == 0:
        return (False, [])

    return (True, mgmtIpInfo)


def getControllerCacheCapacity(cli, lang):
    """获取设备控制器内存

    :param cli:  cli对象
    :param lang: 语言lang
    :return:
    """

    cmd = "show controller general|filterColumn include columnList=Cache\sCapacity"
    checkRet = execCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]:
        return checkRet

    cliRet = checkRet[1]
    cliRetLinesList = cliRet.splitlines()
    cacheCapacityList = []

    for line in cliRetLinesList:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()

        if fieldName == "Cache Capacity":
            cacheCapacity = fieldValue
            cacheCapacityList.append(cacheCapacity)

    if not cacheCapacityList:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return False, cliRet, errMsg

    return True, cacheCapacityList[0], ""


def queryDiskInfo(cli, lang, logger):
    '''
    @summary: 查询硬盘信息
    '''
    diskInfoList = {}
    sortedList = {}
    cliRet = ""
    errMsg = ""

    try:
        cmd = "show disk general |filterColumn include columnList=ID,Manufacturer,Model,Type,Role,Firmware\sVersion,Serial\sNumber,Health\sStatus,Running\sStatus,Disk\sDomain\sID"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]:
            logger.info("Execute cmd[%s] error" % cmd)
            return (False, sortedList, checkRet[2])

        cliRet = checkRet[1]
        diskInfoList = getHorizontalCliRet(cliRet)
        if len(diskInfoList) == 0:
            logger.info("Get vertical cliRet failed")
            errMsg = getMsg(lang, "cannot.get.disk.info")
            return (False, diskInfoList, errMsg)

        return (True, diskInfoList, "")
    except Exception as e:
        logger.error("Query disk info except:%s" % str(e))
        errMsg = getMsg(lang, "cannot.get.disk.info")
        return (False, sortedList, errMsg)


def getEventDetail(cli, lang, sequence):
    '''
    @summary: 通过流水号获取event详细信息
    '''
    eventDetail = {}
    cmd = "show event sequence=%s" % str(sequence)
    ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return eventDetail

    eventDetailList = getVerticalCliRet(ret[1])
    if len(eventDetailList) != 0:
        return eventDetailList[0]
    return eventDetail


def getDiskById(cli, lang, diskId):
    '''
    @summary: 获取单个硬盘信息
    '''
    diskDetail = {}
    cmd = "show disk general disk_id=%s" % str(diskId)
    ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if ret[0] != True:
        return diskDetail

    diskList = getVerticalCliRet(ret[1])
    if len(diskList) != 0:
        return diskList[0]
    return diskDetail


def query_disk_enc_info(cli, lang, logger):
    """查询硬盘框、控制框

    :param cli: cli对象
    :param lang: 语言
    :param logger: 日志对象
    :return: @param0: 查询是否成功 True-成功 False-失败
             @param1: 命令执行回显
             @param2: 错误消息
             @param3: 硬盘框ID列表
             @param4: 引擎框ID列表
    """
    # 硬盘框号列表
    disk_enc_list = []
    # 控制框号列表
    ctrl_enc_list = []
    try:
        cmd = "show enclosure"
        check_ret = excuteCmdInCliMode(cli, cmd, True, lang)
        if not check_ret[0]:
            logger.logError("Execute cmd[%s] error" % cmd)
            return False, check_ret[1], check_ret[2], disk_enc_list, ctrl_enc_list

        cli_ret = check_ret[1]
        enc_info_list = getHorizontalCliRet(cli_ret)
        if not enc_info_list:
            logger.logError("Get vertical cliRet failed")
            err_msg = getMsg(lang, "cannot.get.enclosure.info")
            return False, cli_ret, err_msg, disk_enc_list, ctrl_enc_list

        for enc_info in enc_info_list:
            enc_id = enc_info.get("ID", "")
            if enc_id.startswith("DAE"):
                disk_enc_list.append(enc_id)
            if enc_id.startswith("CTE"):
                ctrl_enc_list.append(enc_id)
        return True, cli_ret, "", disk_enc_list, ctrl_enc_list

    except Exception as e:
        logger.logException(e)
        err_msg = getMsg(lang, "cannot.get.enclosure.info")
        return False, "", err_msg, disk_enc_list, ctrl_enc_list


def get_member_disk_num(cli, lang, logger):
    """获取系统中成员盘信息

    :param cli: cli对象
    :param lang: 语言
    :param logger: 日志对象
    :return: @param0: 查询是否成功 True-成功 False-失败
             @param1: 命令执行回显
             @param2: 错误消息
             @param3: 成员盘列表
    """

    try:
        cmd = "show disk general|filterRow column=Role predict=equal_to value=Member\sDisk"
        check_ret = excuteCmdInCliMode(cli, cmd, True, lang)
        if not check_ret[0]:
            logger.logError("Execute cmd[%s] error" % cmd)
            return False, check_ret[1], check_ret[2], []

        cli_ret = check_ret[1]
        member_disk_list = getHorizontalCliRet(cli_ret)
        return True, "", cli_ret, member_disk_list

    except Exception as e:
        logger.logException(e)
        err_msg = getMsg(lang, "cannot.get.disk.info")
        return False, err_msg, "", []


def get_intf_info(cli, lang, logger):
    """获取系统中接口卡信息

    :param cli: cli对象
    :param lang: 语言
    :param logger: 日志对象
    :return: @param0: 查询是否成功 True-成功 False-失败
             @param1: 命令执行回显
             @param2: 错误消息
             @param3: 接口卡列表
    """
    cli_ret = ""
    try:
        cmd = "show interface_module"
        check_ret = excuteCmdInCliMode(cli, cmd, True, lang)
        cli_ret = check_ret[1]
        if check_ret[0] is not True:
            logger.logError("Execute cmd[%s] error" % cmd)
            return False, cli_ret, check_ret[2], []

        intf_list = getHorizontalCliRet(cli_ret)
        if not intf_list:
            logger.logError("Get vertical cliRet failed")
            err_msg = getMsg(lang, "cannot.get.intf.info")
            return False, cli_ret, err_msg, []
        return True, cli_ret, "", intf_list

    except Exception as e:
        logger.logException(e)
        err_msg = getMsg(lang, "cannot.get.intf.info")
        return False, cli_ret, err_msg, []


def getFreeMemInCliRet(cliRet):
    '''
    @summary:从CLI回显获取空闲内存
    :param cliRet:
    :return:
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    title_list = ""
    for line in cliRetList:
        if "free" in line:
            title_list = line.split()
            continue
        if "Mem:" in line:
            mem_str = line.split(":")[1]
            continue
    free_index = title_list.index("free")
    mem_list = mem_str.split()
    if len(mem_list) != len(title_list):
        return (False, 0)
    free_mem = mem_list[free_index]

    return (True, free_mem)


def restAnalysisMethod(result):
    """
    需要判断是否原始返回值
    :param result:
    :return:
    """
    if type(result) == 'dict' and "data" in result:
        return result.get("data", [])

    return result


def return_one_item_list(cli_ret):
    """
    针对规则中可以直接使用设备型号、设备软件版本和设备热补丁版本的情况，只需返回只有一个元素的列表
    :param cli_ret:
    :return:
    """
    return [cli_ret]


class executeView:
    cli = "cli"
    developer = "developer"
    minisysterm = "minisysterm"
    debug = "debug"
    rest = "rest"
    brocadeCli = "brocadeCli"


analysisMethodMappingView = {
    "Horizontal": getHorizontalCliRet,
    "vertical": getVerticalCliRet,
    "restAnalysisMethod": restAnalysisMethod,
    "return_one_item_list": return_one_item_list
}


def get_elabel_capacity(cli, lang, logger):
    """获取设备下所有盘容量和电子标签容量

    :param cli: cli连接
    :param disk_domain_id: 硬盘域ID
    :return:
    """

    # 获取所有硬盘信息
    cmd = r"show disk general" \
          r"|filterColumn include columnList=ID,Capacity,Disk\sDomain\sID"
    logger.info("get disk info cmd is: %s" % cmd)
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)
    if not flag:
        logger.error("cannot get disk info: %s" % err_msg)
        return flag, {}, err_msg

    disk_info_list = getHorizontalCliRet(cli_ret)
    logger.info("get disk info success: %s" % str(disk_info_list))

    # 按容量整理硬盘硬盘信息并获取电子标签容量
    elabel_capacity_map = {}
    for disk in disk_info_list:
        disk_id = disk.get("ID")
        capacity = disk.get("Capacity")
        elabel_capacity = elabel_capacity_map.get(capacity, None)
        if elabel_capacity is None:
            flag, elabel_capacity, err_msg = \
                get_disk_capacity_by_disk_id(cli, disk_id, lang)
            if not flag:
                continue
            elabel_capacity_map[capacity] = elabel_capacity

    logger.info("elabel_capacity_map is: %s" % str(elabel_capacity_map))
    return True, elabel_capacity_map, ""


def get_disk_capacity_by_disk_id(cli, disk_id, lang, isHasLog=True):
    """根据硬盘ID获取硬盘容量

    :param cli: CLI连接
    :param disk_id: 硬盘ID
    :param lang: 语言
    :param isHasLog: 记录日志
    :return:
    """
    err_msg = ""
    check_ret = get_disk_electronic_label_by_id(cli, disk_id, lang, isHasLog)
    if not check_ret[0]:
        return False, check_ret[1], check_ret[2]
    cli_ret = check_ret[1]
    # 匹配电子标签中的硬盘容量大小
    disk_capacity = None
    cli_ret_list = check_ret[3].split("  ")
    electronic_list = filter(lambda x: x != "", cli_ret_list)
    flag = False
    for line in electronic_list:
        if not re.search("Description", line, re.IGNORECASE):
            continue
        desc_value_list = line.split("=")
        if len(desc_value_list) < 2:
            err_msg = \
                getMsg(lang, "failed.to.get.disk.electronic.label.info",
                       disk_id)
            return False, cli_ret, err_msg
        result_list = desc_value_list[1].split(",")
        for result in result_list:
            disk_value_list = result.strip().split(' ')
            for disk_value in disk_value_list:
                if not re.search(r"^[0-9]+(\.\d+)?((G$)|(GB$)|(T$)|(TB$))",
                                 disk_value):
                    continue
                if disk_value.startswith("6G"):  # SAS速率跳过
                    continue
                if disk_value.endswith("GB") or disk_value.endswith("TB"):
                    flag = True
                    disk_capacity = disk_value
                    break
                if disk_value.endswith("T"):
                    disk_capacity = disk_value.replace("T", "TB")
                    flag = True
                    break
                if disk_value.endswith("G"):
                    disk_capacity = disk_value.replace("G", "GB")
                    flag = True
                    break
            if flag:
                return True, disk_capacity, err_msg

    err_msg = getMsg(lang, "failed.to.get.disk.electronic.label.info", disk_id)
    return False, "", err_msg


def get_disk_electronic_label_by_id(cli, disk_id, lang, isHasLog=True):
    """执行CLI命令获取硬盘容量

    :param cli: CLI连接
    :param disk_id: 硬盘ID
    :param lang: 语言
    :param isHasLog: 记录日志
    :return:
    """
    elec_label_str = ""
    cmd = "show disk general disk_id=%s" % disk_id
    flag, cli_ret, err_msg = execCmdInCliMode(cli, cmd, isHasLog, lang)
    if not flag:
        err_msg = getMsg(lang, "failed.to.get.disk.info", disk_id)
        return False, cli_ret, err_msg, elec_label_str

    ret_list = getVerticalCliRetFilterElabel(cli_ret, isParseElcLabel=False)

    if not ret_list:
        err_msg = getMsg(lang, "failed.to.get.disk.info", disk_id)
        return False, cli_ret, err_msg, elec_label_str

    for line in ret_list:
        if "Electronic Label" in line:
            elec_label_str = line.get("Electronic Label")
            return True, cli_ret, err_msg, elec_label_str
    err_msg = getMsg(lang, "failed.to.get.disk.info", disk_id)

    return False, cli_ret, err_msg, elec_label_str


def reConnectionCli(cli, logger):
    """
    @summary: 重新获取SSH通道
    """
    if cli is None:
        return False
    try:
        cli.reConnect()
        logger.info("reconnect to device by ssh.")
        return True
    except Exception as e:
        logger.error(str(e))
    except Throwable as throwable:
        logger.error(str(throwable))
        return False


def is_in_vstore_mode(cli_ret):
    if Regex.find(VSTORE_MODEL_FLAG, cli_ret, Regex.IGNORECASE):
        return True
    return False


def excute_cmd_in_vstore_model(cli, cmd, lang, vstore_id):
    """
    在租户模式（vstore）下执行命令
    :param cli: ssh 链接对象
    :param cmd: 待执行的命令
    :param lang: 语言
    :param vstore_id: 租户ID
    :return: 执行结果
    """
    enter_vstore_cmd = "change vstore view id={}".format(vstore_id)
    cli_ret = excuteCmdInCliMode(cli, enter_vstore_cmd, True, lang)
    if not cli_ret[0]:
        return cli_ret

    if not is_in_vstore_mode(cli_ret[1]):
        return False, cli_ret[1], ""

    cli_ret = excuteCmdInCliMode(cli, cmd, True, lang)
    if is_in_vstore_mode(cli_ret[1]):
        exit_vstore_mode_to_cli_mode(cli)
    return cli_ret


def exit_vstore_mode_to_cli_mode(cli):
    """
    从租户模式退出到cli模式
    :param cli:
    :return:
    """
    cli.execCmd("exit")


def getProductModelAndVersion(cli, lang):
    """
    @summary: get product model and version
    @return isQryStatusOk,boolean
    @return:  product_model,str
    @return:  product_version, str
    @return: errmsg, str
    """
    cmd = "show system general"
    check_ret = excuteCmdInCliMode(cli, cmd, True, lang)

    if check_ret[0] is not True:
        err_msg = check_ret[2] if len(check_ret) >= 2 else ""
        return False, "", "", err_msg

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    product_model = restUtil.CommonRest.getOriginProductModel(cli.getdNode())
    product_version = ""
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if not product_model and field_name == "Product Model":
            product_model = field_value
        elif field_name == "Product Version":
            product_version = field_value

        if (
                len(product_model) != 0
                and len(product_version) != 0
                and product_model != "--"
                and product_version != "--"
        ):
            return True, product_model, product_version, cli_ret

    return (
        False,
        product_model,
        product_version,
        getMsg(lang, "cannot.get.product.version.info"),
    )


def has_smart_cache_pool(cli, lang, logger):
    cmd = "show smart_cache_pool general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return flag, False, cli_ret, err_msg

    if queryResultWithNoRecord(cli_ret):
        return flag, False, cli_ret, err_msg
    else:
        return flag, True, cli_ret, err_msg


def get_smart_cache_pool_for_nas(cli, lang, logger):
    """
    获取设备上的SCM池ID（只适用于DoradoNas）
    -1 表示无SmartCachePool
    """
    scm_pool_id = "-1"
    cmd = "show smart_cache_pool general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return flag, scm_pool_id, cli_ret, err_msg

    if queryResultWithNoRecord(cli_ret):
        return flag, scm_pool_id, cli_ret, err_msg

    ret_dict = getHorizontalCliRet(cli_ret)
    if ret_dict:
        for record in ret_dict:
            scm_pool_id = record.get("ID")

    return flag, scm_pool_id, cli_ret, err_msg


def get_smart_cache_pool_disk_for_nas(cli, pool_id, lang, logger):
    """
    获取设备上Smart Cache Pool里的Disk ID(只适用于DORADO Nas)
    """
    disk_info_list = []
    cmd = "show smart_cache_pool disk smart_cache_pool_id={}" \
        .format(str(pool_id))
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return flag, disk_info_list, cli_ret, err_msg

    if queryResultWithNoRecord(cli_ret):
        return flag, disk_info_list, cli_ret, err_msg

    ret_dict = getHorizontalCliRet(cli_ret)
    if ret_dict:
        disk_info_list = pack_scm_disk_info_list(ret_dict)

    return flag, disk_info_list, cli_ret, err_msg


def pack_scm_disk_info_list(ret_dict):
    """
    打包scm盘信息列表
    :param ret_dict:
    :return: scm盘信息列表
    """
    disk_info_list = []
    for record in ret_dict:
        disk_info = pack_scm_disk_info(record)
        disk_info_list.append(disk_info)
    return disk_info_list


def pack_scm_disk_info(record):
    """
    打包scm盘信息
    :param record:
    :return: scm盘信息
    """
    return {
        "location": record.get("ID"),
        "healthStatus": record.get("Health Status"),
        "runningStatus": record.get("Running Status"),
        "type": record.get("Type"),
        "capacity": record.get("Capacity")
    }


def get_ip_enclosures_for_dorado_v6(cli, lang, logger):
    """获取dorado v6设备的智能框列表

    :param cli: cli连接
    :return: 智能框列表
    """
    cmd = "show enclosure"
    flag, cli_ret, err_msg = execCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return [], cli_ret

    enclosures = getHorizontalCliRet(cli_ret)
    ip_enclosures = []
    for enclosure in enclosures:
        enclosure_id = enclosure.get("ID")
        enclosure_logic_type = enclosure.get("Logic Type")
        enclosure_type = enclosure.get("Type")
        if enclosure_logic_type == "Expansion Enclosure" \
                and "smart" in enclosure_type.lower():
            ip_enclosures.append(enclosure_id)

    return ip_enclosures, cli_ret


def get_free_scm_disk(cli, lang, logger):
    """获取设备上空闲的SCM盘
    """
    disk_info_list = []
    cmd = "show disk general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return flag, disk_info_list, cli_ret, err_msg

    if queryResultWithNoRecord(cli_ret):
        return flag, disk_info_list, cli_ret, err_msg

    ret_dict = getHorizontalCliRet(cli_ret)
    if ret_dict:
        for record in ret_dict:
            if record.get("ID").startswith("CTE") \
                    and record.get("Type").startswith("SCM") \
                    and record.get("Role").startswith("Free Disk"):
                disk_info = pack_scm_disk_info(record)
                disk_info_list.append(disk_info)

    return flag, disk_info_list, cli_ret, err_msg


def expand_smart_cache_pool(cli, smart_cache_pool_id, disk_id_str, lang,
                            logger):
    """将指定硬盘扩入Smart Cache Pool
    """
    cmd = "add smart_cache_pool disk_list={} id={}".format(disk_id_str,
                                                           smart_cache_pool_id)
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    return flag, cli_ret, err_msg


def get_system_free_memory_with_ret(data_dict):
    '''
    @summary: 系统空闲空间大小，单位MB
    '''
    lang = data_dict.get("language")
    logger = baseUtil.getLogger(data_dict.get("logger"), __file__)
    cli = get_cli_for_svp(data_dict, logger)
    cmd = "free -m"
    free_mem = 0
    cli_ret = ""
    try:
        flag, cli_ret, err_msg = excuteCmdInMinisystemModel(cli, cmd, lang)

        if not flag:
            logger.info("[uploadUtil] get free mem error:%s" % str(err_msg))
            return False, free_mem, err_msg, cli_ret
        flag, free_mem = getFreeMemInCliRet(cli_ret)
        if not flag:
            logger.info("[uploadUtil] format free mem error")
            return False, free_mem, getMsg(lang, "query.result.abnormal"), cli_ret
        logger.info("[uploadUtil]free mem:%s ." % str(free_mem))
        # 退出minisystem
        exitMinisystemToCliMode(cli)
        return True, free_mem, "", cli_ret
    except Exception as ex:
        logger.logException(ex)
        # 退出minisystem
        exitMinisystemToCliMode(cli)
        return False, free_mem, getMsg(lang, "query.result.abnormal"), cli_ret


def get_user_privilege_with_cli_ret(cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, cliRet, userPrivilege, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        cliRet: 返回cli回显结果
        userPrivilege:
            flag为True时，返回用户权限级别
            flag为False时，返回""
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang)
        if checkRet[0] != True:
            return (checkRet[0], checkRet[1], "", checkRet[2])

        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (checkRet[0], checkRet[1], "", checkRet[2])

        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)

        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, "", errMsg)

        for line in cliRetLinesList:
            name = line["Name"]
            if name == userName:
                userPrivilege = line["Level"]
                return (True, cliRet, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, "", errMsg)

    return (False, cliRet, "", errMsg)


def get_performance_layer_id(cli, lang):
    """
    获取设备上的所有性能层ID
    """
    smart_pool_id = []
    cmd = "show performance_layer general"
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return False, smart_pool_id, cli_ret

    ret_dict = getHorizontalCliRet(cli_ret)
    for record in ret_dict:
        if not record.get("ID"):
            continue
        smart_pool_id.append(record.get("ID"))

    return flag, smart_pool_id, cli_ret


def get_disk_num_in_performance_layer(cli, lang, performance_layer_id):
    """
    获取性能层盘数
    :param cli: cli
    :param lang: lang
    :param performance_layer_id: 性能层ID
    :return: 性能层盘数
    """
    disk_num = 0
    cmd = "show performance_layer general performance_layer_id={}".format(
        performance_layer_id)
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)

    if flag is not True:
        return False, disk_num
    ret_dict = getVerticalCliRet(cli_ret)
    if not ret_dict:
        return False, disk_num
    return True, ret_dict[0].get("Disk Number")


def anonymize_ret(content, src, replacement):
    """
    :param content: 待替换的原始文本
    :param src: 被替换的对象
    :param replacement:替换后的目标值
    :return:替换后的文本
    """
    return content.replace(src, replacement)


def get_performance_layer_id_and_engine_dict(cli, lang):
    """
    获取性能层ID和引擎的关联字典
    :param cli: cli连接
    :param lang: 语言
    :return:
    """
    flag, performance_layer_ids, cli_ret = get_performance_layer_id(cli, lang)
    if flag is not True:
        raise Exception("get performance_layer id failed.")
    performance_layer_id_and_engine_dict = {}
    for performance_layer_id in performance_layer_ids:
        engine_list = get_share_performance_layer_engine(performance_layer_id, cli, lang)
        if not engine_list:
            continue
        performance_layer_id_and_engine_dict[str(performance_layer_id)] = engine_list
    return performance_layer_id_and_engine_dict


def get_share_performance_layer_engine(performance_layer_id, cli, lang):
    """
    获取共享性能层的引擎
    :param performance_layer_id: 性能层ID
    :param cli: ssh连接
    :param lang: 语言
    :return:
    """
    cmd = "show performance_layer general performance_layer_id={}".format(
        performance_layer_id)
    flag, cli_ret, err_msg = excuteCmdInCliMode(cli, cmd, True, lang)
    records = getVerticalCliRet(cli_ret)
    for record in records:
        if record.get("Share Strategy") == "Unique":
            return ""
        return record.get("Controller Enclosure", "").split(",")


def excute_cmd_in_minisystem_model_different_enclosure(enclosure_name, cli, cmd, lang, pwd=""):
    """
    根据当前框类型执行不同的内置方法
    """
    if "CTE" not in enclosure_name:
        # 判断当前是否为minisystem状态
        if not testIsInMinisystemMode(cli):
            # 首先进入minisystem
            cli.execCmd("minisystem")
        return cli.execCmd(cmd)
    return excuteCmdInMinisystemModel(cli, cmd, lang, pwd)[1]


def enter_in_multi_zone_model(cli, logger, lang, zone_ids, pwd):
    """
    a800设备进入多zone模式
    :param cli: cli连接
    :param logger: 日志对象
    :param lang: 当前语言
    :param zone_ids: zoneid列表，格式为 [0:0A,1:0A]或者[0,1]，
    冒号前为zoneid，冒号后为控制器编号，任意控制器即可，若没有指定控制器，则全部采用0A
    :return:
    """
    all_ret = []
    flag, origin_info, _ = enterDeveloperMode(cli, lang)
    all_ret.append(origin_info)
    if not flag:
        logger.info("chang to developer model failed with res :%s" % origin_info)
        return flag, os.sep.join(all_ret)
    cmd = "change cluster_scope general scope=multizone"
    ret = cli.execCmd(cmd)
    all_ret.append(ret)
    cli.appendSshEndJudgeIntf(["Please enter the password to login to multizone nodes:"])
    while "y/n" in ret:
        ret = cli.execCmd("y")
        all_ret.append(ret)
    logger.info("start input node pa with ret: %s", ret)
    if "Please enter the password" in ret:
        ret = cli.execCmdNoLog(pwd)
        all_ret.append(ret)
    logger.info("input node pa with res %s" % ret)
    cmd = "change multizone config type=cli node_list=" + ";".join(_build_multi_zone_param(zone_ids))
    logger.info("start set multi zone with cmd %s" % cmd)
    ret = cli.execCmd(cmd)
    all_ret.append(ret)
    if "successfully" not in ret:
        logger.info("chang to multi zone failed with res :%s" % ret)
        return False, os.sep.join(all_ret)
    return True, os.sep.join(all_ret)


def _build_multi_zone_param(zone_ids):
    id_param = []
    for zone_id in zone_ids:
        if ":" not in zone_id:
            id_param.append(zone_id + ":0A")
        else:
            id_param.append(zone_id)
    return id_param


def exit_multi_zone_mode(cli):
    if "[MultiZone]" in cli.execCmd("\n"):
        cli.execCmd("exit")
