# -*- coding: UTF-8 -*-
import ast
import re

from com.huawei.ism.tlv import TLVUtils
from com.huawei.ism.tlv.bean import Param
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tlv.lang import UnsignedInt64
from com.huawei.ism.tlv.docoder import ParamType
from java.util import ArrayList

from cbb.frame.base import config
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import Tlv2Rest
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvDataType
from cbb.frame.tlv.tlvFactory import Tlv
from cbb.frame.tlv.baseFactory import log as Log
from cbb.frame.tlv.tlvEnumFactory import TlvEnum
from cbb.frame.base import jsonUtil
from cbb.frame.context import contextUtil
from cbb.frame.base import constants
from cbb.frame.util.common import wrapAllException
from com.huawei.ism.exception import IsmException
from java.lang import Exception as JException
from com.huawei.ism.tlv.bean import Record

RETRY_TIMES = 5  # 建立连接时重试次数
TLV_CMD_TIMEOUT = 120
TLV_RECORD_END_INDEX = 4095
CMD_DEFAULT_TIMEOUT = 60 * 5  # 执行TLV命令默认超时时间
ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE = "1089101853"
# 命令执行失败
TLV_CAN_NOT_EXECUTE = "1077936892"
COMMUNICATION_FAIL_ERROR_CODE = [
    "16797697",  # 与设备断开连接
    "1073949185",  # 与设备通信异常，请检查网络连接或设备状态是否正常
    "16797698",  # 与设备通信异常
]


def getParamValue(param_type, param_value):
    """将参数转换为tlv参数

    :param param_type: 参数类型
    :param param_value: 参数值
    :return:
    """
    if param_type == tlvDataType.DATA_TYPE_UNSIGN_INT or \
            param_type == tlvDataType.DATA_TYPE_SIGN_INT or \
            param_type == tlvDataType.DATA_TYPE_BOOL or \
            param_type == tlvDataType.DATA_TYPE_ENUM or \
            param_type == tlvDataType.DATA_TYPE_TIME:
        return UnsignedInt32(param_value)

    elif param_type == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        return UnsignedInt64(param_value)

    elif param_type == tlvDataType.DATA_TYPE_STRING or \
            param_type == tlvDataType.DATA_TYPE_ARRAY or \
            param_type == tlvDataType.DATA_TYPE_JSON:
        return str(param_value)

    return param_value


def getSfpOpticalTransceiver(tlv):
    """获取光模块信息

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'],
             tlvData.OM_OBJ_E['SFP_OPTICAL_TRANSCEIVER'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getParamList(params):
    """获取tlv参数列表

    :param params: 参数列表
    :return:
    """
    param_list = []
    if not params:
        return param_list
    for param_define, param_value in params:
        param_index = param_define["index"]
        param_type = param_define["type"]
        param_value = getParamValue(param_type, param_value)
        param_list.append(Param(param_index, param_type, param_value))
    return TLVUtils.paramList(param_list)


def getRecordsValue(records, field, param):
    """获取tlv记录中的值

    :param records: 数据列表
    :param field: 字段
    :param param: 参数
    :return:
    """
    return filter(
        lambda x: str(Tlv2Rest.getRecordValue(x, field)) == str(param),
        records)


def getRecordValueFruOrExp(record, field):
    """获取tlv记录中的值

    :param record: 数据列表
    :param field: 字段
    :return:
    """
    return Tlv2Rest.getRecordValue(record, field)


def getRecordValueByj(record, field):
    """获取tlv记录中的值

    :param record:
    :param field:
    :return:
    """
    try:
        fieldIndex = field["index"]
        fieldType = field["type"]

        if fieldType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
                fieldType == tlvDataType.DATA_TYPE_SIGN_INT or \
                fieldType == tlvDataType.DATA_TYPE_ENUM or \
                fieldType == tlvDataType.DATA_TYPE_TIME:
            val = record.getParamIntValue(fieldIndex)
            if val:
                return val.intValue()
            else:
                return 0

        elif fieldType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
            val = record.getParamIntValue(fieldIndex)
            if val:
                return val.longValue()
            else:
                return 0

        elif fieldType == tlvDataType.DATA_TYPE_STRING or \
                fieldType == tlvDataType.DATA_TYPE_ARRAY or \
                fieldType == tlvDataType.DATA_TYPE_BOOL or \
                fieldType == tlvDataType.DATA_TYPE_JSON:
            val = record.getParamStrValue(fieldIndex)
            if val:
                return str(val)
            else:
                return ""

        return record.getParamValue(fieldIndex)
    except Exception as e:
        raise Exception(config.ERR_CODE_CAN_NOT_EXECUTE,
                        "exec cmd return data format error: %s" % str(e))


def getRecordValue(record, field):
    """获取tlv记录中的值

    :param record:
    :param field:
    :return:
    """
    try:
        if isinstance(record, Record):
            return getRecordValueByj(record, field)
        if type(record) == ArrayList or type(record) == list:
            record = record[0]

        field_index = field.get("index")
        field_type = field.get("type")
        param_info = record.get(str(field_index), None)
        if param_info is None:
            return ''

        param_value = param_info.get("value")

        if field_type == tlvDataType.DATA_TYPE_UNSIGN_INT or \
                field_type == tlvDataType.DATA_TYPE_SIGN_INT or \
                field_type == tlvDataType.DATA_TYPE_BOOL or \
                field_type == tlvDataType.DATA_TYPE_ENUM or \
                field_type == tlvDataType.DATA_TYPE_TIME:
            return int(param_value)
        elif field_type == tlvDataType.DATA_TYPE_JSON:
            return jsonUtil.jsonStr2DictNew(param_value)
        elif field_type == tlvDataType.DATA_TYPE_U64LONG_TYPE:
            return long(param_value)
        elif field_type == tlvDataType.DATA_TYPE_ARRAY:
            return ast.literal_eval(param_value)

        else:
            return unicode(param_value)
    except Exception as e:
        raise Exception(config.ERR_CODE_CAN_NOT_EXECUTE,
                        "exec cmd return data format error: %s" % str(e))


def exec_cmd(tlv, cmd, params, isBatch=False, getBatch=False,
             timeOut=TLV_CMD_TIMEOUT):
    """
    执行tlv命令

    @param tlv: tlv对象
    @param cmd: 命令字
    @param params: 参数列表
    @param isBatch: 是否为批量处理（参考GET_BATCH_NEXT的实现的批量处理，
    批处理最后一条记录有结束符标记）
    @param getBatch: 是否为获取批量内容（参考非GET_BATCH_NEXT的实现的批量处理，
    批处理最后一条记录没有结束符标记）
    @param timeOut: 超时时间
    """
    try:
        records = None
        paramList = params

        if isBatch:
            records = tlv.bacthNext(cmd, paramList, timeOut)
            if records is not None and len(records) > 1:
                lastRecord = records[-1]
                lastRecordParamList = lastRecord.getParams()
                if lastRecordParamList[0].getIndex() == TLV_RECORD_END_INDEX:
                    records.pop()
            return records

        if getBatch:
            recs = tlv.getBatch(cmd, paramList, timeOut, RETRY_TIMES)
            if recs is not None and recs.size() > 0:
                tailRec = recs.get(recs.size() - 1)
                if tailRec.getParams().size() == 0:
                    raise
                headIndex = tailRec.getParams().get(0).getIndex()
                headIndex1 = ""
                if tailRec.getParams().size() > 1:
                    headIndex1 = tailRec.getParams().get(1).getIndex()
                if TLV_RECORD_END_INDEX in [headIndex, headIndex1]:
                    recs.remove(recs.size() - 1)
            return recs

        return tlv.invoke(cmd, paramList, timeOut)
    except IsmException, ex:
        raise ex
    except (Exception, JException), e:
        raise IsmException(int(TLV_CAN_NOT_EXECUTE), e)


def execCmd(tlv, cmd, params, isBatch=False, getBatch=False,
            timeOut=TLV_CMD_TIMEOUT, reTry=True,
            retry_interval=config.REST_DEFAULT_RETRY_INTERVAL):
    """执行tlv命令，下发异常可进行重试

    :param tlv: tlv对象
    :param cmd: 命令字
    :param params: 参数列表
    :param isBatch: 是否为批量处理
    :param getBatch: 是否为获取批量内容
    :param timeOut: 超时时间
    :param reTry: 是否重试
    :param retry_interval: 重试间隔时间
    :return:
    """
    try:
        return execCmdOnce(tlv, cmd, params, isBatch, getBatch, timeOut,
                           retry_interval=retry_interval)
    except Exception as exception:
        err_code = exception.args[0]
        # 往从节点下发消息失败，可以忽略。
        if err_code == ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE:
            return
        raise exception


def execCmdOnce(tlv, cmd, params, isBatch=False, getBatch=False,
                timeOut=TLV_CMD_TIMEOUT,
                retry_interval=config.REST_DEFAULT_RETRY_INTERVAL):
    """执行tlv命令

    :param tlv: tlv对象
    :param cmd: 命令字
    :param params: 参数列表
    :param isBatch: 是否为批量处理
    :param getBatch: 是否为获取批量内容
    :param timeOut: 超时时间
    :param retry_interval: 重试间隔时间
    :return:
    """
    rest_cmd = {'cmd': cmd, 'timeout': timeOut}
    return Tlv2Rest.execCmd(tlv, rest_cmd, params,
                            retry_interval=retry_interval)


def getHardwareRecords(tlv, hardware_type):
    """根据硬件形态获取该硬件的所有记录

    :param tlv: tlv对象
    :param hardware_type: 硬件形态
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E[hardware_type])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getBitMap(ctrls):
    """根据集群中控制器数量获取BitMap

    :param ctrls:
    :return:
    """
    binary_system = "".join(["1" for i in range(0, ctrls)])
    return int(binary_system, 2)


def bit2GB(bit):
    """比特容量转换为GB容量

    :param bit:
    :return:
    """
    return bit * 1.0 / 1024 / 1024 / 1024


def getInterfaceModuleRecords(tlv):
    """获取当前集群所有接口卡的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['INTF_MODULE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getEthPortRecords(tlv):
    """获取当前集群所有ETH端口的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getMgmtPorts(tlv):
    """获取当前集群所有管理端口的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.ETH_PORT['logicType'], tlvData.PORT_LOGIC_TYPE_E["MNGT"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getMntnPorts(tlv):
    """获取当前集群所有维护口的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.ETH_PORT['logicType'],
             tlvData.PORT_LOGIC_TYPE_E["MAINTENANCE"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getScaleOutPortsInfo(tlv, port_location_config):
    """获取指定节点用于ScaleOut扩容的端口集合

    :param tlv: 指定节点
    :param port_location_config: 端口位置
    :return:
    """
    records = getEthPortRecords(tlv)
    if records is None:
        return []

    eth_port_info = []
    for record in records:

        location = getRecordValue(record, tlvData.PUB_ATTR['location'])
        if location not in port_location_config:
            continue

        port_info_dict = {}
        port_info_dict["portID"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['id'])
        port_info_dict["location"] = location
        port_info_dict["healthStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['healthStatus'])
        port_info_dict["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        port_info_dict["speed"] = getRecordValue(
            record,
            tlvData.ETH_PORT['speed'])
        port_info_dict["macAddress"] = getRecordValue(
            record,
            tlvData.ETH_PORT['macAddress'])

        eth_port_info.append(port_info_dict.copy())

    return eth_port_info


def cmpPortDict(port_dict1, port_dict2):
    """端口字典比较

    :param port_dict1: 端口字典1
    :param port_dict2: 端口字典2
    :return: 根据bayid和location进行比较的结果
    """
    if port_dict1["bayid"] < port_dict2["bayid"]:
        return -1
    elif port_dict1["bayid"] > port_dict2["bayid"]:
        return 1
    if port_dict1["location"] < port_dict2["location"]:
        return -1
    elif port_dict1["location"] > port_dict2["location"]:
        return 1
    return 0


def getSASPortsById(tlv, port_id):
    """根据端口ID获取SAS端口信息

    :param tlv: tlv对象
    :param port_id: 端口ID
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SAS_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(port_id))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)


def getEthPort(tlv, port_id):
    """根据特定端口ID获取该端口的信息

    :param tlv: tlv对象
    :param port_id: ETH端口ID
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(port_id))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)


def setEthPortLinkStatus(tlv, port_id, enum_status):
    """设置特定端口状态

    :param tlv: tlv对象
    :param port_id: ETH端口ID
    :param enum_status: 端口状态枚举值
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(port_id))
    params.append(param)
    param = (tlvData.PUB_ATTR['runningStatus'], enum_status)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY"], params, isBatch=False)


def getPCIePortRecords(tlv):
    """获取当前集群所有PCIe端口的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getPciePortLocation(tlv, port_id):
    """根据特定端口ID获取该端口的Location

    :param tlv: tlv对象
    :param port_id: PCIe端口ID
    :return:
    """
    try:
        params = []
        param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
        params.append(param)
        param = (tlvData.PUB_ATTR['id'], str(port_id))
        params.append(param)
        record = execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)
        return getRecordValue(record, tlvData.PUB_ATTR["location"])
    except Exception:
        return ""


def getSASPortRecords(tlv):
    """获取当前集群所有SAS端口的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SAS_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getEnclosureRecords(tlv):
    """获取当前集群所有框信息的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getPowerRecords(tlv):
    """获取当前集群所有电源信息的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['POWER'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getControllerEnclosureRecords(tlv):
    """获取当前集群所有控制框信息的记录

    :param tlv: tlv对象
    :return:
    """
    controller_enc = []
    records = getEnclosureRecords(tlv)
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            controller_enc.append(record)
    return controller_enc


def getDataSwitchRecords(tlv):
    """获取当前集群所有交换机信息的记录

    :param tlv: tlv对象
    :return:
    """
    dsw_list = list()
    enclosures = getEnclosureRecords(tlv)
    for enclosure in enclosures:
        logic_type = getRecordValue(enclosure, tlvData.ENCLOSURE['logicType'])
        if int(logic_type) in [tlvData.ENCLOSURE_TYPE_E.get("DSW"),
                               tlvData.ENCLOSURE_TYPE_E.get("MSW")]:
            dsw_list.append(enclosure)
    return dsw_list


def getEnclosureInfo(tlv):
    """获取当前集群所有框信息的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    records = execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

    enclosure_info_list = []
    for record in records:
        info_dict = {}
        info_dict["id"] = getRecordValue(record, tlvData.PUB_ATTR["id"])
        info_dict["name"] = getRecordValue(record, tlvData.PUB_ATTR["name"])
        info_dict["location"] = getRecordValue(record,
                                               tlvData.PUB_ATTR["location"])
        enclosure_info_list.append(info_dict.copy())
    return enclosure_info_list


def getEnclosureRecord(tlv, enclosure_id):
    """获取当前集群特定框信息的记录

    :param tlv: tlv对象
    :param enclosure_id: 框ID
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(enclosure_id))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)


def getCtrlEnclosureInfo(tlv):
    """获取当前集群框高

    :param tlv: tlv对象
    :return:
    """
    enclosure_info_list = []
    records = getEnclosureRecords(tlv)
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            info_dict = {}
            info_dict["id"] = getRecordValue(
                record,
                tlvData.PUB_ATTR["id"])
            info_dict["name"] = getRecordValue(
                record,
                tlvData.PUB_ATTR["name"])
            info_dict["location"] = getRecordValue(
                record,
                tlvData.PUB_ATTR["location"])
            info_dict["serialNum"] = getRecordValue(
                record,
                tlvData.ENCLOSURE["serialNum"])
    return enclosure_info_list


def getCtrlEnclosureHeight(tlv):
    """获取当前集群框高

    :param tlv: tlv对象
    :return:
    """
    records = getEnclosureRecords(tlv)
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            return getRecordValue(record, tlvData.ENCLOSURE['height'])
    return None


def getEngineNameList(tlv):
    """获取引擎

    :param tlv: tlv对象
    :return:
    """
    engine_list = []
    records = getEnclosureRecords(tlv)
    for record in records:
        name = getRecordValue(record, tlvData.PUB_ATTR["name"])
        if "CTE" in name:
            engine_list.append(name)
    return engine_list


def getAllEnclosureSNs(tlv):
    """获取当前集群所有框SN集合

    :param tlv: tlv对象
    :return:
    """
    records = getEnclosureRecords(tlv)
    enclosure_sn_list = []
    for record in records:
        enclosure_sn_list.append(
            getRecordValue(record, tlvData.ENCLOSURE["serialNum"]))
    return enclosure_sn_list


def getNormalEthInterfaceModuleIDList(tlv):
    """获取当前集群正常的ETH接口卡ID列表

    :param tlv: tlv对象
    :return:
    """
    records = getInterfaceModuleRecords(tlv)
    eth_interface_module_id_list = []
    for record in records:
        interface_module_model = getRecordValue(record,
                                                tlvData.INTF_MODULE['model'])
        if interface_module_model == tlvData.INTF_MODEL_E['IOC_4XSMART']:
            health_status = getRecordValue(record,
                                           tlvData.PUB_ATTR['healthStatus'])
            running_status = getRecordValue(record,
                                            tlvData.PUB_ATTR['runningStatus'])
            if health_status == tlvData.HEALTH_STATUS_E['NORMAL'] and \
                    running_status == tlvData.RUNNING_STATUS_E['RUNNING']:
                interface_module_model_id = getRecordValue(
                    record,
                    tlvData.PUB_ATTR['id'])
                eth_interface_module_id_list.append(interface_module_model_id)
    return eth_interface_module_id_list


def getEthPortsParentIDList(tlv):
    """获取当前集群正常的ETH端口对应的控制器ID列表

    :param tlv: tlv对象
    :return:
    """
    records = getEthPortRecords(tlv)
    eth_ports_parent_id_list = []
    for record in records:
        eth_ports_id = getRecordValue(record, tlvData.PUB_ATTR['parentID'])
        eth_ports_parent_id_list.append(eth_ports_id)
    return eth_ports_parent_id_list


def getControllerRecords(tlv):
    """获取当前集群正常的ETH端口对应的控制器ID列表

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['CONTROLLER'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getInterfModelRecords(tlv):
    """获取当前接口卡信息

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['INTF_MODULE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getDiskRecords(tlv):
    """获取当前集群所有硬盘信息的记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['DISK'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getEnclosureSNs(tlv):
    """获取当前集群正常的ETH端口对应的控制器ID列表

    :param tlv: tlv对象
    :return:
    """
    records = getControllerRecords(tlv)
    enclosure_id_set = set()
    for record in records:
        enclosure_id_set.add(
            getRecordValue(record, tlvData.PUB_ATTR['parentID']))
    current_enclosure_s_ns = []
    for enclosure_id in enclosure_id_set:
        record = getEnclosureRecord(tlv, enclosure_id)
        current_enclosure_s_ns.append(
            getRecordValue(record, tlvData.ENCLOSURE['serialNum']))
    return current_enclosure_s_ns


def getControllersList(tlv):
    """获取当前集群所有的控制器ID列表

    :param tlv: tlv对象
    :return:
    """
    records = getControllerRecords(tlv)
    controllers_list = []
    for record in records:
        controller_id = getRecordValue(record, tlvData.PUB_ATTR['id'])
        controllers_list.append(controller_id)
    controllers_list.sort()
    return controllers_list


def getControllersNum(tlv):
    """获取当前集群的控制器数量

    :param tlv: tlv对象
    :return:
    """
    return len(getControllerRecords(tlv))


@wrapAllException
def getBayRecords(tlv):
    """获取当前集群的所有柜信息

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getBayIds(bay_records):
    """获取当前集群的所有柜ID列表

    :param bay_records: 柜列表
    :return:
    """
    bay_ids = []
    for record in bay_records:
        bay_id = getRecordValue(record, tlvData.BAY['id'])
        reg = re.compile(r"\d+$")
        match = reg.search(bay_id)
        if match:
            bay_ids.append(match.group())
    return bay_ids


def getBayModels(bay_records):
    """获取当前集群的所有柜model列表

    :param bay_records: 柜列表
    :return:
    """
    bay_models = []
    for record in bay_records:
        bay_model = getRecordValue(record, tlvData.BAY['model'])
        bay_models.append(bay_model)
    return bay_models


def getBayNames(bay_records):
    """获取当前集群的所有柜ID列表

    :param bay_records: 柜列表
    :return:
    """
    bay_names = []
    for record in bay_records:
        bay_name = getRecordValue(record, tlvData.BAY['name'])
        bay_names.append(bay_name)
    return bay_names


def getExpansionBayIdRecord(tlv, bay_id, is_ip_scale_out):
    """获取当前集群的特定柜ID的记录

    :param tlv: tlv对象
    :param bay_id: 柜ID
    :param is_ip_scale_out: 是否为ipscaleout
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    if bay_id is not None and bay_id != "":
        param = (tlvData.PUB_ATTR['id'], str(bay_id))
        params.append(param)

    if is_ip_scale_out:
        param = (tlvData.BAY['isIpScaleOut'], True)
        params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_EXPANSION_BAY_ID"], params)


def getBayConfigCtrlNum(tlv, bay_id, isIpScaleOut=False):
    """获取当前集群的特定柜节点的控制器数量

    :param tlv: tlv对象
    :param bay_id: 柜ID
    :param isIpScaleOut: is_ip_scale_out
    :return:
    """
    record = getExpansionBayIdRecord(tlv, bay_id, isIpScaleOut)
    return getRecordValue(record, tlvData.BAY['contrlTotal'])


def getBayConfigClustType(bay_config_record):
    """获取当前集群组网方式

    :param bay_config_record: 柜列表
    :return:
    """
    bay_config_clust_type = getRecordValue(
        bay_config_record,
        tlvData.IP_CLUST_BASE_INFO['netMode'])
    return bay_config_clust_type


def getBayConfigBaseIpAddr(bay_config_record):
    """获取当前集群基地址IP

    :param bay_config_record:
    :return: tlv对象
    """
    bay_config_base_ip_addr = getRecordValue(
        bay_config_record,
        tlvData.IP_CLUST_BASE_INFO['baseIpAddr'])
    return bay_config_base_ip_addr


def readXnetInfo(tlv):
    """获取Xnet信息

    :param tlv: tlv对象
    :return:
    """
    param_list = []
    return execCmd(tlv, tlvData.CMD["READ_IP_CLUST_BASE_INFO"], param_list)


def getSystemRecord(tlv):
    """获取系统信息

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SYSTEM'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params)


def getProductVersion(tlv):
    """获取产品版本

    :param tlv: tlv对象
    :return:
    """
    record = getSystemRecord(tlv)
    if record is None:
        return None
    software_version = getRecordValue(record, tlvData.SYSTEM.get("software_version"))
    if software_version:
        return software_version
    point_release = getRecordValue(record, tlvData.SYSTEM['pointRelease'])
    product_ver = getRecordValue(record, tlvData.SYSTEM['productVersion'])
    return point_release if point_release else product_ver


def getProductVersionAndPatchVersion(tlv):
    """获取产品版本和补丁版本

    :param tlv: tlv对象
    :return: (软件版本, 补丁版本)
    """
    record = getSystemRecord(tlv)
    if record is None:
        return None, None
    patch_ver = getRecordValue(record, tlvData.SYSTEM['patchVersion'])
    software_version = getRecordValue(record, tlvData.SYSTEM.get("software_version"))
    if software_version:
        return software_version, patch_ver
    point_release = getRecordValue(record, tlvData.SYSTEM['pointRelease'])
    product_ver = getRecordValue(record, tlvData.SYSTEM['productVersion'])
    return point_release or product_ver, patch_ver


def getPdtModel(product_model_enum):
    """获取产品类型

    :param product_model_enum:
    :return: tlv对象
    """
    return tlvData.PRODUCT_MODE_E.get(product_model_enum)


def getDoradoPdtType(product_model_enum):
    """获取Dorado V6详细设备形态（SAS/IPSAS/NVMe）

    :param product_model_enum: 产品型号枚举
    :return:
    """
    return tlvData.PRODUCT_MODE_E_DORADO.get(product_model_enum)


def get_product_model_enum(tlv):
    """获取产品型号枚举

    :param tlv: tlv对象
    :return:
    """
    record = getSystemRecord(tlv)
    return getRecordValue(record, tlvData.SYSTEM['productMode'])


def get_product_model_string(tlv):
    """获取产品型号string

    :param tlv: tlv对象
    :return:
    """
    record = getSystemRecord(tlv)
    return getRecordValue(record, tlvData.SYSTEM['product_model_string'])


def getProductModel(tlv):
    """获取产品类型

    :param tlv: tlv对象
    :return:
    """
    product_model_enum = get_product_model_enum(tlv)
    return getPdtModel(product_model_enum)


def getDoradoV6ProductModel(tlv):
    """获取产品类型

    :param tlv: tlv对象
    :return:
    """
    record = getSystemRecord(tlv)
    product_model_enum = getRecordValue(record, tlvData.SYSTEM['productMode'])
    return getDoradoPdtType(product_model_enum)


def getInternalDeviceInfo(tlv):
    """获取内部设备信息(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param tlv: tlv对象
    :return:
    """
    param_list = []
    return execCmd(tlv, tlvData.CMD["GET_INTERNAL_DEVICE_INFO"], param_list)


def getInternalPdtModel(internal_device_info):
    """获取内部型号(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param internal_device_info: 设备内部信息
    :return:
    """
    return getRecordValue(internal_device_info,
                          tlvData.INTERNAL_DEVICE_INFO['internalProductModel'])


def get_overseas(internal_device_info):
    """获取是否为海外版本(新融合6.1.3及以后)

    :param internal_device_info: 设备内部信息
    :return:
    """
    return getRecordValue(internal_device_info,
                          tlvData.INTERNAL_DEVICE_INFO['isOverseas'])


def getCpuModel(internal_device_info):
    """获取CPU型号(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param internal_device_info: 设备内部信息
    :return:
    """
    return getRecordValue(internal_device_info,
                          tlvData.INTERNAL_DEVICE_INFO['cpuModelInfo'])


def getCpuNumber(internal_device_info):
    """获取CPU个数(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param internal_device_info: 设备内部信息
    :return:
    """
    return getRecordValue(internal_device_info,
                          tlvData.INTERNAL_DEVICE_INFO['cpuNumber'])


def getPatchVersion(tlv):
    """获取补丁版本

    :param tlv: tlv对象
    :return: 补丁版本信息；异常时返回--
    """
    params = []
    param = (
        tlvData.CMO_CR_VER_INFO_S['CMO_VER_PACKAGE_TYPE'],
        tlvData.UPGRADE_PKGTYPE_E['UPGRADE_PKGTYPE_TYPE_HOT_PATCH'])
    params.append(param)
    rec = execCmd(tlv, tlvData.CMD["OM_MSG_OP_LST_VER"], params)
    try:
        patch_version = getRecordValue(rec, tlvData.CMO_CR_VER_INFO_S[
            'CMO_VER_CUR_VERSION'])
        return patch_version
    except Exception:
        return "--"


def getFullProductVersion(tlv, logger=None):
    """获取完整版本信息

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    params = []
    param = (tlvData.CMO_CR_VER_INFO_S['CMO_VER_PACKAGE_TYPE'],
             tlvData.UPGRADE_PKGTYPE_E['UPGRADE_PKGTYPE_TYPE_SYS'])
    params.append(param)
    recs = execCmd(tlv, tlvData.CMD["OM_MSG_OP_LST_VER"], params)
    if type(recs) == ArrayList or type(recs) == list:
        for rec in recs:
            package_version = getRecordValue(rec, tlvData.CMO_CR_VER_INFO_S[
                'CMO_VER_CUR_VERSION'])
            if logger:
                logger.logInfo("packageVersion is :" + package_version)
            if package_version and len(package_version) > 1:
                return package_version
    return ""


def getSysConfigModel(tlv):
    """获取系统配置模式

    :param tlv: tlv对象
    :return:
    """
    record = getSystemRecord(tlv)
    config_model_enum = getRecordValue(record, tlvData.SYSTEM['config_model'])
    return getConfigModel(config_model_enum)


def getConfigModel(config_model_enum):
    """获取系统配置模式

    :param config_model_enum: 配置模式枚举
    :return:
    """
    return dict([(tlvData.CONFIG_MODEL_E.get(key), key) for key in
                 tlvData.CONFIG_MODEL_E]).get(config_model_enum)


def isIBCPasswordChanged(tlv):
    """检查IBC密码是否为默认密码

    :param tlv: tlv对象
    :return:
    """
    params = []
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    record = execCmd(tlv, cmd, params)
    return getRecordValue(record,
                          tlvData.CHECK_IBC_PASSWORD['passwordChanged'])


def isIBCKeyPairChanged(tlv):
    """检查IBC公私钥对是否为默认值

    :param tlv: tlv对象
    :return:
    """
    params = []
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    record = execCmd(tlv, cmd, params)
    return getRecordValue(record, tlvData.CHECK_IBC_PASSWORD['keyPairChanged'])


def getCpuInfoRecords(tlv):
    """获取当前集群的控制器CPU信息集合

    :param tlv: tlv对象
    :return:
    """
    params = []
    param0 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'], 0)
    params.append(param0)
    param12 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'],
               "upd sysresourcecheck")
    params.append(param12)
    # 所有节点上执行
    param13 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'], 2)
    params.append(param13)
    return execCmd(tlv, tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"], params,
                   getBatch=True)


def getAllCpuUsage(tlv):
    """获取当前集群的控制器CPU利用率

    :param tlv: tlv对象
    :return:
    """
    cpu_usage_list = []
    records = getCpuInfoRecords(tlv)
    for record in records:
        ctrl_id = Tlv2Rest.getRecordValue(
            record,
            restData.System.CpuUsage.RECORD_CONTROLLER_ID)
        cpu_info = Tlv2Rest.getRecordValue(
            record,
            restData.System.CpuUsage.RECORD_CPU_INFO)
        cpu_usage = cpu_info.split(",")[0]
        cpu_usage_list.append(
            {"ctrlId": ctrl_id, "cpuUsage": cpu_usage}.copy())
    return cpu_usage_list


def modifyBayID(tlv, bay_id):
    """修改柜ID

    :param tlv: tlv对象
    :param bay_id: 柜ID
    :return:
    """
    bay_id = "%s%s" % ("SMB", bay_id)
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(bay_id))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)


def modifyBayIDCtrlNum(tlv, bay_id, ctrl_num):
    """修改柜ID和控制器数量

    :param tlv: tlv对象
    :param bay_id: 柜ID
    :param ctrl_num: 控制器个数
    :return:
    """
    bay_id = "%s%s" % ("SMB", bay_id)
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], str(bay_id))
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(ctrl_num))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)


def modifyCtrlNum(tlv, new_config_ctrl_num):
    """修改控制器数量

    :param tlv: tlv对象
    :param new_config_ctrl_num: 修改后的控制器数目
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(new_config_ctrl_num))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)


def setScaleoutDiskEncNum(tlv, disk_enclosure_num):
    """设置扩容硬盘框数量

    :param tlv: tlv对象
    :param disk_enclosure_num: 硬盘框数目
    :return:
    """
    params = []
    param = (tlvData.SCALEOUT_DISK_ENC_NUM["diskEnclosureNum"],
             int(disk_enclosure_num))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["SET_SCALEOUT_DISK_ENCLOSURE_NUM"], params)


def changeBMC(tlv):
    """修改BMC（6U2C扩容至6U4C时，需将配置信息先写到BMC）

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["change"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params, timeOut=300)


def deleteBMC(tlv):
    """清除BMC（6U2C扩容至6U4C时，配置信息先写到BMC后需清除）

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["delete"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params)


def syncBMC(tlv):
    """同步BMC（2U环境将配置信息直接写入到disk，3U和6U环境将配置信息同步到BMC）

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["sync"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params, timeOut=300)


def expandCtrlCapacity(tlv, isIpScaleOut=True):
    """执行扩容

    :param tlv: tlv对象
    :param isIpScaleOut: 是否为ipscaleout
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    if isIpScaleOut:
        param = (tlvData.BAY['isIpScaleOut'], True)
        params.append(param)
    return execCmd(tlv, tlvData.CMD["CAPACITY_EXPAND"], params)


def oneCmdCtrlExpnsion(tlv, ctrl_num_after_expansion, base_ip_addr0,
                       base_ip_addr1, new_ctr_sn):
    """一个命令执行扩容流程。

    :param tlv: tlv对象
    :param ctrl_num_after_expansion: 扩控后的控制器个数
    :param base_ip_addr0: IP基地址1
    :param base_ip_addr1: IP基地址2
    :param new_ctr_sn: 新控制框SN
    :return:
    """

    type_para = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['EXPANSION_FLOW'])
    ctrl_num_after_exp_para = (
        tlvData.PUB_ATTR['totalCtrlNumAfterCtrlExpansion'],
        ctrl_num_after_expansion)
    base_ip0_para = (tlvData.PUB_ATTR['baseIpAddr0'], base_ip_addr0)
    base_ip1_para = (tlvData.PUB_ATTR['baseIpAddr1'], base_ip_addr1)
    new_ctr_sn_para = (tlvData.PUB_ATTR['newCtrSn'], new_ctr_sn)
    params = [ctrl_num_after_exp_para, type_para, base_ip0_para, base_ip1_para,
              new_ctr_sn_para]

    return execCmd(tlv, tlvData.CMD["CAPACITY_EXPAND"], params)


def queryCtrlExpansionItem(tlv, logger):
    """查询扩控执行流程项

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    try:
        record = qryCtrlExpansionFlow(tlv)[0]

        cur_ctrl_exp_step = getRecordValue(
            record,
            tlvData.PUB_ATTR['curCtrlExpStep'])
        cur_ctrl_exp_step_remain_mins = getRecordValue(
            record,
            tlvData.PUB_ATTR['curCtrlExpStepRemainTime']) / 60
        ctrl_exp_state = getRecordValue(record,
                                        tlvData.PUB_ATTR['ctrlExpState'])
        error_code = getRecordValue(record,
                                    tlvData.PUB_ATTR['errorCode'])
    except Exception as e:
        logger.logException(e)
        return False, (None, None, None, None)
    else:
        return True, (cur_ctrl_exp_step,
                      cur_ctrl_exp_step_remain_mins,
                      ctrl_exp_state,
                      error_code)


def qryCtrlExpansionFlow(tlv):
    """查询扩容流程信息。

    :param tlv: tlv对象
    :return:
    """
    type_para = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['EXPANSION_FLOW'])
    return execCmd(tlv, tlvData.CMD["QUERY_EXP_CTRL_INFO"], [type_para])


def setEnclosureSwitchOn(tlv):
    """打开框的定位灯

    :param tlv: tlv对象
    :return: True: 设置成功  Flase: 设置失败
    """
    return setEnclosureSwitch(tlv, 1)


def setEnclosureSwitchOff(tlv):
    """关闭框的定位灯

    :param tlv: tlv对象
    :return: True: 设置成功  Flase: 设置失败
    """
    return setEnclosureSwitch(tlv, 2)


def setCurrentEnclosureSwitch(tlv, enclosure_id, switch):
    """设置当前框的定位灯状态

    :param tlv: tlv对象
    :param enclosure_id: 框ID
    :param switch: 定位灯状态（ON：1，OFF：2）
    :return: True: 设置成功  Flase: 设置失败
    """
    try:
        params = []
        param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["ENCLOSURE"])
        params.append(param)
        param = (tlvData.PUB_ATTR["id"], enclosure_id)
        params.append(param)
        param = (tlvData.ENCLOSURE["switch"], switch)
        params.append(param)

        execCmd(tlv, tlvData.CMD["MODIFY"], params)

    except Exception:
        return False

    return True


def setEnclosureSwitch(tlv, switch):
    """设置框的定位灯状态

    :param tlv: tlv对象
    :param switch: 定位灯状态（ON：1，OFF：2）
    :return:  True: 设置成功  Flase: 设置失败
    """
    records = getEnclosureRecords(tlv)
    for record in records:
        enclosure_id = getRecordValue(record, tlvData.PUB_ATTR["id"])
        set_switch_ret = setCurrentEnclosureSwitch(tlv, enclosure_id, switch)
        if not set_switch_ret:
            return False

    return True


def getFruRunningStatus(context, fru_type, fru_id):
    """获取备件运行状态

    :param context: 上下文
    :param fru_type: 备件类型
    :param fru_id: 备件ID
    :return:
    """
    rec = getRecordById(context, fru_type, fru_id)
    return Tlv.Record.getIntValue(rec, TlvEnum.pubAttr["runningStatus"])


def getRecordById(context, fru_type, fru_id):
    """根据ID获取FRU信息

    :param context: 上下文
    :param fru_type: 备件类型
    :param fru_id: 备件ID
    :return:
    """
    # 参数检查
    if None is fru_type or None is fru_id:
        Log.error(context, "getFruInfo: exist invalid parameter. fruType="
                  + str(fru_type) + ", fruId=" + str(fru_id))
        return None

    # 批量获取并筛选出指定的FRU
    all_fru_list = getFruListInfo(context, fru_type)
    condition0 = Tlv.Record.getCondition(TlvEnum.pubAttr["id"],
                                         Tlv.Record.ConditionType["EQ"],
                                         fru_id)
    condition_list = Tlv.Record.getConditionList(condition0)
    tmp_recs = Tlv.Record.filter(all_fru_list, condition_list)

    # 返回结果
    if not tmp_recs:
        raise Exception("Get no record about fru, fruType: %s, fruId: %s" %
                        (str(fru_type), str(fru_id)))
    return tmp_recs.get(0)


def getFruListInfo(context, fru_type):
    """获取指定类型的具体FRU信息

    :param context: 数据上下文
    :param fru_type: FRU类型
    :return: None 或  具体FRU记录
    """
    # 参数检查
    if None is context or None is fru_type:
        Log.error(context,
                  "getFruInfo: exist invalid parameter. fruType=" + str(
                      fru_type))
        return None
    # 查询状态
    param0 = Tlv.Param.getParam(TlvEnum.pubAttr["type"], Tlv.Param.TYPE_ENUM,
                                fru_type)
    paramlist = Tlv.Param.getParamList(param0)
    if fru_type == TlvEnum.Enum.OM_OBJ_E["DISK"]:
        recs = Tlv.execmd(context, True, TlvEnum.cmd["GET_BATCH_NEXT"],
                          paramlist, logCmdResult=False)
    else:
        recs = Tlv.execmd(context, True, TlvEnum.cmd["GET_BATCH_NEXT"],
                          paramlist)
    # 设置返回值
    return recs


def checkControllerStatus(tlv):
    """检查指定节点的控制器状态

    :param tlv: tlv对象
    :return:
    """
    records = getControllerRecords(tlv)
    if records is None:
        return True
    for record in records:
        health_status = getRecordValue(record,
                                       tlvData.PUB_ATTR['healthStatus'])
        running_status = getRecordValue(record,
                                        tlvData.PUB_ATTR['runningStatus'])
        return \
            health_status == tlvData.HEALTH_STATUS_E["NORMAL"] and \
            running_status == tlvData.RUNNING_STATUS_E["ONLINE"]


def getAllMemory(tlv):
    """获取内存

    :param tlv: tlv对象
    :return:
    """
    params = []
    param0 = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['MEMORY'])
    params.append(param0)

    cmd = tlvData.CMD["GET_BATCH_NEXT"]

    memory_list = []
    records = execCmd(tlv, cmd, params, isBatch=True)
    if records is None:
        return memory_list

    for record in records:
        memory_dict = {}
        memory_dict["id"] = getRecordValue(record, tlvData.PUB_ATTR['id'])
        memory_dict["parentType"] = getRecordValue(record, tlvData.PUB_ATTR[
            'parentType'])
        memory_dict["parentID"] = getRecordValue(record,
                                                 tlvData.PUB_ATTR['parentID'])
        memory_dict["capacity"] = bit2GB(
            getRecordValue(record, tlvData.MEMORY['capacity']))
        memory_list.append(memory_dict.copy())
    return memory_list


def dec2hex(string_num):
    """十进制转十六进制

    :param string_num: 数字字符串
    :return:
    """
    str_buffer = [str(x) for x in range(10)] + [chr(x) for x in
                                                range(ord('A'), ord('A') + 6)]
    num = int(string_num)
    mid = []
    while True:
        if num == 0:
            break
        num, rem = divmod(num, 16)
        mid.append(str_buffer[rem])

    return "0x" + ''.join([str(x) for x in mid[::-1]])


def getSystemAlarm(tlv):
    """获取指定节点的告警信息

    :param tlv: tlv对象
    :return:
    """
    params = []

    cmd = tlvData.CMD["GET_CURRENT_ALARM"]
    records = execCmd(tlv, cmd, params, getBatch=True)
    if not records:
        return []
    alarm_list = []
    for record in records:
        if TLV_RECORD_END_INDEX in record.keys():
            return alarm_list
        alarm_dict = {}
        alarm_dict["CMO_ALARM_TYPE"] = getRecordValue(
            record,
            tlvData.CMO_ALARM_INFO_E['CMO_ALARM_TYPE'])
        alarm_dict["CMO_ALARM_ID"] = dec2hex(
            getRecordValue(record,
                           tlvData.CMO_ALARM_INFO_E['CMO_ALARM_ID']))
        alarm_dict["CMO_ALARM_LEVEL"] = getRecordValue(
            record,
            tlvData.CMO_ALARM_INFO_E['CMO_ALARM_LEVEL'])
        alarm_dict["CMO_ALARM_CLEAR_TIME"] = getRecordValue(
            record,
            tlvData.CMO_ALARM_INFO_E['CMO_ALARM_CLEAR_TIME'])
        alarm_dict["CMO_ALARM_RECOVERY_TIME"] = getRecordValue(
            record,
            tlvData.CMO_ALARM_INFO_E['CMO_ALARM_RECOVERY_TIME'])
        alarm_list.append(alarm_dict.copy())
    return alarm_list


def getPcieInterfaceRecords(tlv):
    """获取当前集群所有PCIE接口卡的记录

    :param tlv: tlv对象
    :return:
    """
    pcie_interface_records = []
    all_interface_records = getInterfaceModuleRecords(tlv)
    for intf_record in all_interface_records:
        intf_type = getRecordValue(intf_record, tlvData.PUB_ATTR['model'])
        if intf_type == tlvData.INTF_MODEL_E["PCIe_4X5G"]:
            pcie_interface_records.append(intf_record)
    return pcie_interface_records


def getPcieIntfModuleList(tlv, board):
    """获取指定节点用于PCIE扩容的接口卡集合

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    records = getPcieInterfaceRecords(tlv)
    if records is None:
        return []
    pcie_intf_module_list = []
    for record in records:
        pcie_intf_module_info = {}
        pcie_intf_module_info["enclosureSN"] = board['enclosureSN']
        pcie_intf_module_info["location"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['location'])
        pcie_intf_module_info["healthStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['healthStatus'])
        pcie_intf_module_info["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        pcie_intf_module_list.append(pcie_intf_module_info.copy())
    return pcie_intf_module_list


def getPciePortRecords(tlv):
    """获取指定节点上的所有ETH端口记录

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
    params.append(param)

    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    return execCmd(tlv, cmd, params, isBatch=True)


def getPciePortList(tlv, board):
    """获取指定节点用于PCIE扩容的接口卡的端口集合

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    records = getPciePortRecords(tlv)
    if records is None:
        return []
    pcie_port_list = []
    for record in records:
        pcie_port_info = {}
        pcie_port_info["enclosureSN"] = board['enclosureSN']
        pcie_port_info["location"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['location'])
        pcie_port_info["healthStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['healthStatus'])
        pcie_port_info["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        pcie_port_info["current_peer_port_id"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['current_peer_port_id'])
        pcie_port_info["suggest_peer_port_id"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['suggest_peer_port_id'])

        pcie_port_list.append(pcie_port_info.copy())
    return pcie_port_list


def getDiskEncloureNum(records):
    """获取当前集群2U和4U硬盘框的数量

    :param records: tlv返回值
    :return:
    """
    encloure_2u_list, encloure_4u_list = get_disk_enc_2U_4U_list(records)
    return len(encloure_2u_list), len(encloure_4u_list)


def get_disk_enc_2U_4U_list(records):
    """ 获取系统中已有的2U/4U框location列表

    :param records:
    :return:
    """
    encloure_2u_list = []
    encloure_4u_list = []
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        height = getRecordValue(record, tlvData.ENCLOSURE['height'])
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["EXP"]:
            if height == 2:
                encloure_2u_list.append(location)
            elif height == 4:
                encloure_4u_list.append(location)
    return encloure_2u_list, encloure_4u_list


def getCtrlEncNum(records):
    """Get controller enclosure number.

    :param records: All enclosure records.
    :return:
    """
    encloure_num = 0
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            encloure_num += 1
    return encloure_num


def getDiskEnclouresName(records):
    """获取当前集群硬盘框的丝印

    :param records: tlv返回值
    :return:
    """
    encs_name_list = []
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["EXP"]:
            enc_name = getRecordValue(record, tlvData.ENCLOSURE['name'])
            encs_name_list.append(enc_name)
    return encs_name_list


def get_new_disk_enc(records, orgin_disk_enc_names):
    """获取新扩框列表

    :param records: 所有硬盘框列表
    :param orgin_disk_enc_names: 原集群硬盘框列表
    :return:
    """
    encs_name_list = []
    encs_loc_list = []
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        name = getRecordValue(record, tlvData.ENCLOSURE['name'])
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])
        health_status = getRecordValue(record,
                                       tlvData.ENCLOSURE['healthStatus'])
        running_status = getRecordValue(record,
                                        tlvData.ENCLOSURE['runningStatus'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["EXP"] \
                and str(health_status) == \
                str(tlvData.HEALTH_STATUS_E["NORMAL"]) \
                and str(running_status) == \
                str(tlvData.RUNNING_STATUS_E["ONLINE"]) \
                and name not in orgin_disk_enc_names:
            encs_name_list.append(name)
            encs_loc_list.append(location)
    return encs_name_list, encs_loc_list


def getDiskEnclouresLoc(records):
    """获取当前集群硬盘框的位置

    :param records:
    :return:
    """
    enc_locs = []
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["EXP"]:
            enc_loc = getRecordValue(record, tlvData.PUB_ATTR["location"])
            enc_locs.append(enc_loc)
    return enc_locs


def getDiskEncloureModels(records):
    """获取所有硬盘框的类型。

    :param records: 所有框的记录.
    :return:
    """
    enc_models = []
    for record in records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["EXP"]:
            enc_model = getRecordValue(record, tlvData.ENCLOSURE['model'])
            enc_models.append(enc_model)
    return enc_models


def getBackIoConcurrent(tlv, controller_id):
    """获取当前控制器的后端并发信息

    :param tlv: tlv对象
    :param controller_id: 控制器id
    :return:
    """
    params = []
    param = (tlvData.CONTROLLER_IO['id'], controller_id)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BACK_END_CONCURRENT"], params)


def getEngineEnclosureID(tlv):
    """获取当前系统的引擎框

    :param tlv: tlv对象
    :return:
    """
    enc_ids = []
    enc_records = getEnclosureRecords(tlv)
    for record in enc_records:
        enc_name = getRecordValue(record, tlvData.PUB_ATTR["name"])
        if enc_name.startswith("CTE"):
            enc_id = getRecordValue(record, tlvData.PUB_ATTR["id"])
            enc_ids.append(enc_id)

    return enc_ids


def getCabinetDict(bay_records):
    """获取已有柜，并映射为字典

    :param bay_records: 所有柜信息
    :return:
    """
    cabinet_dict = {}
    for record in bay_records:
        cabinet_id = getRecordValue(record, tlvData.PUB_ATTR["id"])
        cabinet_name = getRecordValue(record, tlvData.PUB_ATTR["name"])
        cabinet_dict.setdefault(cabinet_id, cabinet_name)
    return cabinet_dict


def getPowerType(tlv):
    """取电源类型(整柜不存在混合电源场景)

    :param tlv: tlv对象
    :return:
    """
    power_type = None
    power_records = getPowerRecords(tlv)
    for record in power_records:
        location = getRecordValue(record, tlvData.PUB_ATTR["location"])
        if "CTE" in str(location):
            power_type = getRecordValue(record, tlvData.POWER["powerType"])
            break
    return power_type


def getCtrlEnclosureModel(tlv):
    """获取指定框的框model

    :param tlv: tlv对象
    :return:
    """
    enclosure_records = getEnclosureRecords(tlv)
    for record in enclosure_records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            model = getRecordValue(record, tlvData.ENCLOSURE['model'])
            return model
    return None


def executeInheritClusterMgtIP(tlv, controller_num):
    """继承管理IP

    :param tlv: tlv对象
    :param controller_num: 控制器数目
    :return:
    """
    params = []
    param = (tlvData.INHERIT_CLUSTER_MGT_IP_DEFINE['oldNodeNum'],
             int(controller_num))
    params.append(param)
    execCmd(tlv, tlvData.CMD["INHERIT_CLUSTER_MGT_IP"], params)
    return


def getDeviceGlobalConfRecord(tlv):
    """获取设备全局配置信息记录

    :param tlv:
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['DEVICE_GLOBAL_CONF'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params)


def getMaxExpansionDepth(tlv):
    """获取SAS硬盘框和智能硬盘框最大级联深度

    :param tlv:
    :return:
    """
    record = getDeviceGlobalConfRecord(tlv)
    sas_enc_depth = getRecordValue(
        record, tlvData.DEVICE_GLOBAL_CONF['frameCascadeMode'])
    smart_enc_depth = getRecordValue(
        record, tlvData.DEVICE_GLOBAL_CONF['smartFrameCascadeMode'])
    return sas_enc_depth, smart_enc_depth


def isSupportDiskEnclosureConcurrentUpgrade(tlv):
    """判断是否是支持硬盘框并行升级.

    :param tlv: TLV连接
    :return:
    """
    product_version = getProductVersion(tlv)
    product_model = getProductModel(tlv)
    # v7版本不支持该命令，不需要执行
    if product_version.startswith("V700"):
        return False
    return product_model == 'Dorado18000 V3' or product_version >= 'V300R006C20' or product_version >= 'V500R007C00'


def notifyEnclosureConcurrentUpgrade(tlv, new_enc_num, logger):
    """通知升级模块硬盘框并发升级(最好在所有新硬盘框上电前下发）
    升级模块在检测到第一个框接入后如果所有框都接入则触发并发
    升级流程；或者2分钟内无新框接入则启动并发升级流程。

    :param tlv:
    :param new_enc_num:
    :param logger:
    :return:
    """
    try:
        if isSupportDiskEnclosureConcurrentUpgrade(tlv):
            logger.logInfo("set scaleout disk enclosure number start")
            setScaleoutDiskEncNum(tlv, new_enc_num)
            logger.logInfo("set scaleout disk enclosure number End")
    except Exception as e:
        logger.logInfo("set scaleout disk enclosure number Error:" + str(e))


def getCtrlTaskRestrictState(tlv):
    """获取当前系统的控制器后台任务释放被管控

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['TASK_RESTRICT_STATE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def delete_packages(data_dict):
    """
    删除内存中的补丁包，tlv连接使用已有连接，不在这里关闭
    :param data_dict:
    :return:
    """
    logger = contextUtil.getLogger(data_dict)
    try:
        tlv = data_dict["tlv"]
        package_type = constants.TLV_PACKAGE_TYPE.HOT_PATCH_PKG
        param_package_type = Param(0, ParamType.UNSIGN_INT,
                                   UnsignedInt32(package_type))
        param_package_size = Param(2, ParamType.UNSIGN_INT, UnsignedInt32(0))
        params = TLVUtils.paramList(param_package_type, param_package_size)
        # 获取补丁包上传路径
        op_code = tlvData.CMD.get("OM_MSG_OP_GET_PACKAGE_UPLOADPATH")
        recs = tlv.getBatch(op_code, params, TLV_CMD_TIMEOUT)
        logger.info("execute delete packages success:{}".format(recs))
        return True
    except Exception as e:
        logger.error("execute delete packages fail:{}".format(e))
        return False


def get_upgrade_records(tlv, checkItem):
    '''
    @summary: 执行升级检查的tlv命令
    @param tlv: tlv对象
    @param checkItem:升级检查项索引
    '''
    params = []
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'],
             tlvData.UPD_MSG_E['EXE_UPD_MSGTYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_ACTIVETYPE'],
             tlvData.UPD_MSG_E['EXE_UPD_ACTIVETYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'],
             tlvData.UPD_CMD_DEFINE[checkItem]['cmd'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'],
             tlvData.UPD_CMD_DEFINE[checkItem]['id'])
    params.append(param)

    return execCmd(tlv, tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"], params,
                   getBatch=True, timeOut=CMD_DEFAULT_TIMEOUT)
