# -*- coding: UTF-8 -*-
import os
import stat
import gzip
import datetime
import traceback
import sys
import zipfile
from common.util import device
import shutil

DISKID_START_FLAG = "Smart of Disk["
SMART_COLLECT_FAIED_FLAG = "Failed to collect"
OS_OPEN_MODES = stat.S_IWUSR | stat.S_IRUSR  # 拥有者具有读写权限


def format_traceback():
    '''
    @summary: 格式化输出错误堆栈信息
    '''
    new_extracted_list = []
    etype, value, tb = sys.exc_info()
    extracted_list = traceback.extract_tb(tb)
    for item in extracted_list:
        filename, lineno, name, line = item
        filename = os.path.basename(filename)
        new_extracted_list.append((filename, lineno, name, line))
        
    return "".join(traceback.format_list(new_extracted_list) + traceback.format_exception_only(etype, value))

def logException(log):
    '''
    @summary: 日志输出错误堆栈信息
    '''
    log.info(format_traceback())


def getDiskMapInfo4SPC100(gzFile, diskMapInfo, log, diskSNList, devObj):
    '''
    @summary: 830 SPC100及其之前的版本因为smart原始压缩包格式和smart信息格式与SPC200不同，需要单独处理
    '''
    
    tar_class = devObj.get("PYENGINE.PY_ZIP")
    
    tempDirPath = os.path.join(gzFile[0:gzFile.find("disksmartinfo")], "temp")
    createDir(devObj,  tempDirPath, log)
    
    #解压disksmartinfo_0B到临时文件夹，解压后的文件为disksmartinfo_0.tar
    tar_class.decompressZipFile(gzFile, tempDirPath)
    #每个taz文件解压后只有一个tar文件
    tarFile = os.listdir(tempDirPath)[0]
    diskSmartInfoFile = gzip.open(os.path.join(tempDirPath, tarFile), 'rb')
    allContents = diskSmartInfoFile.read()
    
    #每个硬盘信息以45个*分隔
    diskInfoArray = allContents.split(45*"*")
   
    for diskInfo in diskInfoArray:
        #Serial Number:    FK0013121420000073  从以上格式中获取每个硬盘对应的SN号
        SNbeginIndex = diskInfo.find("Serial Number:    ")
        SNEndIndex = diskInfo.find("\n", SNbeginIndex)
        diskSN = diskInfo[SNbeginIndex+18:SNEndIndex].strip()
        
        #SN号CLI命令下无法查出，说明盘未接入系统，不用分析该硬盘信息
        if diskSN not in diskSNList:
            continue
        #如果曾经收集成功，则不再继续处理
        if diskMapInfo.has_key(diskSN) and diskMapInfo[diskSN][0] == True:
            continue
        
        diskMapInfo[diskSN] = [True, diskInfo]
    
    try:
        if diskSmartInfoFile is not None:
            diskSmartInfoFile.close()
        #清除临时文件夹
        shutil.rmtree(tempDirPath)
    except:
        logException(log)
    
    return

  
def getDiskMapInfo(gzFilePath, diskMapInfo, log, diskSNList, devObj):
    '''
    @summary: 读取gz压缩包中文件内容，将信息记入字典。
        key为硬盘id，值为元组。元组第0个元素表示是否收集成功，第1个元素表示该硬盘对应的收集到的信息。
                      如果该硬盘在某个引擎上的信息收集成功，如果对应硬盘在该引擎上其他节点收集失败，仍然认为该硬盘信息收集成功（单链路场景）
    @param gzFilePath: gz文件
    @param diskMapInfo: key为硬盘id，值为元组。元组第0个元素表示是否收集成功，第1个元素表示该硬盘对应的收集到的信息。
    '''
    needUseSnTypeList = ("2800 V3", "5300 V3", "5500 V3")
    needUseSnVersionStart = "V300R001C10" 
    diskSmartInfoFile = None
    try:
        diskSmartInfoFile = gzip.open(gzFilePath, 'rb')
        oneDiskInfo = ""
        while True:
            line = diskSmartInfoFile.readline()
            
            if getDiskID(line, log) or isEnd(line):
                diskID = getDiskID(oneDiskInfo, log)
                
                if diskID and (diskID not in diskMapInfo or diskMapInfo[diskID][0] != True):
                    #对系统盘的盘框一体的设备，会存在ID相同的情况，需要用SN识别, SN不在baseinfo.txt中的不记录
                    deviceType = str(device.getDeviceType(devObj)) #deviceType类型为java类型，需要转换
                    deviceVersion = device.getDeviceVersion(devObj)
                    if deviceType in needUseSnTypeList and deviceVersion.startswith(needUseSnVersionStart):
                        diskSN = getDiskSnBySmartInfo(oneDiskInfo)
                        if diskSN not in diskSNList:
                            oneDiskInfo = ""
                            oneDiskInfo += line
                            log.info("[parseDisk][getDiskMapInfo][not access system disk_id=%s]" % diskID)
                            continue
                    
                    # 缓存上一块硬盘的信息
                    if isCollectSucc(oneDiskInfo):
                        # 因硬盘健康分析需使用Device Model做分隔，所以必须判断此字段存在，否则可能导致分析数据错位。
                        if "Device Model" in oneDiskInfo:
                            diskMapInfo[diskID] = [True, oneDiskInfo]
                            log.info("[parseDisk][getDiskMapInfo][get disk %s success.]" % diskID)
                        else:
                            diskMapInfo[diskID] = [False, oneDiskInfo]
                            log.info("[parseDisk][getDiskMapInfo][the disk do not have Device Model (%s)!]" % diskID)
                    # 硬盘信息收集失败
                    else:
                        diskMapInfo[diskID] = [False, oneDiskInfo]
                        log.info("[parseDisk][getDiskMapInfo][get disk %s failed!]" % diskID)
                
                # 一块硬盘信息读取结束，开始缓存下一块硬盘的信息
                oneDiskInfo = ""
            
            oneDiskInfo += line
                
            # 读取结束
            if isEnd(line):
                log.info("[parseDisk][getDiskMapInfo][parse %s end.]" % gzFilePath)
                break
        return diskMapInfo
    except:
        logException(log)
        raise IOError
    finally:
        if diskSmartInfoFile is not None:
            diskSmartInfoFile.close()


def isEnd(line):
    """
    @summary: 判断文件读取是否结束
    """
    if not line:
        return True
    
    return False


def getDiskID(line, log):
    """
    @summary: 获取硬盘ID
    """
    if DISKID_START_FLAG not in line:
        return ""
    
    temp = line.split(DISKID_START_FLAG)
    if len(temp) < 2:
        return ""
    
    temp = temp[1]
    diskID = temp.split()[0]
    
    if not isDiskID(diskID, log):
        return ""
    
    return diskID


def isDiskID(diskID, log):
    """
    @summary: 判断是否为硬盘ID
    """
    if not diskID:
        return False
    
    if diskID.startswith("CTE") or diskID.startswith("ENG") or diskID.startswith("DAE") or diskID.startswith("ROOT"):
        return True
    
    log.info("[parseDisk][getDiskMapInfo][unhandled disk_id=%s]" % diskID)
    return False


def isCollectSucc(diskInfo):
    """
    @summary: 判断硬盘smart信息是否收集成功
    """
    if SMART_COLLECT_FAIED_FLAG in diskInfo:
        return False
    
    return True
    

def handleSmartInfo(inputPath, log, diskSNList, devObj):
    '''
    @summary: 处理smart信息压缩包文件存放目录中所有压缩包文件，获取总体的diskMapInfo集合
    @param inputPath: 收集到的smart信息压缩包文件存放目录
    '''
    diskMapInfo = {}
    
    gzFiles = os.listdir(inputPath)
    for gzFileName in gzFiles:
        gzFile = os.path.join(inputPath, gzFileName)
        if not gzFileName.endswith(".tgz"):
            log.info("[parseDisk][handleSmartInfo][unhandled gzfile=%s]" % gzFile)
            continue
        log.info("[parseDisk][handleSmartInfo][handled gzFile=%s]" % gzFile)
        
        deviceVersion = device.getDeviceVersion(devObj)
        #以下版本与其之后的版本因为阵列底层的原始smart信息不一致需要单独处理
        if deviceVersion in ["V200R002C20","V300R001C10","V300R001C10SPC100","V100R001C20","V100R001C20SPC100","V300R001C10B012"]:
            getDiskMapInfo4SPC100(gzFile, diskMapInfo, log, diskSNList, devObj) 
        else:
            getDiskMapInfo(gzFile, diskMapInfo, log, diskSNList, devObj)
    return diskMapInfo

def generageSmartFile(smartFilePath, collectContent, log):
    '''
    @summary: 生成smart*.txt
    @param smartFilePath: smart*.txt文件路径
    @param collectContent: 要写入的内容
    '''
    smartFile = None
    try:
        os_open_flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND  # 读写 | 创建并打开
        smartFile = os.fdopen(os.open(smartFilePath, os_open_flags, OS_OPEN_MODES), 'a')
        smartFile.write(collectContent)
        smartFile.write("\n")
        smartFile.flush()
    except:
        logException(log)
        raise IOError
    finally:
        if smartFile is not None:
            smartFile.close()  

def generageSummaryFile(summaryFilePath, summaryInfo, log):
    '''
    @summary: 生成summary.ini
    @param summaryFilePath: summary.ini文件路径
    @param collectContent: 要写入的内容
    '''
    summaryFile = None
    try:
        os_open_flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC  # 写 | 创建并打开 | 截断
        summaryFile = os.fdopen(os.open(summaryFilePath, os_open_flags, OS_OPEN_MODES), 'w')
        summaryFile.write("total=%s" % summaryInfo["total"])
        summaryFile.write("\n")
        summaryFile.write("success=%s" % summaryInfo["success"])
        summaryFile.write("\n")
        currDate = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        summaryFile.write("date=%s" % currDate)
        summaryFile.write("\n")
        summaryFile.flush()
    except:
        logException(log)
        raise IOError
    finally:
        if summaryFile is not None:
            summaryFile.close()

def converSmartInfo(inputPath, outputPath, log, diskIdList, diskSNList, diskDhaInfo, devObj):
    '''
    @summary: 自动根据收集到的smart信息生成summary.ini和smart*.txt
    @param inputPath: smart硬盘信息所在目录
    @param outputPath: 生成summary.ini和smart*.txt所在目录
    '''
    log.info("[parseDisk][converSmartInfo][start]")
    log.info("[parseDisk][converSmartInfo][inputPath=%s]" % inputPath)
    log.info("[parseDisk][converSmartInfo][outputPath=%s]" % outputPath)
    
    successCnt = 0
    failureCnt = 0
    
    #获取总体的diskMapInfo集合
    diskMapInfo = handleSmartInfo(inputPath, log, diskSNList, devObj)
    
    #不同版本处理得到的diskMapInfo的Key不同，部分版本是SN号做Key
    diskIdOrSns = diskMapInfo.keys()
    diskIdOrSns.sort()
    diskDhaFormat = "\nStart Health Mark\nHealth Mark For DHA : %s\n"
    for diskIdOrSn in diskIdOrSns:
        collectFlag = diskMapInfo[diskIdOrSn][0]
        log.info("[parseDisk][converSmartInfo][diskIdOrSn=%s]" % diskIdOrSn)
        log.info("[parseDisk][converSmartInfo][collectFlag=%s]" % collectFlag)
        
        #未接入系统的盘不做处理,show disk general能查到的盘为已接入系统的盘
        if diskIdOrSn not in diskIdList and diskIdOrSn not in diskSNList:
            log.info("[parseDisk][converSmartInfo][the disk(%s) is not accessed]" % diskIdOrSn)
            continue
        
        #只处理成功收集的信息
        if collectFlag != True:
            failureCnt += 1
            continue
        
        #将收集成功的硬盘信息写入文件
        smartFileName = "smart%s.txt" % (successCnt / 20)
        smartFilePath = os.path.join(outputPath, smartFileName)
        collectContent = diskMapInfo[diskIdOrSn][1] + diskDhaFormat % diskDhaInfo.get(diskIdOrSn)
        generageSmartFile(smartFilePath, collectContent, log)
        
        successCnt += 1
    
    #生成summary.ini
    summaryInfo = {"total": failureCnt + successCnt, "success": successCnt}
    summaryFileName = "summary.ini"
    summaryFilePath = os.path.join(outputPath, summaryFileName)
    generageSummaryFile(summaryFilePath, summaryInfo, log)
    
    log.info("[parseDisk][converSmartInfo][summaryInfo=%s]" % summaryInfo)
    log.info("[parseDisk][converSmartInfo][end]")
    return summaryInfo

                
def createDir(devObj, dirPath, log):
    """
          创建文件夹，若存在则清空该文件夹再创建
    """
    try:
        if os.path.exists(dirPath):
            shutil.rmtree(dirPath)
        os.makedirs(dirPath)
    except:
        log.info("create directory failed!")
        logException(log) 

def getDiskSnBySmartInfo(diskInfo):
    '''
    @summary:从SmartInfo中解析出硬盘SN号
    @param diskInfo: 从SmartInfo中得到的单块盘信息
    @return: diskSn
    '''
    diskSn = ""
    SNbeginIndex = diskInfo.find("Serial Number")
    SNEndIndex = diskInfo.find("\n", SNbeginIndex)
    diskSnInfo = diskInfo[SNbeginIndex:SNEndIndex]
    diskSnInfoSplit = diskSnInfo.split(":")
    
    #分割后Sn号存在则返回正确值，不存在则返回空值
    if len(diskSnInfoSplit) == 2:
        diskSn = diskSnInfoSplit[1].strip()
    return diskSn
